<?php
session_start();
$pageTitle = "Edit Record";
include 'header.php';
require 'config.php';

// Define constants
define('DEFAULT_IMAGE', 'https://bitvros.com/session/auth/user/uploads/default.png');
$baseImageUrl = "https://bitvros.com/session/auth/user/uploads/";
$kycTable = 'kyc_documents';

// Create kyc_documents table if it doesn't exist
try {
    $createTableSQL = "CREATE TABLE IF NOT EXISTS `{$kycTable}` (
        `id` INT AUTO_INCREMENT PRIMARY KEY,
        `user_id` INT NOT NULL,
        `id_front` VARCHAR(255),
        `id_back` VARCHAR(255),
        `selfie` VARCHAR(255),
        `document_front` VARCHAR(255),
        `document_back` VARCHAR(255),
        `status` ENUM('pending','approved','rejected') DEFAULT 'pending',
        `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
    $conn->exec($createTableSQL);
} catch (PDOException $e) {
    die("Error creating kyc_documents table: " . $e->getMessage());
}

// Columns that should be shown as images instead of plain text
$allowedDocumentTypes = ['id_front', 'id_back', 'selfie', 'document_front', 'document_back'];

// Fields that should be readonly
$readonlyFields = ['email', 'name'];

// Fields that are datetime/timestamp fields that need special handling
$dateTimeFields = ['token_expiry', 'created_at', 'updated_at', 'last_login', 'verified_at'];

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Check admin authentication
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header("Location: login.php");
    exit();
}

// Validate table and ID
if (!isset($_GET['table']) || !isset($_GET['id'])) {
    die('Table or record not specified.');
}

$table = $_GET['table'];
$id = (int)$_GET['id'];

// Sanitize table name
if (!preg_match('/^[a-zA-Z_]+$/', $table)) {
    die('Invalid table name.');
}

// For users table, ensure the "bill" column exists
if ($table === 'users') {
    $stmt = $conn->prepare("SHOW COLUMNS FROM `users` LIKE 'bill'");
    $stmt->execute();
    if (!$stmt->fetch()) {
        $conn->exec("ALTER TABLE `users` ADD COLUMN `bill` VARCHAR(10) DEFAULT NULL");
    }
}

// Fetch main record
try {
    $stmt = $conn->prepare("SELECT * FROM `$table` WHERE id = ?");
    $stmt->execute([$id]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$record) {
        die('Record not found.');
    }
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

// Process KYC Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['kyc_action'])) {
    if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        die("CSRF token validation failed.");
    }

    try {
        $kycId = (int)$_POST['kyc_id'];
        $action = $_POST['kyc_action'];
        
        $stmt = $conn->prepare("SELECT * FROM `$kycTable` WHERE id = ?");
        $stmt->execute([$kycId]);
        $kycRecord = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$kycRecord) {
            throw new Exception("Invalid KYC record");
        }
        
        $newStatus = ($action === 'approve') ? 'approved' : 'rejected';
        $stmt = $conn->prepare("UPDATE `$kycTable` SET status = ? WHERE id = ?");
        $stmt->execute([$newStatus, $kycId]);
        
        // Optionally update user table KYC status if this is for users
        if ($table === 'users') {
            $stmt = $conn->prepare("UPDATE users SET kyc_status = ? WHERE id = ?");
            $stmt->execute([$newStatus, $id]);
        }
        
        $success = "KYC record {$action}ed successfully";
    } catch (Exception $e) {
        $error = "KYC action failed: " . $e->getMessage();
    }
}

// Process main form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_main'])) {
    if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        die("CSRF token validation failed.");
    }

    try {
        $updates = [];
        $params = [];

        // Get valid fields from the table to avoid updating non-existent columns
        $stmt = $conn->prepare("SHOW COLUMNS FROM `$table`");
        $stmt->execute();
        $tableColumns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $validFields = array_column($tableColumns, 'Field');

        // Create a mapping of column types for better handling
        $columnTypes = [];
        foreach ($tableColumns as $column) {
            $columnTypes[$column['Field']] = $column['Type'];
        }

        foreach ($_POST as $key => $value) {
            // Skip non-field inputs
            if (in_array($key, ['csrf_token', 'update_main', 'kyc_action', 'kyc_id'])) {
                continue;
            }
            // Only update if the column exists in the table
            if (!in_array($key, $validFields)) {
                continue;
            }
            
            // Special handling for datetime/timestamp fields
            if (in_array($key, $dateTimeFields)) {
                // If the value is empty, set it to NULL for datetime fields
                if (empty(trim($value))) {
                    $updates[] = "`$key` = NULL";
                } else {
                    // Validate datetime format
                    $dateTime = DateTime::createFromFormat('Y-m-d H:i:s', $value);
                    if (!$dateTime && $value !== '') {
                        // Try alternative formats
                        $dateTime = DateTime::createFromFormat('Y-m-d', $value);
                        if ($dateTime) {
                            $value = $dateTime->format('Y-m-d H:i:s');
                        } else {
                            // Skip invalid datetime values
                            continue;
                        }
                    }
                    $updates[] = "`$key` = ?";
                    $params[] = $value;
                }
            } else {
                $updates[] = "`$key` = ?";
                $params[] = $value;
            }
        }

        // Handle file uploads
        foreach ($_FILES as $field => $file) {
            if ($file['error'] == UPLOAD_ERR_OK && is_uploaded_file($file['tmp_name'])) {
                $fileName = time() . '_' . basename($file['name']);
                $uploadPath = 'uploads/' . $fileName;
                
                if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
                    if (in_array($field, $validFields)) {
                        $updates[] = "`$field` = ?";
                        $params[] = $uploadPath;
                    }
                }
            }
        }
        
        if (!empty($updates)) {
            $params[] = $id; // Add id for WHERE clause
            $updateQuery = "UPDATE `$table` SET " . implode(', ', $updates) . " WHERE id = ?";
            $stmt = $conn->prepare($updateQuery);
            $stmt->execute($params);
            
            $success = "Record updated successfully";
            
            // Refresh record data
            $stmt = $conn->prepare("SELECT * FROM `$table` WHERE id = ?");
            $stmt->execute([$id]);
            $record = $stmt->fetch(PDO::FETCH_ASSOC);
        }
    } catch (Exception $e) {
        $error = "Update failed: " . $e->getMessage();
    }
}

// Get table columns (for form display)
$stmt = $conn->prepare("SHOW COLUMNS FROM `$table`");
$stmt->execute();
$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch KYC documents (if this is a user record)
$kycDocuments = [];
if ($table === 'users') {
    try {
        $stmt = $conn->prepare("SELECT * FROM `$kycTable` WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$id]);
        $kycDocuments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $error = "Failed to fetch KYC documents: " . $e->getMessage();
    }
}

/**
 * Returns a usable image URL:
 * - If $filename is empty, returns a default placeholder.
 * - If $filename is a full URL, returns it.
 * - Otherwise, prepends $baseImageUrl.
 */
function getImageUrl($filename) {
    global $baseImageUrl;
    if (empty($filename)) {
        return DEFAULT_IMAGE;
    }
    if (filter_var($filename, FILTER_VALIDATE_URL)) {
        return $filename;
    }
    return rtrim($baseImageUrl, '/') . '/' . ltrim($filename, '/');
}

/**
 * Format datetime value for display in form
 */
function formatDateTimeForForm($value) {
    if (empty($value) || $value === '0000-00-00 00:00:00') {
        return '';
    }
    $date = DateTime::createFromFormat('Y-m-d H:i:s', $value);
    return $date ? $date->format('Y-m-d\TH:i') : $value;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?= htmlspecialchars($pageTitle) ?></title>
  <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
  <style>
    .soft-ui {
      box-shadow: 8px 8px 15px rgba(0, 0, 0, 0.05), 
                  -8px -8px 15px rgba(255, 255, 255, 0.7);
    }
    .soft-input {
      background: #f8fafc;
      border: none;
      box-shadow: inset 4px 4px 8px rgba(0, 0, 0, 0.03), 
                  inset -4px -4px 8px rgba(255, 255, 255, 0.6);
      transition: all 0.2s ease;
    }
    .soft-input:focus {
      outline: none;
      box-shadow: inset 4px 4px 8px rgba(0, 0, 0, 0.06), 
                  inset -4px -4px 8px rgba(255, 255, 255, 0.8);
    }
    .soft-button {
      transition: all 0.2s ease;
      box-shadow: 6px 6px 12px rgba(0, 0, 0, 0.06), 
                  -6px -6px 12px rgba(255, 255, 255, 0.7);
    }
    .soft-button:hover {
      box-shadow: 4px 4px 8px rgba(0, 0, 0, 0.09), 
                  -4px -4px 8px rgba(255, 255, 255, 0.8);
      transform: translateY(-2px);
    }
    .soft-button:active {
      box-shadow: inset 4px 4px 8px rgba(0, 0, 0, 0.05), 
                  inset -4px -4px 8px rgba(255, 255, 255, 0.7);
      transform: translateY(0);
    }
    .readonly-field {
      background-color: #f1f5f9;
      color: #64748b;
    }
  </style>
</head>
<body class="bg-gray-100 min-h-screen">
<main class="container mx-auto p-4 md:p-8">
  <div class="bg-white rounded-2xl soft-ui p-6 md:p-8">
    <?php if (isset($success)): ?>
      <div class="mb-6 bg-green-50 p-4 rounded-xl text-green-700 border border-green-200">
        <?= htmlspecialchars($success) ?>
      </div>
    <?php endif; ?>
    <?php if (isset($error)): ?>
      <div class="mb-6 bg-red-50 p-4 rounded-xl text-red-700 border border-red-200">
        <?= htmlspecialchars($error) ?>
      </div>
    <?php endif; ?>
    
    <?php if ($table === 'users'): ?>
      <!-- Display Crypto Balance prominently in red -->
      <div class="mb-6">
        <span class="font-bold text-red-600 text-xl">
          Crypto Balance: <?= htmlspecialchars($record['crypto_balance'] ?? '0') ?>
        </span>
      </div>
    <?php endif; ?>
    
    <!-- Main Edit Form -->
    <form method="POST" enctype="multipart/form-data" class="space-y-6">
      <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
      <h2 class="text-2xl font-bold mb-6 text-gray-800">Edit <?= ucfirst($table) ?> Record</h2>
      <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <?php 
        foreach ($columns as $col):
          $field = $col['Field'];
          $columnType = $col['Type'];
          
          if ($field === 'id') {
              continue; // skip the primary key
          }
          
          // Special handling for msgstatus, mtccode, and billingcode
          if ($field === 'msgstatus'): 
            $value = htmlspecialchars($record[$field] ?? 'no');
        ?>
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">Msg Status</label>
              <div class="flex space-x-4">
                <label class="inline-flex items-center">
                  <input type="radio" name="msgstatus" value="yes" class="form-radio" <?= ($value === 'yes') ? 'checked' : '' ?>>
                  <span class="ml-2">Yes</span>
                </label>
                <label class="inline-flex items-center">
                  <input type="radio" name="msgstatus" value="no" class="form-radio" <?= ($value === 'no') ? 'checked' : '' ?>>
                  <span class="ml-2">No</span>
                </label>
              </div>
            </div>
          <?php elseif ($field === 'mtccode'): 
              $value = htmlspecialchars($record[$field] ?? '');
          ?>
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">MTC Code</label>
              <div class="flex space-x-4">
                <input type="text" id="mtccode" name="mtccode" value="<?= $value ?>" class="w-full p-4 rounded-xl soft-input" maxlength="4">
                <button type="button" onclick="generateMtcCode()" class="px-4 py-2 bg-blue-600 text-white rounded-xl soft-button hover:bg-blue-700 transition-all">
                  Generate
                </button>
              </div>
            </div>
          <?php elseif ($field === 'billingcode'): 
              $value = htmlspecialchars($record[$field] ?? '');
          ?>
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">Billing Code</label>
              <div class="flex space-x-4">
                <input type="text" id="billingcode" name="billingcode" value="<?= $value ?>" class="w-full p-4 rounded-xl soft-input" maxlength="6">
                <button type="button" onclick="generateBillingCode()" class="px-4 py-2 bg-blue-600 text-white rounded-xl soft-button hover:bg-blue-700 transition-all">
                  Generate
                </button>
              </div>
            </div>
          <?php elseif (in_array($field, $allowedDocumentTypes)): 
              $value = htmlspecialchars($record[$field] ?? '');
          ?>
            <!-- Document/Image Field -->
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">
                <?= ucwords(str_replace('_', ' ', $field)) ?>
              </label>
              <div class="flex flex-col gap-4">
                <div class="w-full h-48 rounded-xl overflow-hidden soft-ui p-2 bg-white">
                  <img src="<?= getImageUrl($record[$field]) ?>" 
                       alt="<?= $field ?>" 
                       class="w-full h-full object-contain">
                </div>
                <input type="file" name="<?= $field ?>" 
                       class="w-full p-3 rounded-xl text-sm text-gray-600 
                              file:mr-4 file:py-2 file:px-4 file:rounded-xl file:border-0 
                              file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700
                              hover:file:bg-blue-100">
              </div>
            </div>
          <?php elseif (in_array($field, $dateTimeFields) || strpos($columnType, 'timestamp') !== false || strpos($columnType, 'datetime') !== false): 
              $value = formatDateTimeForForm($record[$field] ?? '');
          ?>
            <!-- DateTime Field -->
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">
                <?= ucwords(str_replace('_', ' ', $field)) ?>
              </label>
              <input type="datetime-local" name="<?= $field ?>" value="<?= $value ?>" 
                     class="w-full p-4 rounded-xl soft-input">
              <small class="text-gray-500">Leave empty to set to NULL</small>
            </div>
          <?php elseif (strpos($field, 'password') !== false): ?>
            <!-- Password Field -->
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">
                <?= ucwords(str_replace('_', ' ', $field)) ?>
              </label>
              <input type="password" name="<?= $field ?>" value="<?= htmlspecialchars($record[$field] ?? '') ?>" 
                     class="w-full p-4 rounded-xl soft-input" 
                     autocomplete="new-password">
            </div>
          <?php else: 
              $value = htmlspecialchars($record[$field] ?? '');
              $isReadOnly = in_array($field, $readonlyFields);
          ?>
            <!-- Regular Text Input -->
            <div class="space-y-2">
              <label class="block font-medium text-gray-700">
                <?= ucwords(str_replace('_', ' ', $field)) ?>
              </label>
              <input type="text" name="<?= $field ?>" value="<?= $value ?>" 
                     class="w-full p-4 rounded-xl soft-input <?= $isReadOnly ? 'readonly-field' : '' ?>" 
                     <?= $isReadOnly ? 'readonly' : '' ?>>
            </div>
          <?php endif; ?>
        <?php endforeach; ?>
      </div>
      
      <?php if ($table === 'users'): ?>
        <!-- Bill Dropdown -->
        <div class="space-y-2 mt-4">
          <label class="block font-medium text-gray-700">BILL</label>
          <select name="bill" id="bill" class="border border-gray-300 rounded-md p-2">
            <option value="MTF" <?= (isset($record['bill']) && $record['bill'] === 'MTF') ? 'selected' : '' ?>>MTF (Market Transaction)</option>
            <option value="STF" <?= (isset($record['bill']) && $record['bill'] === 'STF') ? 'selected' : '' ?>>STF (Spread Trading)</option>
            <option value="NTF" <?= (isset($record['bill']) && $record['bill'] === 'NTF') ? 'selected' : '' ?>>NTF (Network Transaction)</option>
            <option value="LQF" <?= (isset($record['bill']) && $record['bill'] === 'LQF') ? 'selected' : '' ?>>LQF (Liquidity Provider)</option>
            <option value="SWF" <?= (isset($record['bill']) && $record['bill'] === 'SWF') ? 'selected' : '' ?>>SWF (Swap)</option>
            <option value="GTF" <?= (isset($record['bill']) && $record['bill'] === 'GTF') ? 'selected' : '' ?>>GTF (Gas Transaction)</option>
            <option value="WDF" <?= (isset($record['bill']) && $record['bill'] === 'WDF') ? 'selected' : '' ?>>WDF (Withdrawal)</option>
            <option value="DTF" <?= (isset($record['bill']) && $record['bill'] === 'DTF') ? 'selected' : '' ?>>DTF (Deposit Transaction)</option>
            <option value="CCF" <?= (isset($record['bill']) && $record['bill'] === 'CCF') ? 'selected' : '' ?>>CCF (Conversion Currency)</option>
            <option value="OTF" <?= (isset($record['bill']) && $record['bill'] === 'OTF') ? 'selected' : '' ?>>OTF (Over-the-Counter)</option>
          </select>
        </div>
      <?php endif; ?>
      
      <div class="mt-8 flex gap-4">
        <button type="submit" name="update_main" 
                class="px-8 py-3 bg-blue-600 text-white rounded-xl soft-button hover:bg-blue-700 transition-all">
          Save Changes
        </button>
        <a href="manage.php?table=<?= urlencode($table) ?>" 
           class="px-8 py-3 bg-gray-200 text-gray-700 rounded-xl soft-button hover:bg-gray-300 transition-all">
          Cancel
        </a>
      </div>
    </form>
    
    <!-- KYC Verification Section (only if table is "users") -->
    <?php if ($table === 'users' && !empty($kycDocuments)): ?>
      <div class="mt-12 pt-8 border-t border-gray-200">
        <h3 class="text-xl font-bold mb-6 text-gray-800">KYC Verification Documents</h3>
        <?php foreach ($kycDocuments as $kyc): ?>
          <div class="mb-8 rounded-2xl p-6 bg-gray-50 soft-ui">
            <div class="flex flex-wrap justify-between items-center mb-6 gap-4">
              <div class="space-y-2">
                <p class="font-medium text-gray-700">
                  Submitted: <?= date('M d, Y H:i', strtotime($kyc['created_at'])) ?>
                </p>
                <p>
                  Status: 
                  <span class="px-3 py-1 rounded-full text-sm 
                        <?= match($kyc['status']) {
                            'approved' => 'bg-green-100 text-green-800',
                            'rejected' => 'bg-red-100 text-red-800',
                            default => 'bg-yellow-100 text-yellow-800'
                        } ?>">
                    <?= ucfirst($kyc['status']) ?>
                  </span>
                </p>
              </div>
              <form method="POST" class="flex gap-3" enctype="multipart/form-data">
                <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                <input type="hidden" name="kyc_id" value="<?= $kyc['id'] ?>">
                <button type="submit" name="kyc_action" value="approve"
                        class="px-5 py-2 bg-green-600 text-white rounded-xl soft-button hover:bg-green-700 <?= $kyc['status'] === 'approved' ? 'opacity-50 cursor-not-allowed' : '' ?>">
                  Approve
                </button>
                <button type="submit" name="kyc_action" value="reject"
                        class="px-5 py-2 bg-red-600 text-white rounded-xl soft-button hover:bg-red-700 <?= $kyc['status'] === 'rejected' ? 'opacity-50 cursor-not-allowed' : '' ?>">
                  Reject
                </button>
              </form>
            </div>
          </div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>
  </div>
</main>

<!-- JavaScript Functions for Code Generation -->
<script>
  function generateMtcCode() {
    // Generate a random 4-digit code (leading zeros possible)
    var code = Math.floor(Math.random() * 10000).toString().padStart(4, '0');
    document.getElementById('mtccode').value = code;
  }

  function generateBillingCode() {
    // Generate a random 6-digit code (leading zeros possible)
    var code = Math.floor(Math.random() * 1000000).toString().padStart(6, '0');
    document.getElementById('billingcode').value = code;
  }
</script>
</body>
</html>