<?php
$pageTitle = "Edit Record";
include 'header.php';
require 'config.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header("Location: login.php");
    exit();
}

if (!isset($_GET['table']) || !isset($_GET['id'])) {
    die('Table or record not specified.');
}

$table = $_GET['table'];
$id = $_GET['id'];

// Validate table name to prevent SQL injection
if (!preg_match('/^[a-zA-Z0-9_]+$/', $table)) {
    die('Invalid table name.');
}

// Get record data
try {
    $stmt = $conn->prepare("SELECT * FROM `$table` WHERE id = ?");
    $stmt->execute([$id]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$record) {
        die('Record not found.');
    }
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

// Get columns (skip primary key)
$stmt = $conn->prepare("SHOW COLUMNS FROM `$table`");
$stmt->execute();
$columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
$fields = [];
$protected_fields = ['email', 'password', 'google_id', 'google_email']; // Fields that should not be modified

foreach ($columns as $col) {
    if ($col['Field'] != 'id') {
        $fields[] = $col['Field'];
    }
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $set_clause = [];
    $values = [];
    
    foreach ($fields as $field) {
        // Skip protected fields to prevent modification
        if (in_array($field, $protected_fields)) {
            continue;
        }
        
        $set_clause[] = "$field = ?";
        $values[] = $_POST[$field] ?? null;
    }
    
    if (!empty($set_clause)) {
        $values[] = $id;
        $sql = "UPDATE `$table` SET " . implode(', ', $set_clause) . " WHERE id = ?";
        
        try {
            $stmt = $conn->prepare($sql);
            if ($stmt->execute($values)) {
                $success = "Record updated successfully.";
                
                // Refresh the record data
                $stmt = $conn->prepare("SELECT * FROM `$table` WHERE id = ?");
                $stmt->execute([$id]);
                $record = $stmt->fetch(PDO::FETCH_ASSOC);
            } else {
                $error = "Error updating record.";
            }
        } catch (PDOException $e) {
            $error = "Database error: " . $e->getMessage();
        }
    } else {
        $error = "No fields to update.";
    }
}

// Function to check if a field contains image data
function isImageField($fieldName) {
    return stripos($fieldName, 'image') !== false || 
           stripos($fieldName, 'photo') !== false || 
           stripos($fieldName, 'picture') !== false ||
           stripos($fieldName, 'avatar') !== false ||
           stripos($fieldName, 'icon') !== false ||
           stripos($fieldName, 'logo') !== false ||
           stripos($fieldName, 'thumbnail') !== false;
}

// Function to get file extension
function getFileExtension($filename) {
    $ext = pathinfo($filename, PATHINFO_EXTENSION);
    return strtolower($ext);
}

// Function to check if file exists
function fileExists($path) {
    return file_exists($path) && is_file($path);
}

// Function to get proper image URL
function getImageUrl($filename) {
    // Check common image directories
    $possible_paths = [
        "../uploads/$filename",
        "uploads/$filename",
        "../images/$filename",
        "images/$filename",
        "../assets/images/$filename",
        "assets/images/$filename",
        "../media/$filename",
        "media/$filename",
        $filename // In case it's already a full path
    ];
    
    foreach ($possible_paths as $path) {
        if (fileExists($path)) {
            return $path;
        }
    }
    
    // If we can't find the file, return the most likely path anyway
    return "../uploads/$filename";
}
?>

<main class="container mx-auto p-8">
  <div class="mb-6">
    <h2 class="text-2xl font-bold">Edit Record in <?= ucfirst(htmlspecialchars($table)) ?></h2>
    <?php if ($success): ?>
      <div class="bg-green-100 text-green-700 p-2 rounded mt-2"><?= $success ?></div>
    <?php endif; ?>
  </div>
  
  <?php if ($error): ?>
    <div class="bg-red-100 text-red-700 p-2 rounded mb-4"><?= $error ?></div>
  <?php endif; ?>
  
  <div class="bg-white rounded-lg shadow-md p-6">
    <form method="POST" action="">
      <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <?php foreach ($fields as $field): ?>
          <div class="mb-4">
            <label class="block text-gray-700 font-medium mb-2"><?= ucfirst(str_replace('_', ' ', $field)) ?></label>
            
            <?php if (in_array($field, $protected_fields)): ?>
              <!-- Display protected fields as read-only -->
              <input type="text" value="<?= htmlspecialchars($record[$field]) ?>" class="w-full border rounded p-2 bg-gray-100" readonly>
              <p class="text-sm text-gray-500 mt-1">This field cannot be modified for security reasons.</p>
            
            <?php elseif (isImageField($field) && !empty($record[$field])): ?>
              <!-- Special handling for image fields - Hidden input to preserve value -->
              <input type="hidden" name="<?= $field ?>" value="<?= htmlspecialchars($record[$field]) ?>">
              
              <div class="p-4 border rounded bg-gray-50">
                <?php 
                  $image_filename = htmlspecialchars($record[$field]);
                  $image_path = getImageUrl($image_filename);
                  $file_extension = getFileExtension($image_filename);
                  $image_mime = in_array($file_extension, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg']) ? true : false;
                ?>
                
                <div class="text-sm mb-2">Current value: <code class="bg-gray-200 px-1 rounded"><?= $image_filename ?></code></div>
                
                <?php if ($image_mime): ?>
                  <!-- Image preview -->
                  <div class="mb-4 text-center">
                    <img src="<?= $image_path ?>" alt="Image Preview" class="max-h-64 max-w-full inline-block rounded border object-contain">
                  </div>
                <?php else: ?>
                  <div class="mb-4 text-center p-4 border border-dashed border-gray-300 rounded">
                    <div class="text-gray-500">
                      <i class="fas fa-file-alt text-3xl mb-2"></i><br>
                      Non-image file or invalid image path
                    </div>
                  </div>
                <?php endif; ?>
                
                <!-- File path input for modification -->
                <div class="mb-3">
                  <label class="block text-sm font-medium text-gray-700 mb-1">Modify Image Path</label>
                  <input type="text" name="<?= $field ?>" value="<?= htmlspecialchars($record[$field]) ?>" class="w-full border rounded p-2">
                </div>
                
                <!-- Download and view links -->
                <div class="flex space-x-2 justify-center">
                  <a href="<?= $image_path ?>" class="inline-block bg-blue-500 text-white px-3 py-1 rounded hover:bg-blue-600 text-sm" download="<?= basename($image_filename) ?>">
                    <i class="fas fa-download mr-1"></i>Download
                  </a>
                  <a href="<?= $image_path ?>" target="_blank" class="inline-block bg-gray-500 text-white px-3 py-1 rounded hover:bg-gray-600 text-sm">
                    <i class="fas fa-external-link-alt mr-1"></i>View Full Size
                  </a>
                </div>
              </div>
            
            <?php else: ?>
              <!-- Regular input fields -->
              <input type="text" name="<?= $field ?>" value="<?= htmlspecialchars($record[$field]) ?>" class="w-full border rounded p-2">
            <?php endif; ?>
          </div>
        <?php endforeach; ?>
      </div>
      
      <div class="flex space-x-4 mt-6">
        <button type="submit" class="bg-yellow-600 text-white px-4 py-2 rounded hover:bg-yellow-700">
          <i class="fas fa-save mr-2"></i>Update Record
        </button>
        <a href="manage.php?table=<?= htmlspecialchars($table) ?>" class="bg-gray-500 text-white px-4 py-2 rounded hover:bg-gray-600">
          <i class="fas fa-times mr-2"></i>Cancel
        </a>
      </div>
    </form>
  </div>
</main>

<?php include 'footer.php'; ?>

<script>
// Add client-side image validation
document.addEventListener('DOMContentLoaded', function() {
  // For each image that failed to load, show an error placeholder
  document.querySelectorAll('img').forEach(img => {
    img.onerror = function() {
      this.style.display = 'none';
      const errorDiv = document.createElement('div');
      errorDiv.className = 'p-4 border rounded bg-red-50 text-red-500 text-center';
      errorDiv.innerHTML = '<i class="fas fa-exclamation-triangle mr-2"></i>Image not found or cannot be displayed';
      this.parentNode.appendChild(errorDiv);
    };
  });
});
</script>