<?php
// tx_lookup.php

// Enable errors for debugging (disable in production)
// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);

require 'db.php';           // Your MySQLi connection ($conn)
require '../session.php';   // isLoggedIn() and getUserDetails()

// Set default timezone
date_default_timezone_set('UTC');

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json; charset=utf-8');
    
    // Redirect if not logged in
    if (!isLoggedIn()) {
        echo json_encode([
            'success' => false,
            'error'   => 'Not authenticated.'
        ]);
        exit();
    }
    
    $hash    = trim((string)($_POST['trans_hash'] ?? ''));
    $network = trim((string)($_POST['network']    ?? 'bitcoin'));
    
    // Validate input
    if ($hash === '') {
        echo json_encode([
            'success' => false,
            'error'   => 'Transaction hash cannot be empty.'
        ]);
        exit();
    }
    
    if (!isValidHash($hash)) {
        echo json_encode([
            'success' => false,
            'error'   => 'Please enter a valid 64‑character hexadecimal transaction hash.'
        ]);
        exit();
    }
    
    $tx = lookupTransaction($hash, $network, $conn);
    
    if ($tx) {
        echo json_encode([
            'success' => true,
            'data'    => $tx
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'error'   => 'Transaction not found in our database or on the blockchain.'
        ]);
    }
    exit();
}

/**
 * Validate that the string is a 64‑character hexadecimal (Bitcoin‑style TX hash).
 *
 * @param string $h
 * @return bool
 */
function isValidHash(string $h): bool {
    return preg_match('/^[0-9a-fA-F]{64}$/', $h) === 1;
}

/**
 * Sanitize and validate network name
 *
 * @param string $network
 * @return string
 */
function sanitizeNetwork(string $network): string {
    $validNetworks = ['bitcoin', 'ethereum', 'cardano', 'bitcoin-cash', 'litecoin', 'xrp', 'tether'];
    $network = strtolower(trim($network));
    return in_array($network, $validNetworks, true) ? $network : 'bitcoin';
}

/**
 * Fetch live prices for a set of cryptocurrencies from CoinGecko with caching.
 *
 * @return array<string, float>
 */
function fetchCryptoPrices(): array {
    static $cachedPrices = null;
    static $lastFetch = null;
    
    // Cache for 5 minutes
    if ($cachedPrices !== null && $lastFetch !== null && (time() - $lastFetch) < 300) {
        return $cachedPrices;
    }
    
    $cryptos = ['bitcoin', 'ethereum', 'cardano', 'bitcoin-cash', 'litecoin'];
    $prices  = [];
    
    // Fallback hardcoded values
    $fallbackPrices = [
        'bitcoin'       => 78000.0,
        'ethereum'      => 3500.0,
        'cardano'       => 0.75,
        'bitcoin-cash'  => 350.0,
        'litecoin'      => 85.0
    ];
    
    try {
        $cryptoList = implode(',', $cryptos);
        $url = "https://api.coingecko.com/api/v3/simple/price?ids={$cryptoList}&vs_currencies=usd";
        
        $ctx = stream_context_create([
            'http' => [
                'method'  => 'GET',
                'header'  => "User-Agent: Bitvros/1.0\r\n" .
                            "Accept: application/json\r\n",
                'timeout' => 10,
            ]
        ]);

        $resp = @file_get_contents($url, false, $ctx);
        if ($resp !== false) {
            $data = json_decode($resp, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($data)) {
                foreach ($cryptos as $crypto) {
                    if (isset($data[$crypto]['usd']) && is_numeric($data[$crypto]['usd'])) {
                        $prices[$crypto] = (float)$data[$crypto]['usd'];
                    } else {
                        $prices[$crypto] = $fallbackPrices[$crypto];
                    }
                }
            } else {
                $prices = $fallbackPrices;
            }
        } else {
            $prices = $fallbackPrices;
        }
    } catch (Exception $e) {
        error_log("CoinGecko API Error: " . $e->getMessage());
        $prices = $fallbackPrices;
    }
    
    // Ensure all required cryptos have prices
    foreach ($fallbackPrices as $crypto => $fallbackPrice) {
        if (!isset($prices[$crypto])) {
            $prices[$crypto] = $fallbackPrice;
        }
    }
    
    $cachedPrices = $prices;
    $lastFetch = time();
    
    return $prices;
}

/**
 * Lookup transaction in database and blockchain
 *
 * @param string $hash
 * @param string $network
 * @param mysqli $conn
 * @return array|null
 */
function lookupTransaction(string $hash, string $network, mysqli $conn): ?array {
    $network = sanitizeNetwork($network);
    
    // 1) Look in crypto_withdrawal table
    $sql1 = "
        SELECT
            network,
            wallet,
            btc,
            total_amount,
            fee,
            transhash,
            created_at,
            status,
            confirmation
        FROM crypto_withdrawal
        WHERE transhash = ?
        LIMIT 1
    ";
    
    if ($stmt1 = $conn->prepare($sql1)) {
        $stmt1->bind_param('s', $hash);
        if ($stmt1->execute()) {
            $result = $stmt1->get_result();
            if ($row1 = $result->fetch_assoc()) {
                $stmt1->close();
                $tx = $row1;
                $tx['source'] = 'Blockchain';
                $tx['fee'] = $tx['fee'] ?? 0;
                $tx['total_amount'] = $tx['total_amount'] ?? $tx['btc'];
                return formatTransactionData($tx);
            }
        }
        $stmt1->close();
    }

    // 2) Look in payments table
    $sql2 = "
        SELECT
            wallet_address AS wallet,
            network,
            amount      AS btc,
            transhash,
            created_at,
            status,
            NULL        AS confirmation
        FROM payments
        WHERE transhash = ?
        LIMIT 1
    ";
    
    if ($stmt2 = $conn->prepare($sql2)) {
        $stmt2->bind_param('s', $hash);
        if ($stmt2->execute()) {
            $result = $stmt2->get_result();
            if ($row2 = $result->fetch_assoc()) {
                $stmt2->close();
                $tx = $row2;
                $tx['source'] = 'Payment';
                $tx['fee'] = 0;
                $tx['total_amount'] = $tx['btc'];
                return formatTransactionData($tx);
            }
        }
        $stmt2->close();
    }

    // 3) Blockchair API fallback
    return lookupOnBlockchain($hash, $network);
}

/**
 * Lookup transaction on blockchain via Blockchair API
 *
 * @param string $hash
 * @param string $network
 * @return array|null
 */
function lookupOnBlockchain(string $hash, string $network): ?array {
    $validNets = ['bitcoin', 'ethereum', 'cardano', 'bitcoin-cash', 'litecoin'];
    if (!in_array($network, $validNets, true)) {
        $network = 'bitcoin';
    }
    
    $api = "https://api.blockchair.com/{$network}/dashboards/transaction/{$hash}";
    
    try {
        $ctx = stream_context_create([
            'http' => [
                'method'  => 'GET',
                'header'  => "User-Agent: Bitvros/1.0\r\n" .
                            "Accept: application/json\r\n",
                'timeout' => 15,
            ]
        ]);
        
        $json = @file_get_contents($api, false, $ctx);
        if ($json === false) {
            return null;
        }
        
        $data = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            return null;
        }
        
        if (!isset($data['data'][$hash]['transaction'])) {
            return null;
        }
        
        $t = $data['data'][$hash]['transaction'];
        $outs = $data['data'][$hash]['outputs'] ?? [];
        
        $sumOut = 0;
        foreach ($outs as $output) {
            if (isset($output['value']) && is_numeric($output['value'])) {
                $sumOut += (float)$output['value'];
            }
        }
        
        $feeSat = isset($t['fee']) && is_numeric($t['fee']) ? (float)$t['fee'] : 0;
        $divisor = $network === 'ethereum' ? 1e18 : 1e8;
        
        $cryptoOut = $sumOut / $divisor;
        $feeCrypto = $feeSat / $divisor;
        
        $cryptoPrices = fetchCryptoPrices();
        $price = $cryptoPrices[$network] ?? $cryptoPrices['bitcoin'] ?? 50000;
        
        $tx = [
            'network'       => ucfirst(str_replace('-', ' ', $network)),
            'wallet'        => 'N/A',
            'btc'           => $cryptoOut,
            'total_amount'  => $cryptoOut,
            'fee'           => $feeCrypto,
            'fee_usd'       => round($feeCrypto * $price, 2),
            'total_usd'     => round($cryptoOut * $price, 2),
            'transhash'     => $hash,
            'created_at'    => isset($t['time']) ? date('d/m/Y - H:i:s', strtotime($t['time'])) : date('d/m/Y - H:i:s'),
            'status'        => (($t['confirmations'] ?? 0) > 0) ? 'Confirmed' : 'Unconfirmed',
            'confirmation'  => $t['confirmations'] ?? 0,
            'source'        => 'Blockchain',
            'block_height'  => $t['block_id'] ?? 'N/A',
            'size'          => $t['size'] ?? 'N/A',
        ];
        
        return $tx;
        
    } catch (Exception $e) {
        error_log("Blockchair API Error: " . $e->getMessage());
        return null;
    }
}

/**
 * Format transaction data for consistent display
 *
 * @param array $tx
 * @return array
 */
function formatTransactionData(array $tx): array {
    // Ensure required fields exist
    $tx['network'] = $tx['network'] ?? 'Unknown';
    $tx['wallet'] = $tx['wallet'] ?? 'N/A';
    $tx['btc'] = isset($tx['btc']) ? (float)$tx['btc'] : 0;
    $tx['total_amount'] = isset($tx['total_amount']) ? (float)$tx['total_amount'] : $tx['btc'];
    $tx['fee'] = isset($tx['fee']) ? (float)$tx['fee'] : 0;
    $tx['transhash'] = $tx['transhash'] ?? '';
    $tx['status'] = $tx['status'] ?? 'Unknown';
    $tx['confirmation'] = $tx['confirmation'] ?? 0;
    $tx['source'] = $tx['source'] ?? 'Database';
    
    // Format date
    if (isset($tx['created_at'])) {
        $timestamp = strtotime($tx['created_at']);
        if ($timestamp !== false) {
            $tx['created_at'] = date('d/m/Y - H:i:s', $timestamp);
        }
    } else {
        $tx['created_at'] = date('d/m/Y - H:i:s');
    }
    
    return $tx;
}

// Initialize variables
$error_message = '';
$success_message = '';
$tx = null;
$crypto_prices = fetchCryptoPrices();

// Handle form submission (non-AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['ajax'])) {
    // Redirect if not logged in
    if (!isLoggedIn()) {
        header('Location: ../login.php');
        exit();
    }
    
    $hash = trim((string)($_POST['trans_hash'] ?? ''));
    $network = trim((string)($_POST['network'] ?? 'bitcoin'));
    
    // Validate input
    if ($hash === '') {
        $error_message = 'Transaction hash cannot be empty.';
    } elseif (!isValidHash($hash)) {
        $error_message = 'Please enter a valid 64‑character hexadecimal transaction hash.';
    } else {
        $tx = lookupTransaction($hash, $network, $conn);
        
        if ($tx) {
            $success_message = 'Transaction found successfully.';
        } else {
            $error_message = 'Transaction not found in our database or on the blockchain.';
        }
    }
}

// Redirect if not logged in (for GET requests)
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>bitvros.com - Blockchain Explorer</title>
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <!-- Chart.js and ApexCharts (if you wish to add charts later) -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
      <!-- Favicon -->
  <link rel="icon" href="https://bitvros.com/session/favicon.png" type="image/x-icon">
  
  
    <script src="https://cdn.jsdelivr.net/npm/apexcharts"></script>
    <!-- Custom Styles -->
    
     <?php include './includes/header.php'; ?>

    <style>
        /* Custom color scheme and user-friendly styles */
        .bg-crypto-light {
            background-color: #f8fafc;
        }
        .glass-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 1rem;
            padding: 1.5rem;
            border: 1px solid rgba(203, 213, 224, 0.3);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }
        .gradient-btn {
            background: linear-gradient(90deg, #1f2937, #374151);
            transition: all 0.3s ease;
        }
        .gradient-btn:hover {
            background: linear-gradient(90deg, #111827, #1f2937);
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(31, 41, 55, 0.3);
        }
        .search-input {
            border: 1px solid #d1d5db;
            background-color: #ffffff;
            color: #374151;
            transition: all 0.3s ease;
        }
        .search-input:focus {
            border-color: #059669;
            box-shadow: 0 0 0 3px rgba(5, 150, 105, 0.1);
        }
        .price-ticker {
            background: linear-gradient(90deg, rgba(255, 255, 255, 0.9), rgba(248, 250, 252, 0.9));
            backdrop-filter: blur(5px);
            border-radius: 0.5rem;
            border: 1px solid rgba(203, 213, 224, 0.3);
        }
        .crypto-card {
            transition: all 0.3s ease;
        }
        .crypto-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px -5px rgba(5, 150, 105, 0.1);
        }
        .network-selector {
            background: rgba(5, 150, 105, 0.1);
            border: 1px solid rgba(5, 150, 105, 0.2);
            transition: all 0.3s ease;
        }
        .network-selector:hover {
            background: rgba(5, 150, 105, 0.15);
            border-color: rgba(5, 150, 105, 0.3);
        }
        body {
            font-family: 'Inter', sans-serif;
        }
        .tx-details-card {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-left: 4px solid #059669;
        }
        .detail-item {
            padding: 12px 0;
            border-bottom: 1px solid #e5e7eb;
        }
        .detail-item:last-child {
            border-bottom: none;
        }
        .detail-label {
            font-weight: 600;
            color: #374151;
            min-width: 150px;
        }
        .detail-value {
            color: #6b7280;
            word-break: break-all;
        }
        .loading {
            opacity: 0.6;
            pointer-events: none;
        }
        .spinner {
            border: 3px solid #f3f4f6;
            border-radius: 50%;
            border-top: 3px solid #059669;
            width: 20px;
            height: 20px;
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body class="bg-crypto-light text-gray-900">
    <!-- Main Container -->
    <div class="min-h-screen flex flex-col">
       <?php include './includes/nav.php'; ?>

        <!-- Main Content -->
        <div class="flex-grow container mx-auto px-4 py-8">
            <!-- Transaction Lookup Section -->
            <div class="mb-8">
                <div class="glass-card">
                    <h2 class="text-2xl font-bold mb-6 text-gray-800">Transaction Lookup</h2>
                    
                    <!-- Error Message -->
                    <?php if (!empty($error_message)): ?>
                        <div class="bg-red-50 border border-red-200 text-red-800 px-4 py-3 rounded-lg mb-6 flex items-start">
                            <div class="text-red-500 mr-3 mt-0.5">
                                <i class="fas fa-exclamation-circle"></i>
                            </div>
                            <div>
                                <p class="font-medium">Transaction Not Found</p>
                                <p class="text-sm"><?php echo htmlspecialchars($error_message); ?></p>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- Success Message -->
                    <?php if (!empty($success_message)): ?>
                        <div class="bg-green-50 border border-green-200 text-green-800 px-4 py-3 rounded-lg mb-6 flex items-start">
                            <div class="text-green-500 mr-3 mt-0.5">
                                <i class="fas fa-check-circle"></i>
                            </div>
                            <div>
                                <p class="font-medium">Transaction Found</p>
                                <p class="text-sm"><?php echo htmlspecialchars($success_message); ?></p>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Search Form -->
                    <form method="post" action="" class="mb-6" id="searchForm">
                        <div class="flex flex-col md:flex-row gap-4">
                            <div class="flex-grow">
                                <label for="trans_hash" class="block text-sm font-medium text-gray-700 mb-2">
                                    <i class="fas fa-search mr-2 text-emerald-600"></i>Transaction Hash
                                </label>
                                <input 
                                    type="text" 
                                    id="trans_hash" 
                                    name="trans_hash" 
                                    class="w-full bg-white border border-gray-300 text-gray-900 px-4 py-3 rounded-lg search-input focus:outline-none" 
                                    placeholder="Enter 64-character transaction hash" 
                                    value="<?php echo isset($_POST['trans_hash']) ? htmlspecialchars($_POST['trans_hash']) : ''; ?>"
                                    required
                                    maxlength="64"
                                    pattern="[0-9a-fA-F]{64}"
                                >
                            </div>
                            <div class="md:w-1/4">
                                <label for="network" class="block text-sm font-medium text-gray-700 mb-2">
                                    <i class="fas fa-network-wired mr-2 text-emerald-600"></i>Network
                                </label>
                                <select 
                                    id="network" 
                                    name="network" 
                                    class="w-full bg-white border border-gray-300 text-gray-900 px-4 py-3 rounded-lg search-input focus:outline-none"
                                >
                                    <?php
                                    $networks = [
                                        'bitcoin' => 'Bitcoin',
                                        'bitcoin-cash' => 'Bitcoin Cash',
                                        'litecoin' => 'Litecoin',
                                        'ethereum' => 'Ethereum',
                                        'cardano' => 'Cardano',
                                        'xrp' => 'XRP',
                                        'tether' => 'Tether'
                                    ];
                                    $selectedNetwork = $_POST['network'] ?? 'bitcoin';
                                    foreach ($networks as $value => $label) {
                                        $selected = ($value === $selectedNetwork) ? 'selected' : '';
                                        echo "<option value=\"{$value}\" {$selected}>{$label}</option>";
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="md:w-auto self-end">
                                <button type="submit" class="w-full md:w-auto gradient-btn text-white px-6 py-3 rounded-lg font-medium" id="searchBtn">
                                    <i class="fas fa-search-dollar mr-2"></i>Search
                                </button>
                            </div>
                        </div>
                    </form>
                    
                     <!-- Transaction Details Display -->
            <?php if ($tx): ?>
            <div class="mb-8">
                <div class="glass-card tx-details-card">
                    <h3 class="text-xl font-bold mb-6 text-gray-800 flex items-center">
                        <i class="fas fa-receipt mr-3 text-emerald-600"></i>
                        Transaction Details
                        <span class="ml-auto text-sm font-medium px-3 py-1 bg-emerald-100 text-emerald-800 rounded-full">
                            <?php echo htmlspecialchars($tx['source']); ?>
                        </span>
                    </h3>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div class="space-y-4">
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Transaction Hash:</span>
                                <span class="detail-value font-mono text-sm"><?php echo htmlspecialchars($tx['transhash']); ?></span>
                            </div>
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Network:</span>
                                <span class="detail-value"><?php echo htmlspecialchars($tx['network']); ?></span>
                            </div>
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Status:</span>
                                <span class="detail-value">
                                    <?php if ($tx['status'] === 'confirmed'): ?>
                                        <span class="inline-flex items-center px-2 py-1 bg-green-100 text-green-800 rounded-full text-sm">
                                            <i class="fas fa-check-circle mr-1"></i>
                                            <?php echo htmlspecialchars($tx['status']); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="inline-flex items-center px-2 py-1 bg-yellow-100 text-yellow-800 rounded-full text-sm">
                                            <i class="fas fa-clock mr-1"></i>
                                            <?php echo htmlspecialchars($tx['status']); ?>
                                        </span>
                                    <?php endif; ?>
                                </span>
                            </div>
                           
                           
                        </div>
                        
                        <div class="space-y-4">
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Amount:</span>
                                <span class="detail-value font-semibold text-lg"><?php echo number_format($tx['total_amount'], 2); ?> USD</span>
                            </div>
                            <?php if (isset($tx['total_usd'])): ?>
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">USD Value:</span>
                                <span class="detail-value font-semibold text-lg text-green-600">$<?php echo number_format($tx['total_usd'], 2); ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Fee:</span>
                                <span class="detail-value"><?php echo $tx['fee'] ? number_format($tx['fee'], 8) : 'N/A'; ?></span>
                            </div>
                            <?php if (isset($tx['fee_usd'])): ?>
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Fee (USD):</span>
                                <span class="detail-value">$<?php echo number_format($tx['fee_usd'], 2); ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="detail-item flex flex-col md:flex-row">
                                <span class="detail-label">Date:</span>
                                <span class="detail-value"><?php echo htmlspecialchars($tx['created_at']); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <?php if (isset($tx['block_height']) && $tx['block_height'] !== 'N/A'): ?>
                    <div class="mt-6 pt-4 border-t border-gray-200">
                        <div class="flex flex-wrap gap-4 text-sm text-gray-600">
                            <div><strong>Block Height:</strong> <?php echo htmlspecialchars($tx['block_height']); ?></div>
                            <?php if (isset($tx['size']) && $tx['size'] !== 'N/A'): ?>
                            <div><strong>Size:</strong> <?php echo htmlspecialchars($tx['size']); ?> bytes</div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>
                    
                 
                        <!-- Quick Networks Selection -->
                    <div class="flex flex-wrap justify-center gap-4 mt-6">
                        <div class="flex items-center space-x-2 network-selector px-4 py-3 rounded-lg cursor-pointer" data-network="bitcoin">
                            <img src="https://bitvros.com/cryptos/bitcoin.png" alt="BTC" class="w-6 h-6">
                            <span class="text-sm font-medium text-gray-700">Bitcoin</span>
                        </div>
                        <div class="flex items-center space-x-2 network-selector px-4 py-3 rounded-lg cursor-pointer" data-network="ethereum">
                            <img src="https://bitvros.com/cryptos/ethereum.png" alt="ETH" class="w-6 h-6">
                            <span class="text-sm font-medium text-gray-700">Ethereum</span>
                        </div>
                        <div class="flex items-center space-x-2 network-selector px-4 py-3 rounded-lg cursor-pointer" data-network="litecoin">
                            <img src="https://bitvros.com/cryptos/ltc.png" alt="LTC" class="w-6 h-6">
                            <span class="text-sm font-medium text-gray-700">Litecoin</span>
                        </div>
                        <div class="flex items-center space-x-2 network-selector px-4 py-3 rounded-lg cursor-pointer" data-network="cardano">
                            <img src="https://bitvros.com/cryptos/ada.png" alt="ADA" class="w-6 h-6">
                            <span class="text-sm font-medium text-gray-700">Cardano</span>
                        </div>
                        <div class="flex items-center space-x-2 network-selector px-4 py-3 rounded-lg cursor-pointer" data-network="bitcoin-cash">
                            <img src="https://bitvros.com/cryptos/bch.png" alt="BCH" class="w-6 h-6">
                            <span class="text-sm font-medium text-gray-700">Bitcoin Cash</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Live Price Ticker -->
            <div class="mb-8">
                <div class="glass-card">
                    <h3 class="text-xl font-bold mb-4 text-gray-800 flex items-center">
                        <i class="fas fa-chart-line mr-3 text-emerald-600"></i>
                        Live Cryptocurrency Prices
                    </h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                        <?php foreach ($crypto_prices as $crypto => $price): ?>
                        <div class="price-ticker p-4 crypto-card">
                            <div class="flex items-center justify-between">
                                <div class="flex items-center space-x-2">
                                    <img src="https://bitvros.com/cryptos/<?php echo $crypto; ?>.png" alt="<?php echo strtoupper($crypto); ?>" class="w-8 h-8">
                                    <div>
                                        <p class="text-sm font-medium text-gray-700"><?php echo ucfirst(str_replace('-', ' ', $crypto)); ?></p>
                                        <p class="text-xs text-gray-500"><?php echo strtoupper(str_replace('-', '', $crypto)); ?></p>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <p class="text-lg font-bold text-gray-900">$<?php echo number_format($price, 2); ?></p>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

            <!-- Help Section -->
            <div class="mb-8">
                <div class="glass-card">
                    <h3 class="text-xl font-bold mb-4 text-gray-800 flex items-center">
                        <i class="fas fa-question-circle mr-3 text-emerald-600"></i>
                        How to Use the Blockchain Explorer
                    </h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <h4 class="font-semibold text-gray-700 mb-2">Transaction Hash Format</h4>
                            <p class="text-sm text-gray-600 mb-4">Enter a 64-character hexadecimal transaction hash (e.g., a1b2c3d4e5f6...)</p>
                            
                            <h4 class="font-semibold text-gray-700 mb-2">Supported Networks</h4>
                            <ul class="text-sm text-gray-600 space-y-1">
                                <li>• Bitcoin (BTC)</li>
                                <li>• Ethereum (ETH)</li>
                                <li>• Litecoin (LTC)</li>
                                <li>• Cardano (ADA)</li>
                                <li>• Bitcoin Cash (BCH)</li>
                            </ul>
                        </div>
                        <div>
                            <h4 class="font-semibold text-gray-700 mb-2">Data Sources</h4>
                            <p class="text-sm text-gray-600 mb-4">We search both our internal database and live blockchain data to provide the most comprehensive transaction information.</p>
                            
                            <h4 class="font-semibold text-gray-700 mb-2">Real-time Prices</h4>
                            <p class="text-sm text-gray-600">Live cryptocurrency prices are updated every 5 minutes via CoinGecko API.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Footer -->
        <footer class="bg-gray-800 text-white py-6">
            <div class="container mx-auto px-4 text-center">
                <p>&copy; 2024 Bitvros. All rights reserved.</p>
                <p class="text-sm text-gray-400 mt-2">Blockchain Explorer - Real-time Transaction Lookup</p>
            </div>
        </footer>
    </div>

    <!-- JavaScript -->
    <script>
        // Network selector functionality
        document.querySelectorAll('.network-selector').forEach(selector => {
            selector.addEventListener('click', function() {
                const network = this.dataset.network;
                const networkSelect = document.getElementById('network');
                
                // Update select value
                networkSelect.value = network;
                
                // Visual feedback
                document.querySelectorAll('.network-selector').forEach(s => {
                    s.classList.remove('bg-emerald-100', 'border-emerald-300');
                    s.classList.add('bg-emerald-50', 'border-emerald-200');
                });
                
                this.classList.remove('bg-emerald-50', 'border-emerald-200');
                this.classList.add('bg-emerald-100', 'border-emerald-300');
            });
        });

        // Form submission with loading state
        document.getElementById('searchForm').addEventListener('submit', function(e) {
            const btn = document.getElementById('searchBtn');
            const originalText = btn.innerHTML;
            
            btn.innerHTML = '<div class="spinner inline-block mr-2"></div>Searching...';
            btn.disabled = true;
            
            // Re-enable button after 10 seconds as fallback
            setTimeout(() => {
                btn.innerHTML = originalText;
                btn.disabled = false;
            }, 10000);
        });

        // Auto-format transaction hash input
        document.getElementById('trans_hash').addEventListener('input', function(e) {
            // Remove non-hexadecimal characters
            let value = e.target.value.replace(/[^0-9a-fA-F]/g, '');
            
            // Limit to 64 characters
            if (value.length > 64) {
                value = value.substring(0, 64);
            }
            
            e.target.value = value;
        });

        // Initialize network selector visual state
        document.addEventListener('DOMContentLoaded', function() {
            const currentNetwork = document.getElementById('network').value;
            const currentSelector = document.querySelector(`[data-network="${currentNetwork}"]`);
            
            if (currentSelector) {
                currentSelector.classList.remove('bg-emerald-50', 'border-emerald-200');
                currentSelector.classList.add('bg-emerald-100', 'border-emerald-300');
            }
        });

        // Copy transaction hash to clipboard
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(function() {
                // Show success message
                const toast = document.createElement('div');
                toast.className = 'fixed top-4 right-4 bg-green-500 text-white px-4 py-2 rounded-lg shadow-lg z-50';
                toast.textContent = 'Transaction hash copied to clipboard!';
                document.body.appendChild(toast);
                
                setTimeout(() => {
                    toast.remove();
                }, 3000);
            }).catch(function(err) {
                console.error('Could not copy text: ', err);
            });
        }

        // Add click-to-copy functionality for transaction hash
        document.addEventListener('DOMContentLoaded', function() {
            const hashElements = document.querySelectorAll('.detail-value.font-mono');
            hashElements.forEach(element => {
                element.style.cursor = 'pointer';
                element.title = 'Click to copy';
                element.addEventListener('click', function() {
                    copyToClipboard(this.textContent);
                });
            });
        });
    </script>
</body>
</html>