<?php
// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Required parameters
$required_params = [
    'user_id', 
    'email', 
    'network', 
    'wallet', 
    'btc_amount', 
    'total_amount', 
    'fee', 
    'transhash'
];

// Check for missing parameters
$missing_params = [];
foreach ($required_params as $param) {
    if (!isset($_GET[$param])) {
        $missing_params[] = $param;
    } elseif (trim($_GET[$param]) === '') {
        $missing_params[] = $param;
    }
}

if (!empty($missing_params)) {
    die(json_encode([
        'status' => 'error',
        'message' => 'Missing parameters: ' . implode(', ', $missing_params)
    ]));
}

// Validate numeric parameters
$numeric_params = ['btc_amount', 'total_amount', 'fee'];
foreach ($numeric_params as $param) {
    if (!is_numeric($_GET[$param])) {
        die("Invalid value for $param. Must be numeric.");
    }
}

// Sanitize inputs
$user_id = htmlspecialchars($_GET['user_id']);
$email = filter_var($_GET['email'], FILTER_SANITIZE_EMAIL);
$network = htmlspecialchars($_GET['network']);
$wallet = htmlspecialchars($_GET['wallet']);
$btc_amount = floatval($_GET['btc_amount']);
$total_amount = floatval($_GET['total_amount']);
$fee = floatval($_GET['fee']);
$transhash = htmlspecialchars($_GET['transhash']);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>BTC Transaction Confirmation</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .swipe-track {
            width: 100%;
            max-width: 300px;
            height: 60px;
            background: #e2e8f0;
            border-radius: 30px;
            position: relative;
            margin: 0 auto;
        }
        .swipe-button {
            width: 50px;
            height: 50px;
            background: #881ff9;
            border-radius: 50%;
            position: absolute;
            left: 5px;
            top: 5px;
            transition: left 0.3s ease;
            cursor: pointer;
        }
        .swipe-button svg {
            width: 24px;
            height: 24px;
        }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto max-w-md min-h-screen p-4">
        <!-- Confirmation Card -->
        <div class="bg-white rounded-xl shadow-lg p-6 mb-6">
            <!-- Header -->
            <div class="text-center mb-6">
                <div class="w-16 h-16 bg-orange-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <img src="https://bitvros.com/cryptos/bitcoin.png" 
                         alt="Bitcoin" 
                         class="w-8 h-8">
                </div>
                <h1 class="text-2xl font-bold text-gray-800 mb-2">
                    Confirm BTC Transfer
                </h1>
                <p class="text-gray-600">Swipe to confirm transaction</p>
            </div>

            <!-- Transaction Details -->
            <div class="space-y-4 mb-8">
                <div class="flex justify-between">
                    <span class="text-gray-600">Amount (USD)</span>
                    <span class="font-semibold">$<?= number_format($total_amount, 2) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Network Fee</span>
                    <span class="font-semibold" id="feeDisplay"><?= $fee ?> BTC</span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Recipient</span>
                    <span class="font-semibold text-blue-600 break-all"><?= $wallet ?></span>
                </div>
            </div>

            <!-- Swipe Confirmation -->
            <form id="confirmForm" method="POST" action="process_transaction.php">
                <input type="hidden" name="user_id" value="<?= $user_id ?>">
                <input type="hidden" name="email" value="<?= $email ?>">
                <input type="hidden" name="network" value="<?= $network ?>">
                <input type="hidden" name="wallet" value="<?= $wallet ?>">
                <input type="hidden" name="btc_amount" value="<?= $btc_amount ?>">
                <input type="hidden" name="total_amount" value="<?= $total_amount ?>">
                <input type="hidden" name="fee" value="<?= $fee ?>">
                <input type="hidden" name="transhash" value="<?= $transhash ?>">

                <div class="swipe-track" id="swipeTrack">
                    <div class="swipe-button" id="swipeButton">
                        <svg class="text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                                  d="M14 5l7 7m0 0l-7 7m7-7H3"></path>
                        </svg>
                    </div>
                </div>
            </form>
        </div>

        <!-- Live Price Ticker -->
        <div class="bg-white rounded-xl shadow p-4 text-center">
            <p class="text-sm text-gray-600" id="btcPriceTicker">
                Loading BTC price...
            </p>
        </div>
    </div>

    <script>
        // Swipe Logic
        const swipeButton = document.getElementById('swipeButton');
        const swipeTrack = document.getElementById('swipeTrack');
        let isDragging = false;
        let startPosition = 0;
        let currentTranslate = 0;

        const getMaxTranslate = () => 
            swipeTrack.offsetWidth - swipeButton.offsetWidth - 10;

        // Touch Events
        swipeButton.addEventListener('touchstart', e => {
            isDragging = true;
            startPosition = e.touches[0].clientX;
        });

        swipeButton.addEventListener('touchmove', e => {
            if (!isDragging) return;
            const currentPosition = e.touches[0].clientX;
            const diff = currentPosition - startPosition;
            currentTranslate = Math.min(Math.max(5, diff), getMaxTranslate());
            swipeButton.style.left = `${currentTranslate}px`;
        });

        swipeButton.addEventListener('touchend', () => {
            if (currentTranslate >= getMaxTranslate() * 0.9) {
                document.getElementById('confirmForm').submit();
            } else {
                swipeButton.style.left = '5px';
            }
            isDragging = false;
        });

        // Mouse Events
        swipeButton.addEventListener('mousedown', e => {
            isDragging = true;
            startPosition = e.clientX;
        });

        document.addEventListener('mousemove', e => {
            if (!isDragging) return;
            const currentPosition = e.clientX;
            const diff = currentPosition - startPosition;
            currentTranslate = Math.min(Math.max(5, diff), getMaxTranslate());
            swipeButton.style.left = `${currentTranslate}px`;
        });

        document.addEventListener('mouseup', () => {
            if (currentTranslate >= getMaxTranslate() * 0.9) {
                document.getElementById('confirmForm').submit();
            } else {
                swipeButton.style.left = '5px';
            }
            isDragging = false;
        });

        // Live Price Updates
        async function updateBtcPrice() {
            try {
                const response = await fetch('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=usd');
                const data = await response.json();
                const price = data.bitcoin.usd.toLocaleString('en-US', {
                    style: 'currency',
                    currency: 'USD'
                });
                document.getElementById('btcPriceTicker').textContent = 
                    `1 BTC = ${price}`;
                
                // Update fee display
                const feeBTC = <?= $fee ?>;
                const feeUSD = (feeBTC * data.bitcoin.usd).toFixed(2);
                document.getElementById('feeDisplay').textContent = 
                    `${feeBTC.toFixed(8)} BTC ($${feeUSD})`;
                    
            } catch (error) {
                document.getElementById('btcPriceTicker').textContent = 'Error fetching BTC price.';
                console.error('Error fetching BTC price:', error);
            }
        }

        // Initial load
        updateBtcPrice();
        // Update every 30 seconds
        setInterval(updateBtcPrice, 30000);
    </script>
</body>
</html>
