<?php
// Enable error reporting (for development only)
// Remove or disable in production
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in; if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id      = $userDetails['google_id'];
$name           = $userDetails['name'];
$email          = $userDetails['email'];
$created_at     = $userDetails['created_at'];
$picture        = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin            = $userDetails['pin'];
$status         = $userDetails['status'];
$country        = $userDetails['country'];
$phone          = $userDetails['phone'];
$usd_balance    = isset($userDetails['balance']) ? (float)$userDetails['balance'] : 0.00;
$btc            = isset($userDetails['btc']) ? (float)$userDetails['btc'] : 0.000000;
$eth            = isset($userDetails['eth']) ? (float)$userDetails['eth'] : 0.000000;
$usdt           = isset($userDetails['usdt']) ? (float)$userDetails['usdt'] : 0.000000;
$xrp            = isset($userDetails['xrp']) ? (float)$userDetails['xrp'] : 0.000000;
$doge           = isset($userDetails['doge']) ? (float)$userDetails['doge'] : 0.000000;
$ada            = isset($userDetails['ada']) ? (float)$userDetails['ada'] : 0.000000;
// FIX: The 'sol' column contains USD value, not SOL tokens
$sol_usd_value  = isset($userDetails['sol']) ? (float)$userDetails['sol'] : 0.000000;
$extra          = $userDetails['extra'];
$update_time    = $userDetails['update_time'];

// Define cache file and cache duration (2 hours)
$cache_file = "sol_price_cache.json";
$cache_duration = 7200; // 2 hours

// Function to fetch SOL price from CoinGecko API using cURL
function fetch_sol_price() {
    $api_url = "https://api.coingecko.com/api/v3/simple/price?ids=solana&vs_currencies=usd";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Ignore SSL verification issues
    curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Timeout after 10 seconds
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Ensure response is valid and HTTP status code is 200 (OK)
    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return false;
}

// Use cache if valid
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_duration) {
    $data = json_decode(file_get_contents($cache_file), true);
} else {
    // Fetch fresh SOL price
    $data = fetch_sol_price();

    // Store new price in cache if valid
    if ($data && isset($data['solana']['usd'])) {
        file_put_contents($cache_file, json_encode($data));
    }
}

// Ensure SOL price is available
$sol_price = $data['solana']['usd'] ?? 100.00; // Default fallback price

// Calculate actual SOL tokens from USD value
$sol_tokens = $sol_price > 0 ? ($sol_usd_value / $sol_price) : 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Solana (SOL) | bitvros.com</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css" rel="stylesheet">
    <link rel="icon" href="https://bitvros.com/session/favicon.png" type="image/x-icon">
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <style>
        .sol-bg { background: linear-gradient(135deg, #9945FF 0%, #14F195 100%); }
        .sol-text { color: #9945FF; }
        .sol-secondary { color: #14F195; }
        .btn-sol {
            background: linear-gradient(135deg, #9945FF 0%, #14F195 100%);
            color: white;
            transition: all 0.3s;
            border: none;
        }
        .btn-sol:hover {
            background: linear-gradient(135deg, #8A2BE2 0%, #00FF7F 100%);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(153, 69, 255, 0.4);
        }
        .tradingview-widget-container {
            width: 100%;
            min-height: 450px;
        }
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        .progress-bar {
            height: 4px;
            background-color: #e2e8f0;
            border-radius: 2px;
            overflow: hidden;
        }
        .progress-value {
            height: 100%;
            background: linear-gradient(90deg, #9945FF 0%, #14F195 100%);
            transition: width 0.5s ease;
        }
        .description-collapse {
            max-height: 100px;
            overflow: hidden;
            position: relative;
        }
        .description-collapse::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(to top, rgba(255,255,255,1), rgba(255,255,255,0));
        }
        .description-expanded {
            max-height: none;
        }
        .speed-indicator {
            background: linear-gradient(135deg, #9945FF 0%, #14F195 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: inline-block;
            margin-top: 4px;
        }
        .sol-card {
            background: linear-gradient(135deg, #F3F0FF 0%, #E6FFF7 100%);
            border: 2px solid #9945FF;
        }
        .price-bounce {
            animation: bounce 2s infinite;
        }
        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-10px); }
            60% { transform: translateY(-5px); }
        }
        .feature-tag {
            background: linear-gradient(135deg, #9945FF 0%, #14F195 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .fast-tag {
            background: linear-gradient(135deg, #FF6B6B 0%, #FF8E53 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .nft-tag {
            background: linear-gradient(135deg, #FF6B9D 0%, #C44569 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .performance-text {
            font-family: 'Courier New', monospace;
            color: #9945FF;
            font-weight: bold;
        }
        .glow-effect {
            box-shadow: 0 0 20px rgba(153, 69, 255, 0.3);
        }
        .tps-counter {
            font-family: 'Courier New', monospace;
            font-size: 18px;
            color: #14F195;
            font-weight: bold;
            text-shadow: 0 0 10px rgba(20, 241, 149, 0.5);
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen p-4">
    <div class="max-w-4xl mx-auto bg-white p-4 md:p-6 rounded-2xl shadow-sm">
        <!-- Header -->
        <header class="flex items-center justify-between mb-8">
            <button onclick="history.back()" class="sol-text p-2 rounded-full hover:bg-gray-100 transition-colors">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                </svg>
            </button>
            <h1 class="text-xl font-semibold text-gray-800 flex items-center">
                <img src="https://bitvros.com/cryptos/solana.png" class="w-8 h-8 mr-2" alt="Solana">
                Solana (SOL)
            </h1>
            <div class="p-2 rounded-full hover:bg-gray-100 transition-colors">
                <div id="connection-status" class="w-3 h-3 rounded-full bg-green-500"></div>
            </div>
        </header>

        <!-- Price Section -->
        <section class="text-center mb-8 fade-in">
            <div class="price-text text-4xl font-bold mb-2 flex items-center justify-center">
                <span id="sol-price" class="text-gray-800">$0.00</span>
                <span id="sol-arrow" class="ml-2"></span>
            </div>
            <div id="sol-change" class="text-lg font-medium text-gray-600">0.00% (24h)</div>
            <div class="speed-indicator">
                ⚡ 65,000+ TPS Theoretical
            </div>
            <div class="text-sm font-medium text-green-500 mt-2">
                1 USD ≈ <span id="sol-equivalent" class="text-gray-700">0.00 SOL</span>
            </div>
            <div class="flex justify-center mt-2 space-x-1">
                <span class="feature-tag">PoH</span>
                <span class="fast-tag">⚡ Ultra Fast</span>
                <span class="nft-tag">🎨 NFTs</span>
                <span class="feature-tag">DeFi Hub</span>
            </div>
        </section>

        <!-- TradingView Widget Section -->
        <section class="mb-6" style="width: 100%;">
            <div class="tradingview-widget-container" style="width: 100%; height: 420px;">
                <div class="tradingview-widget-container__widget" style="width: 100%; height: 100%;"></div>
                <script type="text/javascript" src="https://s3.tradingview.com/external-embedding/embed-widget-mini-symbol-overview.js" async>
                {
                    "symbol": "BINANCE:SOLUSDT",
                    "width": "100%",
                    "height": "250",
                    "locale": "en",
                    "dateRange": "12M",
                    "colorTheme": "light",
                    "isTransparent": true,
                    "autosize": false,
                    "largeChartUrl": ""
                }
                </script>
            </div>
        </section>

        <!-- Wallet Card -->
        <section class="wallet-card sol-card rounded-xl p-6 mb-8 fade-in glow-effect">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm">Your SOL Balance</p>
                    <h2 class="text-2xl font-bold text-gray-800 mt-1">
                        <?php echo number_format($sol_tokens, 6); ?> SOL
                    </h2>
                    <p class="text-gray-600 mt-1">
                        ≈ $<span id="sol-balance-usd"><?php echo number_format($sol_usd_value, 2); ?></span> USD
                    </p>
                    <p class="performance-text text-xs mt-1">Built for web3 scale</p>
                </div>
                <img src="https://bitvros.com/cryptos/solana.png" class="w-12 h-12" alt="Solana">
            </div>
            <div class="mt-4 flex space-x-2">
                <a href="sols.php" class="btn-sol py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-up mr-2"></i>Send
                </a>
                <a href="solr.php" class="btn-sol py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-down mr-2"></i>Receive
                </a>
            </div>
        </section>

        <!-- Performance Stats -->
        <section class="bg-gradient-to-r from-purple-50 to-green-50 rounded-xl p-6 mb-8 fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Network Performance</h3>
            <div class="grid grid-cols-2 gap-4">
                <div class="text-center">
                    <div class="tps-counter" id="current-tps">~2,500</div>
                    <div class="text-xs text-gray-500">Current TPS</div>
                </div>
                <div class="text-center">
                    <div class="tps-counter">~400ms</div>
                    <div class="text-xs text-gray-500">Block Time</div>
                </div>
                <div class="text-center">
                    <div class="tps-counter">$0.00025</div>
                    <div class="text-xs text-gray-500">Avg. Fee</div>
                </div>
                <div class="text-center">
                    <div class="tps-counter" id="validator-count">~1,900</div>
                    <div class="text-xs text-gray-500">Validators</div>
                </div>
            </div>
        </section>

        <!-- Market Data -->
        <section class="bg-gray-50 rounded-xl p-6 mb-8 fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Market Overview</h3>
            <div class="space-y-4 text-sm">
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap Rank</span>
                        <span id="market-cap-rank">#5</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap</span>
                        <span id="market-cap">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>24h Volume</span>
                        <span id="volume">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Circulating Supply</span>
                        <span id="supply">0 SOL</span>
                    </div>
                    <div class="progress-bar mt-1">
                        <div id="supply-progress" class="progress-value" style="width: 0%"></div>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        <span id="circulating-supply">0</span> / <span id="total-supply">588M</span> SOL
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Staking Participation</span>
                        <span id="staking-rate" class="text-purple-600 font-medium">~68%</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Percentage of SOL staked
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Average APY</span>
                        <span id="staking-apy" class="text-green-600 font-medium">~7.5%</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Annual staking rewards
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>DeFi TVL</span>
                        <span id="defi-tvl" class="text-blue-600 font-medium">$1.2B+</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Total Value Locked in DeFi
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Active Projects</span>
                        <span id="active-projects" class="text-indigo-600 font-medium">400+</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Built on Solana
                    </div>
                </div>
            </div>
        </section>

        <!-- Speed & Performance Info -->
        <section class="bg-gradient-to-r from-purple-50 to-green-50 border-2 border-purple-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span class="text-purple-600 text-lg">⚡</span>
                </div>
                <div>
                    <h4 class="font-semibold text-purple-800 mb-1">High-Performance Blockchain</h4>
                    <p class="text-sm text-purple-700">
                        Solana combines Proof of History (PoH) with Proof of Stake (PoS) to achieve unprecedented speed and low costs. The network can theoretically handle 65,000+ transactions per second.
                    </p>
                    <div class="mt-2 space-y-1">
                        <div class="text-xs text-purple-600">✓ 65,000+ TPS theoretical capacity</div>
                        <div class="text-xs text-purple-600">✓ 400ms block times</div>
                        <div class="text-xs text-purple-600">✓ $0.00025 average transaction fee</div>
                        <div class="text-xs text-purple-600">✓ Proof of History innovation</div>
                        <div class="text-xs text-purple-600">✓ Rust-based smart contracts</div>
                    </div>
                </div>
            </div>
        </section>

        <!-- Ecosystem Info -->
        <section class="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span class="text-blue-600 text-lg">🌐</span>
                </div>
                <div>
                    <h4 class="font-semibold text-blue-800 mb-1">Thriving Ecosystem</h4>
                    <p class="text-sm text-blue-700">
                        Solana hosts a vibrant ecosystem of DeFi protocols, NFT marketplaces, Web3 applications, and gaming projects. The network's speed and low costs make it ideal for consumer applications.
                    </p>
                    <div class="mt-2 space-y-1">
                        <div class="text-xs text-blue-600">✓ 400+ projects building on Solana</div>
                        <div class="text-xs text-blue-600">✓ Leading NFT marketplaces</div>
                        <div class="text-xs text-blue-600">✓ Major DeFi protocols</div>
                        <div class="text-xs text-blue-600">✓ Web3 gaming hub</div>
                        <div class="text-xs text-blue-600">✓ Mobile-first approach</div>
                    </div>
                </div>
            </div>
        </section>

        <!-- About Section -->
        <section class="mb-8 fade-in">
            <div class="flex justify-between items-center mb-4">
                <h3 class="font-semibold text-gray-800">About Solana</h3>
                <button id="toggle-description" class="text-purple-500 text-sm font-medium">Show More</button>
            </div>
            <div id="description-content" class="description-collapse text-gray-600 text-sm leading-relaxed">
                Loading Solana description...
            </div>
        </section>

        <!-- Official Links -->
        <section class="fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Resources</h3>
            <div class="grid grid-cols-2 gap-3">
                <a href="https://solana.com" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-purple-500 font-medium">Official Website</div>
                </a>
                <a href="https://github.com/solana-labs/solana" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-purple-500 font-medium">Source Code</div>
                </a>
                <a href="blockchain_details.php" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-purple-500 font-medium">Block Explorer</div>
                </a>
                <a href="https://docs.solana.com" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-purple-500 font-medium">Documentation</div>
                </a>
                <a href="https://solana.com/ecosystem" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-purple-500 font-medium">Ecosystem</div>
                </a>
                <a href="https://solanabeach.io" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-purple-500 font-medium">Network Stats</div>
                </a>
            </div>
        </section>
    </div>

    <script>
        // DOM Elements
        const solPriceEl = document.getElementById('sol-price');
        const solArrowEl = document.getElementById('sol-arrow');
        const solChangeEl = document.getElementById('sol-change');
        const solEquivalentEl = document.getElementById('sol-equivalent');
        const solBalanceUsdEl = document.getElementById('sol-balance-usd');
        const marketCapEl = document.getElementById('market-cap');
        const volumeEl = document.getElementById('volume');
        const supplyEl = document.getElementById('supply');
        const marketCapRankEl = document.getElementById('market-cap-rank');
        const stakingRateEl = document.getElementById('staking-rate');
        const stakingApyEl = document.getElementById('staking-apy');
        const defiTvlEl = document.getElementById('defi-tvl');
        const activeProjectsEl = document.getElementById('active-projects');
        const descriptionContentEl = document.getElementById('description-content');
        const toggleDescriptionBtn = document.getElementById('toggle-description');
        const connectionStatusEl = document.getElementById('connection-status');
        const supplyProgressEl = document.getElementById('supply-progress');
        const circulatingSupplyEl = document.getElementById('circulating-supply');
        const totalSupplyEl = document.getElementById('total-supply');
        const currentTpsEl = document.getElementById('current-tps');
        const validatorCountEl = document.getElementById('validator-count');

        // User data from PHP - FIX: Get the correct USD value and calculate tokens
        const userSOL_USDValue = <?php echo json_encode($sol_usd_value); ?>;
        let userSOL_Tokens = 0; // Will be calculated based on current price
        let previousPrice = 0;
        let coinData = {};
        let descriptionExpanded = false;

        // Toggle description
        toggleDescriptionBtn.addEventListener('click', () => {
            descriptionExpanded = !descriptionExpanded;
            descriptionContentEl.classList.toggle('description-collapse', !descriptionExpanded);
            descriptionContentEl.classList.toggle('description-expanded', descriptionExpanded);
            toggleDescriptionBtn.textContent = descriptionExpanded ? 'Show Less' : 'Show More';
        });

        // Format numbers with commas
        function formatNumber(num, decimals = 0) {
            return new Intl.NumberFormat('en-US', {
                minimumFractionDigits: decimals,
                maximumFractionDigits: decimals
            }).format(num);
        }

        // Format currency
        function formatCurrency(amount) {
            return new Intl.NumberFormat('en-US', {
                style: 'currency',
                currency: 'USD',
                minimumFractionDigits: 2,
                maximumFractionDigits: 4
            }).format(amount);
        }

        // Update UI with price data
        function updatePriceUI(price, change) {
            // Update price
            solPriceEl.textContent = formatCurrency(price);
            
            // Update 24h change
            const changeFixed = change.toFixed(2);
            solChangeEl.textContent = `${changeFixed}% (24h)`;
            solChangeEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update arrow
            solArrowEl.textContent = change >= 0 ? '↑' : '↓';
            solArrowEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update equivalent
            const solEquivalent = 1 / price;
            solEquivalentEl.textContent = solEquivalent.toFixed(4) + ' SOL';
            
            // FIX: Calculate SOL tokens from USD value and update display
            userSOL_Tokens = price > 0 ? (userSOL_USDValue / price) : 0;
            
            // Update the balance display in the wallet card
            const solBalanceElement = document.querySelector('.wallet-card h2');
            if (solBalanceElement) {
                solBalanceElement.textContent = formatNumber(userSOL_Tokens, 6) + ' SOL';
            }
            
            // The USD value remains the same (from database)
            solBalanceUsdEl.textContent = formatNumber(userSOL_USDValue, 2);
            
            // Price change animation with bounce effect
            if (previousPrice && previousPrice !== price) {
                solPriceEl.classList.remove('text-green-500', 'text-red-500', 'price-bounce');
                void solPriceEl.offsetWidth; // Trigger reflow
                solPriceEl.classList.add(price > previousPrice ? 'text-green-500' : 'text-red-500');
                solPriceEl.classList.add('price-bounce');
                
                setTimeout(() => {
                    solPriceEl.classList.remove('text-green-500', 'text-red-500', 'price-bounce');
                    solPriceEl.classList.add('text-gray-800');
                }, 3000);
            }
            
            previousPrice = price;
        }

        
            // Update market data
        function updateMarketData(data) {
            marketCapEl.textContent = formatCurrency(data.market_data.market_cap.usd);
            volumeEl.textContent = formatCurrency(data.market_data.total_volume.usd);
            
            // Update supply data
            const circulatingSupply = data.market_data.circulating_supply;
            const totalSupply = data.market_data.total_supply || 588000000; // Default to 588M
            
            supplyEl.textContent = formatNumber(circulatingSupply, 0) + ' SOL';
            circulatingSupplyEl.textContent = formatNumber(circulatingSupply, 0);
            totalSupplyEl.textContent = formatNumber(totalSupply, 0) + 'M';
            
            // Update supply progress bar
            const supplyPercentage = (circulatingSupply / totalSupply) * 100;
            supplyProgressEl.style.width = supplyPercentage + '%';
            
            // Update market cap rank
            marketCapRankEl.textContent = '#' + (data.market_data.market_cap_rank || 5);
            
            // Update connection status
            connectionStatusEl.className = 'w-3 h-3 rounded-full bg-green-500';
        }

        // Fetch SOL data from CoinGecko API
        async function fetchSOLData() {
            try {
                const response = await axios.get('https://api.coingecko.com/api/v3/coins/solana', {
                    timeout: 10000
                });
                
                coinData = response.data;
                
                // Update price and market data
                const price = coinData.market_data.current_price.usd;
                const change = coinData.market_data.price_change_percentage_24h;
                
                updatePriceUI(price, change);
                updateMarketData(coinData);
                
                // Update description
                if (coinData.description && coinData.description.en) {
                    descriptionContentEl.innerHTML = coinData.description.en;
                } else {
                    descriptionContentEl.innerHTML = `
                        <p>Solana is a high-performance blockchain platform designed for decentralized applications and crypto-currencies. It aims to provide fast, secure, and scalable solutions for DeFi, NFTs, and Web3 applications.</p>
                        <p>Key features include Proof of History (PoH) consensus mechanism, theoretical throughput of 65,000+ transactions per second, and extremely low transaction costs averaging $0.00025.</p>
                        <p>The Solana ecosystem includes major DeFi protocols, NFT marketplaces, and gaming applications, making it one of the most active blockchain networks in the crypto space.</p>
                    `;
                }
                
                connectionStatusEl.className = 'w-3 h-3 rounded-full bg-green-500';
                
            } catch (error) {
                console.error('Error fetching SOL data:', error);
                connectionStatusEl.className = 'w-3 h-3 rounded-full bg-red-500';
                
                // Fallback data
                updatePriceUI(100.00, 0);
                descriptionContentEl.innerHTML = `
                    <p>Solana is a high-performance blockchain platform designed for decentralized applications and crypto-currencies. It aims to provide fast, secure, and scalable solutions for DeFi, NFTs, and Web3 applications.</p>
                    <p>Key features include Proof of History (PoH) consensus mechanism, theoretical throughput of 65,000+ transactions per second, and extremely low transaction costs averaging $0.00025.</p>
                    <p>The Solana ecosystem includes major DeFi protocols, NFT marketplaces, and gaming applications, making it one of the most active blockchain networks in the crypto space.</p>
                `;
            }
        }

        // Simulate TPS counter animation
        function animateTPS() {
            const tpsValues = [2100, 2300, 2500, 2800, 2200, 2600, 2400];
            let currentIndex = 0;
            
            setInterval(() => {
                currentTpsEl.textContent = '~' + formatNumber(tpsValues[currentIndex]);
                currentIndex = (currentIndex + 1) % tpsValues.length;
            }, 3000);
        }

        // Simulate validator count updates
        function animateValidatorCount() {
            const validatorCounts = [1850, 1900, 1920, 1880, 1950, 1870];
            let currentIndex = 0;
            
            setInterval(() => {
                validatorCountEl.textContent = '~' + formatNumber(validatorCounts[currentIndex]);
                currentIndex = (currentIndex + 1) % validatorCounts.length;
            }, 5000);
        }

        // Initialize the page
        document.addEventListener('DOMContentLoaded', function() {
            // Fetch initial data
            fetchSOLData();
            
            // Start animations
            animateTPS();
            animateValidatorCount();
            
            // Set up auto-refresh every 30 seconds
            setInterval(fetchSOLData, 30000);
        });

        // Handle visibility change to pause/resume updates
        document.addEventListener('visibilitychange', function() {
            if (document.hidden) {
                // Page is hidden, you might want to pause updates
                console.log('Page hidden - pausing updates');
            } else {
                // Page is visible, resume updates
                console.log('Page visible - resuming updates');
                fetchSOLData(); // Fetch fresh data when page becomes visible
            }
        });
    </script>
</body>
</html>