<?php
// Enable full error reporting (for development only, remove or disable in production)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in, if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);

if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id     = $userDetails['google_id'];
$name          = $userDetails['name'];
$email         = $userDetails['email'];
$created_at    = $userDetails['created_at'];
$picture       = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin           = $userDetails['pin'];
$status        = $userDetails['status'];
$country       = $userDetails['country'];
$phone         = $userDetails['phone'];
$balance       = $userDetails['balance'];
$btc           = $userDetails['btc'];
$eth           = $userDetails['eth'];
$usdt          = $userDetails['usdt'];
$xrp           = $userDetails['xrp'];
$message         = $userDetails['message'];
$update_time   = $userDetails['update_time'];
?>
<?php
// Enable error reporting for debugging (remove in production)

if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['email']);
    }
}

if (!function_exists('getUserDetails')) {
    function getUserDetails($conn) {
        $email = $_SESSION['email'] ?? null;
        if (!$email) return null;

        $stmt = $conn->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->bind_param('s', $email);
        $stmt->execute();
        $result = $stmt->get_result();
        $userDetails = $result->fetch_assoc();
        $stmt->close();

        return $userDetails;
    }
}

if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

$userEmail = $userDetails['email'];
$userName = $userDetails['name']; // Get the user's name for bank withdrawal filtering

// Helper function to get the first deposit date
function getFirstDepositDate($conn, $email) {
    $query = "SELECT MIN(created_at) as first_deposit FROM payments WHERE email = ? AND amount > 0";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('s', $email);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    
    if ($row && $row['first_deposit']) {
        return date('Y-m-d', strtotime($row['first_deposit']));
    }
    return date('Y-m-d', strtotime('-30 days')); // Default to 30 days ago if no deposits
}

// Helper functions
function getStatusColor($status) {
    switch (strtolower($status)) {
        case 'completed':
        case 'complete':
        case 'confirmed':
            return 'bg-green-100 text-green-800';
        case 'pending':
        case 'unconfirmed':
            return 'bg-yellow-100 text-yellow-800'; // Changed from red to yellow
        case 'confirming':
        case 'processing':

            return 'bg-blue-100 text-blue-800';
        case 'declined':
        case 'rejected':
        case 'failed':
            return 'bg-red-100 text-red-800'; // Added declined status as red
        default:
            return 'bg-gray-100 text-gray-800';
    }
}

function truncateHash($hash) {
    return substr($hash, 0, 6) . '...' . substr($hash, -4);
}

function errorMessage($message) {
    return '<div class="text-red-500">' . htmlspecialchars($message) . '</div>';
}

function getNetworkIcon($network) {
    $network = strtolower(trim($network));
    switch ($network) {
        case 'bitcoin':
        case 'btc':
            return 'https://bitvros.com/cryptos/bitcoin.png';
        case 'ethereum':
        case 'eth':
            return 'https://bitvros.com/cryptos/eth.png';
        case 'usdt':
        case 'erc20':
        case 'trc20':
            return 'https://bitvros.com/cryptos/usdt.png';
        case 'xrp':
            return 'https://bitvros.com/cryptos/xrp.png';
        case 'dogecoin':
        case 'doge':
            return 'https://bitvros.com/cryptos/doge.png';
        case 'cardano':
        case 'ada':
            return 'https://bitvros.com/cryptos/ada.png';
        case 'bnb':
        case 'binance':
            return 'https://bitvros.com/cryptos/bnb.png';
        case 'pepe':
            return 'https://bitvros.com/cryptos/pepe.png';
        case 'sol':
            return 'https://bitvros.com/cryptos/sol.png';
        case 'bank':
        case 'wire':
            return 'https://cdn-icons-png.flaticon.com/512/1019/1019607.png'; // Bank icon
        default:
            return 'https://cryptologos.cc/logos/measurable-data-token-mdt-logo.png';
    }
}

function getNetworkDisplayName($network) {
    $network = strtolower(trim($network));
    switch ($network) {
        case 'btc':
            return 'Bitcoin';
        case 'eth':
            return 'Ethereum';
        case 'usdt':
        case 'erc20':
        case 'trc20':
            return 'USDT';
        case 'xrp':
            return 'XRP';
        case 'doge':
        case 'dogecoin':
            return 'Dogecoin';
        case 'ada':
        case 'cardano':
            return 'Cardano';
        case 'bnb':
        case 'binance':
            return 'BNB';
        case 'pepe':
            return 'PEPE';
        case 'sol':
            return 'Solana';
        case 'bank':
        case 'wire':
            return 'Bank Transfer';
        default:
            return ucfirst($network);
    }
}

function getCoinGeckoId($network) {
    $network = strtolower(trim($network));
    switch ($network) {
        case 'btc':
        case 'bitcoin':
            return 'bitcoin';
        case 'eth':
        case 'ethereum':
            return 'ethereum';
        case 'usdt':
        case 'erc20':
        case 'trc20':
            return 'tether';
        case 'xrp':
            return 'ripple';
        case 'doge':
        case 'dogecoin':
            return 'dogecoin';
        case 'ada':
        case 'cardano':
            return 'cardano';
        case 'bnb':
        case 'binance':
            return 'binancecoin';
        case 'pepe':
            return 'pepe';
        case 'sol':
            return 'solana';
        default:
            return null;
    }
}

// Helper function to format amounts with commas and decimals
function formatAmount($amount, $isCrypto = false) {
    if ($amount == 0) return '0';
    
    if ($isCrypto) {
        // Format crypto amounts with up to 8 decimals, remove trailing zeros
        $formatted = rtrim(number_format($amount, 8, '.', ','), '0');
        return rtrim($formatted, '.');
    }
    
    // Format USD amounts with 2 decimals and commas
    return number_format($amount, 2, '.', ',');
}

// AJAX endpoint for getting crypto prices
if (isset($_GET['get_prices'])) {
    header('Content-Type: application/json');
    
    $networks = ['bitcoin', 'ethereum', 'tether', 'ripple', 'dogecoin', 'cardano', 'binancecoin', 'pepe', 'solana'];
    $ids = implode(',', $networks);
    
    $url = "https://api.coingecko.com/api/v3/simple/price?ids={$ids}&vs_currencies=usd";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response !== false) {
        echo $response;
    } else {
        echo json_encode(['error' => 'Failed to fetch prices']);
    }
    exit();
}

// --- AJAX Endpoint ---
if (isset($_GET['get_transactions'])) {
    header('Content-Type: text/html');

    // Validate and assign GET parameters
    $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ["options" => ["default" => 1, "min_range" => 1]]);
    $perPage = filter_input(INPUT_GET, 'perPage', FILTER_VALIDATE_INT, ["options" => ["default" => 6, "min_range" => 1, "max_range" => 100]]);
    $offset = ($page - 1) * $perPage;

    $filterType = (isset($_GET['type']) && in_array($_GET['type'], ['received', 'sent', 'bank_withdrawal'])) ? $_GET['type'] : 'sent';
    $start = (isset($_GET['start']) && !empty($_GET['start'])) ? $_GET['start'] : null;
    $end   = (isset($_GET['end']) && !empty($_GET['end'])) ? $_GET['end'] : null;

    // Base query and parameters - FIXED: Removed 'id' from bank_withdrawal query
    if ($filterType === 'sent') {
        $baseQuery = "SELECT id, email, total_amount AS amount, created_at, status, network, transhash, 'crypto' as type
                      FROM crypto_withdrawal WHERE email = ?";
    } elseif ($filterType === 'bank_withdrawal') {
        // FIXED: Use transid instead of id, and filter by account_name matching user's name
        $baseQuery = "SELECT transid, user_id, country, bank_name, account_number, account_name, amount, currency, status, transactionDate as created_at, 'bank' as type
                      FROM bank_withdrawal WHERE account_name = ?";
    } else {
        $baseQuery = "SELECT id, email, amount, created_at, status, network, transhash, 'crypto' as type
                      FROM payments WHERE email = ?";
    }

    $conditions = '';
    // FIXED: Use account_name for bank withdrawals instead of email
    $params = ($filterType === 'bank_withdrawal') ? [$userName] : [$userEmail];
    $paramTypes = 's';

    if ($start) {
        $conditions .= " AND " . ($filterType === 'bank_withdrawal' ? 'transactionDate' : 'created_at') . " >= ?";
        $params[] = $start . ' 00:00:00';
        $paramTypes .= 's';
    }
    if ($end) {
        $conditions .= " AND " . ($filterType === 'bank_withdrawal' ? 'transactionDate' : 'created_at') . " <= ?";
        $params[] = $end . ' 23:59:59';
        $paramTypes .= 's';
    }

    if (!empty($conditions)) {
        $baseQuery .= $conditions;
    }

    $orderQuery = $baseQuery . " ORDER BY " . ($filterType === 'bank_withdrawal' ? 'transactionDate' : 'created_at') . " DESC";

    // Count query
    $countQuery = "SELECT COUNT(*) FROM ($baseQuery) AS sub";

    try {
        $stmt = $conn->prepare($countQuery);
        $stmt->bind_param($paramTypes, ...$params);
        $stmt->execute();
        $stmt->bind_result($total);
        $stmt->fetch();
        $stmt->close();
    } catch (mysqli_sql_exception $e) {
        echo errorMessage('Count query failed: ' . $e->getMessage());
        exit();
    }

    // Data query
    $dataQuery = $orderQuery . " LIMIT ? OFFSET ?";
    $paramsForDataQuery = $params;
    $paramTypesForDataQuery = $paramTypes . 'ii';
    $paramsForDataQuery[] = $perPage;
    $paramsForDataQuery[] = $offset;

    try {
        $stmt = $conn->prepare($dataQuery);
        $stmt->bind_param($paramTypesForDataQuery, ...$paramsForDataQuery);
        $stmt->execute();
        $result = $stmt->get_result();
        $transactions = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    } catch (mysqli_sql_exception $e) {
        echo errorMessage('Query failed: ' . $e->getMessage());
        exit();
    }

    // Output transactions
    ?>
    <div id="transactionsListContent" class="grid gap-6">
    <?php if (!empty($transactions)): ?>
        <?php foreach ($transactions as $tx): 
            $isSent = ($filterType === 'sent');
            $isBankWithdrawal = ($filterType === 'bank_withdrawal');
            $status = (floatval($tx['amount']) == 0.0) ? 'Confirming' : ($tx['status'] ?? 'Pending');
            $statusColor = getStatusColor($status);
            $amount = floatval($tx['amount']);
            $formattedAmount = formatAmount($amount, false); // USD formatting
            $date = new DateTime($tx['created_at']);
            $displayDate = htmlspecialchars($date->format('M d, Y H:i'));
            
            if ($isBankWithdrawal) {
                $transactionType = 'Bank Withdrawal';
                $network = 'bank';
                $txHash = $tx['transid'] ?? 'N/A'; // FIXED: Use transid instead of transid
                $currency = $tx['currency'] ?? 'USD';
                $bankName = $tx['bank_name'] ?? 'N/A';
                $accountNumber = $tx['account_number'] ?? 'N/A';
                $accountName = $tx['account_name'] ?? 'N/A';
                $country = $tx['country'] ?? 'N/A';
                $transactionId = $tx['transid'] ?? ''; // For linking to bankd.php
            } else {
                $transactionType = $isSent ? 'Sent' : 'Received';
                $network = htmlspecialchars($tx['network'] ?? 'Unknown');
                $txHash = $tx['transhash'] ?? 'N/A';
                $transactionId = $tx['id'] ?? '';
            }
            
            $networkDisplayName = getNetworkDisplayName($network);
            $coinGeckoId = getCoinGeckoId($network);
            $networkIcon = getNetworkIcon($network);
        ?>
        <!-- Enhanced neumorphic card -->
        <a href="<?= $isBankWithdrawal ? 'bankd.php?transid=' . urlencode($transactionId) : 'detailed.php?id=' . urlencode($transactionId) . '&type=' . urlencode($filterType) ?>" class="block">
            <div class="group flex flex-col md:flex-row items-center justify-between p-6 bg-gradient-to-br from-gray-50 to-white rounded-2xl hover:from-gray-100 hover:to-gray-50 transition-all duration-300 
                        shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.8)] 
                        hover:shadow-[12px_12px_24px_rgba(163,177,198,0.8),-12px_-12px_24px_rgba(255,255,255,0.9)]
                        border border-gray-100/50 backdrop-blur-sm">
                <div class="flex items-center space-x-5">
                    <!-- Enhanced neumorphic icon container -->
                    <div class="<?= $isSent || $isBankWithdrawal ? 'bg-gradient-to-br from-red-50 to-red-100' : 'bg-gradient-to-br from-green-50 to-green-100' ?> 
                                p-4 rounded-2xl 
                                shadow-[inset_6px_6px_12px_rgba(163,177,198,0.3),inset_-6px_-6px_12px_rgba(255,255,255,0.9)]
                                border border-white/50">
                        <?php if ($isBankWithdrawal): ?>
                            <svg xmlns="http://www.w3.org/2000/svg" class="w-7 h-7 text-red-500" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M12 2L2 7v10c0 5.55 3.84 9.74 9 11 5.16-1.26 9-5.45 9-11V7l-10-5z"/>
                                <path d="M9 12l2 2 4-4"/>
                            </svg>
                        <?php else: ?>
                            <svg xmlns="http://www.w3.org/2000/svg" class="w-7 h-7 <?= $isSent ? 'text-red-500' : 'text-green-500' ?>" fill="currentColor" viewBox="0 0 24 24">
                                <?php if($isSent): ?>
                                    <path d="M13 7.41V20a1 1 0 0 1-2 0V7.41l-5.3 5.3a1 1 0 0 1-1.4-1.42l7-7a1 1 0 0 1 1.4 0l7 7a1 1 0 0 1-1.4 1.42L13 7.4z"/>
                                <?php else: ?>
                                    <path d="M11 16.59V4a1 1 0 0 1 2 0v12.59l5.3-5.3a1 1 0 0 1 1.4 1.42l-7 7a1 1 0 0 1-1.4 0l-7-7a1 1 0 0 1 1.4-1.42l5.3 5.3z"/>
                                <?php endif; ?>
                            </svg>
                        <?php endif; ?>
                    </div>
                    
                    <div>
                        <p class="font-semibold text-gray-800 text-lg">
                            <?= $transactionType ?>
                        </p>
                        <div class="flex items-center text-sm text-gray-600 mt-1">
                            <img src="<?= $networkIcon ?>" alt="<?= $networkDisplayName ?>" class="w-5 h-5 mr-2 rounded-full">
                            <?= $networkDisplayName ?>
                        </div>
                        <?php if ($isBankWithdrawal): ?>
                            <p class="text-sm text-gray-500 mt-1"><?= htmlspecialchars($bankName) ?> - <?= htmlspecialchars($country) ?></p>
                            <p class="text-sm text-gray-500 mt-1">Account: <?= htmlspecialchars($accountName) ?></p>
                        <?php endif; ?>
                        <p class="text-sm text-gray-500 mt-1"><?= $displayDate ?></p>
                    </div>
                </div>
                
                <div class="text-right mt-4 md:mt-0">
                    <!-- Main display: Amount with currency -->
                    <p class="text-xl font-bold <?= $isSent || $isBankWithdrawal ? 'text-red-600' : 'text-green-600' ?>">
                        <?php if(floatval($tx['amount']) == 0.0): ?>
                            <i class="fas fa-spinner fa-spin text-gray-400"></i>
                        <?php else: ?>
                            <?php if ($isBankWithdrawal): ?>
                                <?= '-' . htmlspecialchars($currency) . ' ' . $formattedAmount ?>
                            <?php else: ?>
                                <?= (($isSent) ? '-' : '+') . '$' . $formattedAmount ?>
                            <?php endif; ?>
                        <?php endif; ?>
                    </p>
                    
                    <!-- Crypto equivalent (calculated from USD) for non-bank transactions -->
                    <?php if (!$isBankWithdrawal && $coinGeckoId): ?>
                    <p class="text-sm <?= $isSent ? 'text-red-500' : 'text-green-500' ?> crypto-amount mt-1" 
                       data-usd-amount="<?= $amount ?>" 
                       data-network="<?= $coinGeckoId ?>" 
                       data-is-sent="<?= $isSent ? 'true' : 'false' ?>"
                       data-network-symbol="<?= strtoupper($network) ?>">
                        <?php if (floatval($tx['amount']) != 0.0): ?>
                            <!-- Crypto equivalent will be populated by JavaScript -->
                            <span class="text-gray-400">Converting...</span>
                        <?php endif; ?>
                    </p>
                    <?php endif; ?>
                    
                    <!-- Enhanced neumorphic status badge -->
                    <div class="inline-flex items-center mt-2 px-4 py-2 rounded-xl text-sm font-medium
                                shadow-[4px_4px_8px_rgba(163,177,198,0.4),-4px_-4px_8px_rgba(255,255,255,0.8)]
                                border border-white/50 <?= $statusColor ?>">
                        <span><?= htmlspecialchars(ucfirst($status)) ?></span>
                    </div>
                    
                    <?php if ($isBankWithdrawal): ?>
                        <p class="text-xs text-gray-500 mt-2 font-mono bg-gray-50 rounded-lg px-3 py-1 
                                 shadow-[inset_2px_2px_4px_rgba(163,177,198,0.3),inset_-2px_-2px_4px_rgba(255,255,255,0.9)]">
                            <span>Account: <?= htmlspecialchars(substr($accountNumber, 0, 4)) ?>****<?= htmlspecialchars(substr($accountNumber, -4)) ?></span>
                        </p>
                        <p class="text-xs text-gray-500 mt-1 font-mono bg-gray-50 rounded-lg px-3 py-1 
                                 shadow-[inset_2px_2px_4px_rgba(163,177,198,0.3),inset_-2px_-2px_4px_rgba(255,255,255,0.9)]">
                            <span title="<?= htmlspecialchars($txHash) ?>">Trans ID: <?= htmlspecialchars($txHash) ?></span>
                        </p>
                    <?php else: ?>
                        <p class="text-xs text-gray-500 mt-2 font-mono bg-gray-50 rounded-lg px-3 py-1 
                                 shadow-[inset_2px_2px_4px_rgba(163,177,198,0.3),inset_-2px_-2px_4px_rgba(255,255,255,0.9)]">
                            <span title="<?= htmlspecialchars($txHash) ?>">Transhash:
                                <?= htmlspecialchars(truncateHash($txHash)) ?>
                            </span>
                        </p>
                    <?php endif; ?>
                </div>
            </div>
        </a>
        <?php endforeach; ?>
    <?php else: ?>
        <div class="text-center py-12 text-gray-500">
            <div class="bg-gradient-to-br from-gray-50 to-white rounded-2xl p-8 
                        shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.8)]
                        border border-gray-100/50">
                <i class="fas fa-coins text-4xl mb-4 text-gray-400"></i>
                <p class="text-lg">No transactions found</p>
            </div>
        </div>
    <?php endif; ?>
    </div>

    <?php
    // Enhanced neumorphic pagination with <1234> buttons
    $totalPages = ceil($total / $perPage);
    if ($totalPages > 1):
        // Calculate visible page range
        $visiblePages = 4; // Show up to 4 page numbers
        $startPage = max(1, min($page - floor($visiblePages/2), $totalPages - $visiblePages + 1));
        $endPage = min($startPage + $visiblePages - 1, $totalPages);
    ?>
    <div id="paginationContent" class="flex justify-center gap-3 mt-8">
        <?php if ($page > 1): ?>
        <button onclick="loadTransactions('<?= $filterType ?>', <?= $page-1 ?>)" 
                class="px-4 py-2 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                       text-gray-700 transition-all duration-200
                       shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                       hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                       border border-gray-100/50">
            <i class="fas fa-chevron-left"></i>
        </button>
        <?php endif; ?>

        <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
            <?php if ($i == $page): ?>
                <button class="px-4 py-2 bg-gradient-to-br from-[#4669db] to-[#3b56c4] text-white rounded-xl 
                               shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                               border border-[#4669db]/20">
                    <?= $i ?>
                </button>
            <?php else: ?>
                <button onclick="loadTransactions('<?= $filterType ?>', <?= $i ?>)" 
                        class="px-4 py-2 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                               text-gray-700 transition-all duration-200
                               shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                               hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                               border border-gray-100/50">
                    <?= $i ?>
                </button>
            <?php endif; ?>
        <?php endfor; ?>

        <?php if ($page < $totalPages): ?>
        <button onclick="loadTransactions('<?= $filterType ?>', <?= $page+1 ?>)" 
                class="px-4 py-2 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                       text-gray-700 transition-all duration-200
                       shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                       hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                       border border-gray-100/50">
            <i class="fas fa-chevron-right"></i>
        </button>
        <?php endif; ?>
    </div>
    <?php endif;
    exit();
}

// Get the first deposit date for default date range
$firstDepositDate = getFirstDepositDate($conn, $userEmail);
$currentDate = date('Y-m-d');
?>

<?php include './includes/header.php'; ?>

<body class="bg-gradient-to-br from-gray-100 to-gray-200 min-h-screen">
  <?php include './includes/nav.php'; ?>
 
   <div style="padding: 4px; margin-top: 65px;">

       <!-- Main Content -->
       <div class="mb-24">
           <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8">
               <div class="flex space-x-3 mb-4 md:mb-0">
                   
                   <!-- Enhanced neumorphic transaction type filters -->
               <div class="flex flex-wrap gap-3">
                   <button onclick="loadTransactions('received', 1)" 
                           class="px-6 py-3 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                                  text-gray-700 transition-all duration-200 filter-btn
                                  shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                  hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                                  border border-gray-100/50 font-medium">
                       <i class="fas fa-arrow-down text-green-500 mr-2"></i>
                       Received
                   </button>
                   <button onclick="loadTransactions('sent', 1)" 
                           class="px-6 py-3 bg-gradient-to-br from-[#4669db] to-[#3b56c4] text-white rounded-xl 
                                  shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                  border border-[#4669db]/20 font-medium filter-btn active">
                       <i class="fas fa-arrow-up text-white mr-2"></i>
                       Sent
                   </button>
                   <button onclick="loadTransactions('bank_withdrawal', 1)" 
                           class="px-6 py-3 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                                  text-gray-700 transition-all duration-200 filter-btn
                                  shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                  hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                                  border border-gray-100/50 font-medium">
                       <i class="fas fa-university text-blue-500 mr-2"></i>
                       Bank Withdrawals
                   </button>
               </div>
           </div>

           <!-- Enhanced neumorphic date range filters -->
           <div class="bg-gradient-to-br from-white to-gray-50 rounded-2xl p-6 mb-8 
                       shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.8)]
                       border border-gray-100/50">
               <div class="flex flex-col sm:flex-row gap-4 items-center">
                   <div class="flex items-center space-x-4">
                       <label class="text-sm font-medium text-gray-700">From:</label>
                       <input type="date" id="startDate" value="<?= htmlspecialchars($firstDepositDate) ?>" 
                              class="px-4 py-2 rounded-lg border border-gray-200 focus:border-blue-400 focus:outline-none
                                     shadow-[inset_4px_4px_8px_rgba(163,177,198,0.3),inset_-4px_-4px_8px_rgba(255,255,255,0.9)]
                                     bg-white/90">
                   </div>
                   <div class="flex items-center space-x-4">
                       <label class="text-sm font-medium text-gray-700">To:</label>
                       <input type="date" id="endDate" value="<?= htmlspecialchars($currentDate) ?>" 
                              class="px-4 py-2 rounded-lg border border-gray-200 focus:border-blue-400 focus:outline-none
                                     shadow-[inset_4px_4px_8px_rgba(163,177,198,0.3),inset_-4px_-4px_8px_rgba(255,255,255,0.9)]
                                     bg-white/90">
                   </div>
                   <button onclick="applyDateFilter()" 
                           class="px-6 py-2 bg-gradient-to-br from-[#4669db] to-[#3b56c4] text-white rounded-xl
                                  shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                  hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                                  border border-[#4669db]/20 font-medium transition-all duration-200">
                       <i class="fas fa-filter mr-2"></i>
                       Apply Filter
                   </button>
               </div>
           </div>

           <!-- Loading spinner -->
           <div id="loadingSpinner" class="hidden text-center py-12">
               <div class="bg-gradient-to-br from-white to-gray-50 rounded-2xl p-8 
                           shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.8)]
                           border border-gray-100/50">
                   <i class="fas fa-spinner fa-spin text-4xl text-blue-500 mb-4"></i>
                   <p class="text-gray-600">Loading transactions...</p>
               </div>
           </div>

           <!-- Transactions Container -->
           <div id="transactionsContainer">
               <!-- Initial transactions will be loaded here -->
           </div>
       </div>
   </div>

   <?php include './includes/footer.php'; ?>

   <script>
   let currentFilter = 'sent';
   let currentPage = 1;
   let cryptoPrices = {};

   // Fetch crypto prices on page load
   async function fetchCryptoPrices() {
       try {
           const response = await fetch('?get_prices=1');
           const data = await response.json();
           if (!data.error) {
               cryptoPrices = data;
               updateCryptoAmounts();
           }
       } catch (error) {
           console.error('Error fetching crypto prices:', error);
       }
   }

   // Update crypto amounts in the DOM
   function updateCryptoAmounts() {
       document.querySelectorAll('.crypto-amount').forEach(element => {
           const usdAmount = parseFloat(element.dataset.usdAmount);
           const network = element.dataset.network;
           const isSent = element.dataset.isSent === 'true';
           const networkSymbol = element.dataset.networkSymbol;
           
           if (cryptoPrices[network] && cryptoPrices[network].usd && usdAmount > 0) {
               const cryptoAmount = usdAmount / cryptoPrices[network].usd;
               const formattedAmount = cryptoAmount.toFixed(8).replace(/\.?0+$/, '');
               const prefix = isSent ? '-' : '+';
               element.innerHTML = `${prefix}${formattedAmount} ${networkSymbol}`;
           } else if (usdAmount > 0) {
               element.innerHTML = '<span class="text-gray-400">Price unavailable</span>';
           }
       });
   }

   // Load transactions with filters
   async function loadTransactions(type, page = 1) {
       currentFilter = type;
       currentPage = page;
       
       const startDate = document.getElementById('startDate').value;
       const endDate = document.getElementById('endDate').value;
       
       // Update button states
       document.querySelectorAll('.filter-btn').forEach(btn => {
           btn.classList.remove('active');
           btn.classList.remove('bg-gradient-to-br', 'from-[#4669db]', 'to-[#3b56c4]', 'text-white');
           btn.classList.add('bg-gradient-to-br', 'from-white', 'to-gray-50', 'text-gray-700');
       });
       
       event.target.classList.add('active');
       event.target.classList.remove('bg-gradient-to-br', 'from-white', 'to-gray-50', 'text-gray-700');
       event.target.classList.add('bg-gradient-to-br', 'from-[#4669db]', 'to-[#3b56c4]', 'text-white');
       
       // Show loading spinner
       document.getElementById('loadingSpinner').classList.remove('hidden');
       document.getElementById('transactionsContainer').style.opacity = '0.5';
       
       try {
           const url = `?get_transactions=1&type=${type}&page=${page}&start=${startDate}&end=${endDate}`;
           const response = await fetch(url);
           const html = await response.text();
           
           document.getElementById('transactionsContainer').innerHTML = html;
           
           // Update crypto amounts after loading new transactions
           setTimeout(() => {
               updateCryptoAmounts();
           }, 100);
           
       } catch (error) {
           console.error('Error loading transactions:', error);
           document.getElementById('transactionsContainer').innerHTML = 
               '<div class="text-center py-12 text-red-500">Error loading transactions. Please try again.</div>';
       } finally {
           document.getElementById('loadingSpinner').classList.add('hidden');
           document.getElementById('transactionsContainer').style.opacity = '1';
       }
   }

   // Apply date filter
   function applyDateFilter() {
       loadTransactions(currentFilter, 1);
   }

   // Initialize page
   document.addEventListener('DOMContentLoaded', function() {
       fetchCryptoPrices();
       loadTransactions('sent', 1);
       
       // Refresh crypto prices every 60 seconds
       setInterval(fetchCryptoPrices, 60000);
   });
   </script>

</body>
</html>