<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Redirect with an error message.
 *
 * @param string $message The error message to set in the session.
 */
function redirectWithError($message) {
    $_SESSION['transaction_error'] = true;
    $_SESSION['error_message'] = $message;
    header("Location: dashboard.php");
    exit();
}

/**
 * Redirect with a success message.
 *
 * @param string $message The success message to set in the session.
 */
function redirectWithSuccess($message) {
    $_SESSION['transaction_success'] = true;
    $_SESSION['success_message'] = $message;
    header("Location: dashboard.php");
    exit();
}

/**
 * Enhanced logging function for debugging
 */
function debugLog($message, $context = []) {
    $timestamp = date('Y-m-d H:i:s');
    $contextStr = !empty($context) ? ' | Context: ' . json_encode($context) : '';
    error_log("[$timestamp] EMAIL_DEBUG: $message$contextStr");
}

// Ensure the request method is POST.
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: dashboard.php");
    exit();
}

// Validate required fields.
$required = ['walletAddress', 'network', 'email', 'name'];
foreach ($required as $field) {
    if (empty($_POST[$field])) {
        redirectWithError("Missing required field: $field");
    }
}

// Sanitize input values.
$wallet_address = htmlspecialchars($_POST['walletAddress'], ENT_QUOTES, 'UTF-8');
$network        = htmlspecialchars($_POST['network'], ENT_QUOTES, 'UTF-8');
$amount         = isset($_POST['amount']) ? $_POST['amount'] : 'confirming';
$email          = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
$name           = htmlspecialchars($_POST['name'], ENT_QUOTES, 'UTF-8');
$ip_address     = $_SERVER['REMOTE_ADDR'];

// Validate email format.
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    redirectWithError("Invalid email format");
}

// Process $amount: if it's not 'confirming', cast to float.
if ($amount !== 'confirming') {
    $amount = (float) $amount;
}

// Debug: Log file upload details for 'prove'.
if (isset($_FILES['prove'])) {
    error_log("File details: " . print_r($_FILES['prove'], true));
    error_log("File upload error code: " . $_FILES['prove']['error']);
}

/**
 * Handles file upload.
 *
 * @param string $fileInputName Name of the file input.
 * @param string $uploadDir Absolute path of the upload directory.
 * @param array  $allowedExtensions Allowed file extensions.
 * @param int    $maxSize Maximum allowed file size in bytes.
 * @return string Returns the new file name (relative path) if successful, or an empty string.
 */
function handleFileUpload($fileInputName, $uploadDir, $allowedExtensions, $maxSize) {
    if (isset($_FILES[$fileInputName]) && $_FILES[$fileInputName]['error'] === UPLOAD_ERR_OK) {
        // Check file size.
        if ($_FILES[$fileInputName]['size'] > $maxSize) {
            redirectWithError("File is too large. Maximum size is " . ($maxSize / (1024 * 1024)) . "MB.");
        }
        $fileTmpPath = $_FILES[$fileInputName]['tmp_name'];
        $fileName    = $_FILES[$fileInputName]['name'];
        
        // Get file extension and validate.
        $fileNameComponents = explode(".", $fileName);
        $fileExtension = strtolower(end($fileNameComponents));
        if (!in_array($fileExtension, $allowedExtensions)) {
            redirectWithError("File type not allowed. Allowed types: " . implode(', ', $allowedExtensions));
        }
        
        // Generate a secure, unique file name.
        $newFileName = bin2hex(random_bytes(16)) . '.' . $fileExtension;
        
        // Create the upload directory if it doesn't exist.
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                error_log("Failed to create upload directory: " . $uploadDir);
                redirectWithError("Failed to create upload directory.");
            }
            chmod($uploadDir, 0755);
        }
        
        // Verify that the directory is writable.
        if (!is_writable($uploadDir)) {
            error_log("Upload directory is not writable: " . $uploadDir);
            redirectWithError("Upload directory is not writable.");
        }
        
        $destPath = $uploadDir . $newFileName;
        if (move_uploaded_file($fileTmpPath, $destPath)) {
            chmod($destPath, 0644); // Set secure file permissions.
            error_log("File successfully uploaded to: " . $destPath);
            return $newFileName; // Return the relative file name for storage.
        } else {
            error_log("Error moving uploaded file. Tmp Path: $fileTmpPath, Destination: $destPath");
            error_log("Current directory permissions: " . substr(sprintf('%o', fileperms($uploadDir)), -4));
            error_log("PHP user: " . exec('whoami'));
            redirectWithError("There was an error moving the uploaded file.");
        }
    } else if (isset($_FILES[$fileInputName]) && $_FILES[$fileInputName]['error'] !== UPLOAD_ERR_NO_FILE) {
        // Convert PHP file upload error code to human-readable message.
        $uploadErrors = [
            UPLOAD_ERR_INI_SIZE   => "The uploaded file exceeds the upload_max_filesize directive in php.ini",
            UPLOAD_ERR_FORM_SIZE  => "The uploaded file exceeds the MAX_FILE_SIZE directive in the HTML form",
            UPLOAD_ERR_PARTIAL    => "The uploaded file was only partially uploaded",
            UPLOAD_ERR_NO_TMP_DIR => "Missing a temporary folder",
            UPLOAD_ERR_CANT_WRITE => "Failed to write file to disk",
            UPLOAD_ERR_EXTENSION  => "A PHP extension stopped the file upload"
        ];
        $errorCode = $_FILES[$fileInputName]['error'];
        $errorMessage = isset($uploadErrors[$errorCode]) ? $uploadErrors[$errorCode] : "Unknown upload error";
        error_log("File upload error: " . $errorMessage);
        redirectWithError("File upload failed: " . $errorMessage);
    }
    return "";
}

// Process file upload for 'prove' - Updated to support all major image formats
$uploadDir = __DIR__ . '/uploads/';
$allowedExtensions = [
    // Common image formats
    'jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg', 'tiff', 'tif',
    // Icon formats
    'ico', 'icon',
    // Modern image formats
    'avif', 'heic', 'heif',
    // Document formats (keeping original document support)
    'pdf', 'doc', 'docx', 'txt'
];
$maxFileSize = 10 * 1024 * 1024; // 10MB limit.
$provePath = handleFileUpload('prove', $uploadDir, $allowedExtensions, $maxFileSize);

// --- Database Insertion Section ---

// It is best practice to store credentials in a separate configuration file or environment variables.
$dbHost = "localhost";
$dbUser = "lrbpxbxy_userv";
$dbPass = "WHEagP2O9sOzG9gk";
$dbName = "lrbpxbxy_dbv";

// Create a new MySQLi connection.
$conn = new mysqli($dbHost, $dbUser, $dbPass, $dbName);
if ($conn->connect_error) {
    redirectWithError("Database connection failed. Please try again later.");
}

// Generate a unique transaction hash.
$transhash = hash('sha256', $email . microtime(true) . random_int(1000, 9999));
$status = "pending";

// Prepare SQL statement for insertion.
$sql = "INSERT INTO payments (
    wallet_address, 
    network, 
    amount, 
    email, 
    name, 
    ip_address,
    status,
    prove,
    created_at, 
    transhash
) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)";
$stmt = $conn->prepare($sql);
if (!$stmt) {
    redirectWithError("Prepare statement failed: " . $conn->error);
}

// Bind parameters based on whether $amount is 'confirming' or a numeric value.
if ($amount === 'confirming') {
    $stmt->bind_param("sssssssss", 
        $wallet_address,
        $network,
        $amount,
        $email,
        $name,
        $ip_address,
        $status,
        $provePath,
        $transhash
    );
} else {
    $stmt->bind_param("ssdssssss", 
        $wallet_address,
        $network,
        $amount,
        $email,
        $name,
        $ip_address,
        $status,
        $provePath,
        $transhash
    );
}

// Execute the statement and check for errors.
if (!$stmt->execute()) {
    error_log("Database insertion error: " . $stmt->error);
    redirectWithError("Database insertion failed: " . $stmt->error);
}
$stmt->close();

// --- Email Sending Section using PHPMailer ---

debugLog("Starting email process", ['user_email' => $email, 'transhash' => $transhash]);

// Check if PHPMailer exists
$autoloadPath = '../vendor/autoload.php';
if (!file_exists($autoloadPath)) {
    debugLog("PHPMailer autoload file not found", ['path' => $autoloadPath]);
    $conn->close();
    redirectWithError("Email system not configured. Transaction saved with ID: $transhash");
}

require $autoloadPath;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

debugLog("PHPMailer loaded successfully");

$mail = new PHPMailer(true);

try {
    // Enable verbose debug output
    $mail->SMTPDebug = SMTP::DEBUG_SERVER;
    $mail->Debugoutput = function($str, $level) {
        debugLog("PHPMailer Debug", ['level' => $level, 'message' => trim($str)]);
    };
    
    debugLog("Configuring SMTP settings");
    
    // Server settings
    $mail->isSMTP();
    $mail->Host       = 'bitvros.com';
    $mail->SMTPAuth   = true;
    $mail->Username   = 'noreply@bitvros.com';
    $mail->Password   = 'Bitv@9034';
    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
    $mail->Port       = 587;
    
    // Test alternative settings if the above fails
    $mail->SMTPOptions = array(
        'ssl' => array(
            'verify_peer' => false,
            'verify_peer_name' => false,
            'allow_self_signed' => true
        )
    );
    
    debugLog("SMTP settings configured", [
        'host' => $mail->Host,
        'port' => $mail->Port,
        'username' => $mail->Username,
        'encryption' => $mail->SMTPSecure
    ]);
    
    // Recipients
    $mail->setFrom('noreply@bitvros.com', 'bitvros.com Wallet');
    $mail->addAddress($email, $name);
    
    // Add BCC instead of CC for blind copy
    $mail->addBCC('support@bitvros.com', 'Support Team');
    
    debugLog("Recipients configured", [
        'to' => $email,
        'bcc' => 'support@bitvros.com'
    ]);
    
    // Attach the uploaded file if available.
    $attachmentPath = __DIR__ . '/uploads/' . $provePath;
    if (!empty($provePath) && file_exists($attachmentPath)) {
        $mail->addAttachment($attachmentPath);
        debugLog("Attachment added", ['file' => $attachmentPath]);
    } else {
        debugLog("No attachment or file not found", ['provePath' => $provePath, 'attachmentPath' => $attachmentPath]);
    }
    
    $mail->isHTML(true);
    $amountDisplay = $amount === 'confirming' ? 'confirming' : '$' . number_format($amount, 2) . ' USD';
    $mail->Subject = 'Transaction Confirmation - bitvros.com Wallet';
    
    debugLog("Email content prepared", [
        'subject' => $mail->Subject,
        'amount_display' => $amountDisplay
    ]);
    
    // Enhanced email body with professional table layout
    $mail->Body = "<html><body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f4f4f4;'>
        <div style='max-width: 650px; margin: 20px auto; background-color: #ffffff; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);'>
            <!-- Header Section -->
            <div style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px; text-align: center; border-radius: 8px 8px 0 0;'>
                <h1 style='color: #ffffff; margin: 0; font-size: 28px; font-weight: 300; letter-spacing: 1px;'>
                    bitvros.com
                </h1>
                <p style='color: #ffffff; margin: 10px 0 0 0; font-size: 16px; opacity: 0.9;'>
                    Transaction Confirmation
                </p>
            </div>
            
            <!-- Content Section -->
            <div style='padding: 30px;'>
                <p style='font-size: 16px; margin-bottom: 20px; color: #555;'>
                    Dear <strong>" . htmlspecialchars($name, ENT_QUOTES, 'UTF-8') . "</strong>,
                </p>
                <p style='font-size: 16px; margin-bottom: 30px; color: #555;'>
                    Your transaction has been successfully processed. Please find the details below:
                </p>
                
                <!-- Transaction Details Table -->
                <table style='width: 100%; border-collapse: collapse; margin: 20px 0; background-color: #ffffff; box-shadow: 0 2px 4px rgba(0,0,0,0.1);'>
                    <thead>
                        <tr style='background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);'>
                            <th colspan='2' style='color: #ffffff; padding: 15px; text-align: center; font-size: 18px; font-weight: 600; letter-spacing: 0.5px;'>
                                TRANSACTION DETAILS
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr style='border-bottom: 1px solid #e0e0e0;'>
                            <td style='padding: 15px; font-weight: 600; color: #2c3e50; background-color: #f8f9fa; width: 40%;'>
                                Transaction ID
                            </td>
                            <td style='padding: 15px; color: #27ae60; font-family: monospace; font-size: 14px; word-break: break-all;'>
                                $transhash
                            </td>
                        </tr>
                        <tr style='border-bottom: 1px solid #e0e0e0;'>
                            <td style='padding: 15px; font-weight: 600; color: #2c3e50; background-color: #f8f9fa;'>
                                Status
                            </td>
                            <td style='padding: 15px;'>
                                <span style='background-color: #f39c12; color: #ffffff; padding: 5px 12px; border-radius: 20px; font-size: 12px; font-weight: bold; text-transform: uppercase; letter-spacing: 0.5px;'>
                                    $status
                                </span>
                            </td>
                        </tr>
                        <tr style='border-bottom: 1px solid #e0e0e0;'>
                            <td style='padding: 15px; font-weight: 600; color: #2c3e50; background-color: #f8f9fa;'>
                                Network
                            </td>
                            <td style='padding: 15px; color: #555; font-weight: 500;'>
                                " . htmlspecialchars($network, ENT_QUOTES, 'UTF-8') . "
                            </td>
                        </tr>
                        <tr style='border-bottom: 1px solid #e0e0e0;'>
                            <td style='padding: 15px; font-weight: 600; color: #2c3e50; background-color: #f8f9fa;'>
                                Wallet Address
                            </td>
                            <td style='padding: 15px; font-family: monospace; color: #555; font-size: 13px; word-break: break-all; background-color: #ecf0f1; border-radius: 4px;'>
                                " . htmlspecialchars($wallet_address, ENT_QUOTES, 'UTF-8') . "
                            </td>
                        </tr>
                       
                        <tr style='border-bottom: 1px solid #e0e0e0;'>
                            <td style='padding: 15px; font-weight: 600; color: #2c3e50; background-color: #f8f9fa;'>
                                Date & Time
                            </td>
                            <td style='padding: 15px; color: #555; font-weight: 500;'>
                                " . date('F j, Y \a\t g:i A T') . "
                            </td>
                        </tr>
                        <tr>
                            <td style='padding: 15px; font-weight: 600; color: #2c3e50; background-color: #f8f9fa;'>
                                IP Address
                            </td>
                            <td style='padding: 15px; color: #555; font-family: monospace; font-size: 14px;'>
                                $ip_address
                            </td>
                        </tr>
                    </tbody>
                </table>
                
                <!-- Status Information -->
                <div style='background: linear-gradient(135deg, #e8f5e8 0%, #d5f4e6 100%); padding: 20px; border-radius: 8px; border-left: 5px solid #27ae60; margin: 25px 0;'>
                    <h3 style='color: #27ae60; margin: 0 0 10px 0; font-size: 16px; font-weight: 600;'>
                        🔄 Next Steps
                    </h3>
                    <p style='margin: 0; color: #2c3e50; font-size: 14px; line-height: 1.6;'>
                        Your transaction is currently being processed and verified on the blockchain. You will receive another notification once the transaction is completed and confirmed.
                    </p>
                </div>
                
                <!-- Support Information -->
                <div style='background-color: #f8f9fa; padding: 20px; border-radius: 8px; border: 1px solid #e0e0e0; margin: 25px 0;'>
                    <h3 style='color: #2c3e50; margin: 0 0 10px 0; font-size: 16px; font-weight: 600;'>
                        💬 Need Help?
                    </h3>
                    <p style='margin: 0; color: #555; font-size: 14px; line-height: 1.6;'>
                        If you have any questions about this transaction, please contact our support team at 
                        <a href='mailto:support@bitvros.com' style='color: #667eea; text-decoration: none; font-weight: 600;'>support@bitvros.com</a> 
                        and include your Transaction ID.
                    </p>
                </div>
            </div>
            
            <!-- Footer Section -->
            <div style='background-color: #2c3e50; padding: 25px; text-align: center; border-radius: 0 0 8px 8px;'>
                <p style='color: #ecf0f1; margin: 0 0 10px 0; font-size: 16px; font-weight: 500;'>
                    Thank you for using bitvros.com Wallet
                </p>
                <p style='color: #95a5a6; margin: 0; font-size: 12px;'>
                    &copy; " . date('Y') . " bitvros.com. All rights reserved.
                </p>
                <div style='margin-top: 15px; padding-top: 15px; border-top: 1px solid #34495e;'>
                    <p style='color: #95a5a6; margin: 0; font-size: 11px; line-height: 1.4;'>
                        This is an automated message. Please do not reply to this email.<br>
                        If you did not initiate this transaction, please contact our support team immediately.
                    </p>
                </div>
            </div>
        </div>
        </body></html>";
    
    $mail->AltBody = "bitvros.com - Transaction Confirmation\n\n" .
        "Dear $name,\n\n" .
        "Your transaction has been successfully processed. Please find the details below:\n\n" .
        "=== TRANSACTION DETAILS ===\n" .
        "Transaction ID: $transhash\n" .
        "Status: $status\n" .
        "Network: $network\n" .
        "Wallet Address: $wallet_address\n" .
        "Amount: $amountDisplay\n" .
        "Date & Time: " . date('F j, Y \a\t g:i A T') . "\n" .
        "IP Address: $ip_address\n\n" .
        "=== NEXT STEPS ===\n" .
        "Your transaction is currently being processed and verified on the blockchain. " .
        "You will receive another notification once the transaction is completed and confirmed.\n\n" .
        "=== NEED HELP? ===\n" .
        "If you have any questions about this transaction, please contact our support team at " .
        "support@bitvros.com and include your Transaction ID.\n\n" .
        "Thank you for using bitvros.com Wallet.\n\n" .
        "© " . date('Y') . " bitvros.com. All rights reserved.\n" .
        "This is an automated message. Please do not reply to this email.\n" .
        "If you did not initiate this transaction, please contact our support team immediately.";
    
    debugLog("Attempting to send email...");
    
    // Try to send the email
    if ($mail->send()) {
        debugLog("Email sent successfully", [
            'to' => $email,
            'bcc' => 'support@bitvros.com',
            'subject' => $mail->Subject
        ]);
        
        // Close database connection
        $conn->close();
        
        // Redirect with success message for SweetAlert
        redirectWithSuccess("Transaction processed successfully! A confirmation email has been sent to your email address.");
    } else {
        throw new Exception("Email send failed: " . $mail->ErrorInfo);
    }
    
} catch (Exception $e) {
    debugLog("Email sending failed", [
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'smtp_error' => isset($mail) ? $mail->ErrorInfo : 'N/A'
    ]);
    
    // Log additional server information for debugging
    debugLog("Server environment info", [
        'php_version' => phpversion(),
        'openssl_loaded' => extension_loaded('openssl'),
        'curl_loaded' => extension_loaded('curl'),
        'socket_loaded' => extension_loaded('sockets'),
        'mail_function_exists' => function_exists('mail'),
        'server_name' => $_SERVER['SERVER_NAME'] ?? 'unknown',
        'server_port' => $_SERVER['SERVER_PORT'] ?? 'unknown'
    ]);
    
    // Test basic mail function
    $testResult = mail('support@bitvros.com', 'Test Subject', 'Test Body', 'From: noreply@bitvros.com');
    debugLog("Basic mail() function test", ['result' => $testResult]);
    
    $conn->close();
    redirectWithError("Transaction was saved but email could not be sent. Please contact support with your transaction ID: $transhash. Error: " . $e->getMessage());
}
?>