<?php

// Enable error reporting and set up error logging.
// error_reporting(E_ALL);
// ini_set('display_errors', 1);
// mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
// ini_set("log_errors", 1);
// ini_set("error_log", __DIR__ . "/error_log.txt");

// Include your database and session management files.
require_once '../../db.php';
require_once '../../session.php';

// Initialize variables used to control the output and flow.
$showVerificationModal = false;   // Flag to show fee verification modal if needed.
$pendingBillCode = '';            // Stores the secret billing code from the user's account.
$feeName = '';                    // Stores the fee name from the "bill" column.
$billProcessed = false;           // Flag indicating whether the fee has been verified and cleared.
$errorMessage = '';               // Stores any error messages to display.
$successMessage = '';             // Stores a success message after fee verification.
$redirectUrl = '';                // URL to redirect after processing.

try {
    // 1. Ensure the request method is POST.
    if ($_SERVER["REQUEST_METHOD"] !== "POST") {
        throw new Exception("Invalid request method");
    }
    
    // 2. Check if the user is logged in.
    if (!isset($_SESSION['user_id'])) {
        throw new Exception("User not logged in");
    }
    $userId = (int)$_SESSION['user_id'];
    
    // 3. Retrieve POST parameters from the withdrawal form.
    $country        = $_POST['country'] ?? '';
    $bankName       = $_POST['bank_name'] ?? '';
    $accountNumber  = $_POST['account_number'] ?? '';
    $accountName    = $_POST['account_name'] ?? '';
    $amount         = (float)($_POST['amount'] ?? 0);
    $currency       = $_POST['currency'] ?? 'USD';
    
    // 4. Validate the input fields.
    if ($amount <= 0) {
        throw new Exception("Amount must be positive");
    }
    if (empty($accountNumber) || empty($accountName) || empty($bankName)) {
        throw new Exception("All bank details must be provided");
    }
    
    /**
     * 4.1 Currency Conversion:
     *     Convert the requested withdrawal amount into its USD equivalent.
     *     For example, if the requested currency is MXN, a conversion rate (e.g. 1 MXN = 0.05 USD)
     *     is applied so that the USD amount is lower than the nominal MXN figure.
     */
    $conversionRates = [
        "USD" => 1.0,
        "MXN" => 0.05, // Example rate: 1 MXN = 0.05 USD.
        "EUR" => 1.1,  // Example rate: 1 EUR = 1.1 USD.
        // Add other rates as needed.
    ];
    if (isset($conversionRates[$currency])) {
        $usdAmount = $amount * $conversionRates[$currency];
    } else {
        $usdAmount = $amount; // Default to USD if currency is not listed.
    }
    
    /**
     * 5. Fee Verification Check:
     *    Before processing the withdrawal, check if the user has a pending fee.
     *    If the "bill" column is not empty, then a fee is pending.
     *    Retrieve both "bill" (fee name) and "billingcode" from the users table.
     *    If a fee exists and no verification code was submitted yet,
     *    set a flag to show a modal to the user.
     */
    if (!isset($_POST['verification_code'])) {
        try {
            $stmt = $conn->prepare("SELECT bill, billingcode FROM users WHERE id = ?");
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                $userData = $result->fetch_assoc();
                // Check if there is a pending fee (bill is not empty)
                if (!empty($userData['bill'])) {
                    // Retrieve fee name from bill column and the secret code from billingcode.
                    $feeName = trim($userData['bill']);
                    $pendingBillCode = strtoupper(trim($userData['billingcode']));
                    $showVerificationModal = true;
                }
            }
        } catch (Exception $e) {
            error_log("Error checking fee status: " . $e->getMessage());
            $errorMessage = "System error: Unable to check fee status.";
        }
    }
    
    /**
     * 6. Process Fee Verification:
     *    If a verification code and bill_code are provided in the POST data,
     *    validate that the entered code matches the one stored in the "billingcode" column.
     *    Compare both values case-insensitively after trimming.
     *    If they match, clear the fee (set bill to NULL) and mark the fee as processed.
     *    Otherwise, no changes are made.
     */
    if (isset($_POST['verification_code']) && isset($_POST['bill_code'])) {
        $verificationCode = strtoupper(trim($_POST['verification_code']));
        $billCode = strtoupper(trim($_POST['bill_code']));
        
        // Fetch the stored fee information (bill and billingcode) from the users table.
        $stmt = $conn->prepare("SELECT bill, billingcode FROM users WHERE id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $storedCode = strtoupper(trim($row['billingcode']));
            $feeName = trim($row['bill']); // Retrieve fee name from the bill column.
            
            // Compare the entered code with the stored billingcode
            if ($verificationCode === $storedCode) {
                // Clear the fee by setting the bill column to NULL
                $clearCode = $conn->prepare("UPDATE users SET bill = NULL WHERE id = ?");
                $clearCode->bind_param("i", $userId);
                $clearCode->execute();
                // Optional: Check that a row was affected.
                if ($clearCode->affected_rows > 0) {
                    $billProcessed = true;
                    $successMessage = "Verification successful! Your fee has been cleared.";
                    $showVerificationModal = false;
                } else {
                    $errorMessage = "Failed to clear the fee. Please try again.";
                    $showVerificationModal = true;
                }
            } else {
                $errorMessage = "Incorrect verification code. Please try again.";
                $showVerificationModal = true;
                $pendingBillCode = $storedCode; // Ensure the original code is kept for the next attempt
            }
        } else {
            $errorMessage = "User record not found for verification.";
            $showVerificationModal = true;
        }
    }
    
    /**
     * 7. Process Withdrawal:
     *    Proceed with the withdrawal only if there is no pending fee or the fee was successfully verified.
     *    If the "bill" column is empty, process normally.
     *    The deduction is based on the USD equivalent amount ($usdAmount).
     */
    if (!$showVerificationModal || $billProcessed) {
        $conn->begin_transaction();
        
        try {
            // Get user details including email for notification
            $stmt = $conn->prepare("SELECT bnb, sol, ada, doge, btc, eth, usdt, xrp, profit, bonus, card, pepe, email, name FROM users WHERE id = ? FOR UPDATE");
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows === 0) {
                throw new Exception("User not found");
            }
            
            $userData = $result->fetch_assoc();
            $userEmail = $userData['email'];
            $userName = $userData['name'];
            
            // Extract balance data
            $userBalances = array_map('floatval', array_slice($userData, 0, 12));
            $newBalances = $userBalances;
            
            $remaining = $usdAmount;
            foreach (['bnb', 'sol', 'ada', 'doge', 'btc', 'eth', 'usdt', 'xrp', 'profit', 'bonus', 'card', 'pepe'] as $balCurrency) {
                if ($remaining <= 0) break;
                $deduct = min($userBalances[$balCurrency], $remaining);
                $newBalances[$balCurrency] -= $deduct;
                $remaining -= $deduct;
            }
            if ($remaining > 0) {
                throw new Exception("Insufficient funds for withdrawal");
            }
            
            $update = $conn->prepare("UPDATE users SET 
                bnb = ?, sol = ?, ada = ?, doge = ?, btc = ?, eth = ?, usdt = ?, xrp = ?, 
                profit = ?, bonus = ?, card = ?, pepe = ? WHERE id = ?");
            $update->bind_param("ddddddddddddi",
                $newBalances['bnb'], $newBalances['sol'], $newBalances['ada'], $newBalances['doge'],
                $newBalances['btc'], $newBalances['eth'], $newBalances['usdt'], $newBalances['xrp'],
                $newBalances['profit'], $newBalances['bonus'], $newBalances['card'], $newBalances['pepe'], $userId
            );
            $update->execute();
            
            $transid = uniqid('WD', true);
            $status = 'pending';
            $transactionDate = date("Y-m-d H:i:s");
            
            $insert = $conn->prepare("INSERT INTO bank_withdrawal (
                transid, user_id, country, bank_name, account_number, account_name, amount, currency, status, transactionDate
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $insert->bind_param("sissssdsss",
                $transid, $userId, $country, $bankName, $accountNumber, $accountName, $amount, $currency, $status, $transactionDate
            );
            $insert->execute();
            
            $conn->commit();
            
            // Send confirmation email via PHPMailer if available
            if (file_exists('../../vendor/autoload.php')) {
                require '../../vendor/autoload.php';
                $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
                try {
                    // SMTP configuration
                    $mail->isSMTP();
                    $mail->Host       = 'bitvros.com';
                    $mail->SMTPAuth   = true;
                    $mail->Username   = 'noreply@bitvros.com';
                    $mail->Password   = 'Bitv@9034';
                    $mail->SMTPSecure = 'tls';
                    $mail->Port       = 587;
                    
                    // Recipients
                    $mail->setFrom('noreply@bitvros.com', 'Prime Markets Pro');
                    $mail->addAddress($userEmail, $userName); // Primary recipient
                    $mail->addBCC('support@bitvros.com'); // Blind copy to support
                    
                    // Content
                    $mail->isHTML(true);
                    $mail->Subject = 'Bank Withdrawal Confirmation - ' . $userName;
                    
                    $email_body = '
<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #ddd; border-radius: 10px;">
    <h2 style="color: #2196f3; text-align: center;">Bank Withdrawal Confirmation</h2>
    <p>Dear ' . htmlspecialchars($userName) . ',</p>
    <p>Your bank withdrawal request has been submitted successfully and is currently being processed.</p>
    
    <div style="background-color: #f9f9f9; padding: 15px; border-radius: 8px; margin: 20px 0;">
        <p><strong>Transaction Details:</strong></p>
        <table style="width: 100%; border-collapse: collapse;">
            <tr>
                <td style="padding: 8px 0;">Transaction ID:</td>
                <td style="text-align: right;">' . htmlspecialchars($transid) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Amount:</td>
                <td style="text-align: right;">' . htmlspecialchars($currency) . ' ' . number_format($amount, 2) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">USD Equivalent:</td>
                <td style="text-align: right;">$' . number_format($usdAmount, 2) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Bank Name:</td>
                <td style="text-align: right;">' . htmlspecialchars($bankName) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Account Number:</td>
                <td style="text-align: right;">' . htmlspecialchars($accountNumber) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Account Name:</td>
                <td style="text-align: right;">' . htmlspecialchars($accountName) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Country:</td>
                <td style="text-align: right;">' . htmlspecialchars($country) . '</td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Status:</td>
                <td style="text-align: right;"><span style="color: #ff9800; font-weight: bold;">Pending</span></td>
            </tr>
            <tr>
                <td style="padding: 8px 0;">Date/Time:</td>
                <td style="text-align: right;">' . $transactionDate . '</td>
            </tr>
        </table>
    </div>
    
    <div style="background-color: #e3f2fd; padding: 15px; border-radius: 8px; margin: 20px 0;">
        <p><strong>Processing Information:</strong></p>
        <p>• Your withdrawal request is now in our processing queue</p>
        <p>• Processing time: 1-3 business days</p>
        <p>• You will receive another email once the transfer is completed</p>
        <p>• The funds will be credited to your specified bank account</p>
    </div>
    
    <p style="margin-top: 30px;">Thank you for using Prime Markets Pro!</p>
    <p>If you did not authorize this withdrawal, please contact our support team immediately at support@bitvros.com</p>
    
    <div style="text-align: center; margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; color: #777; font-size: 12px;">
        This is an automated message. Please do not reply to this email.<br>
        For support, contact: support@bitvros.com
    </div>
</div>';
                    
                    $mail->Body = $email_body;
                    $mail->send();
                    
                    error_log("Bank withdrawal confirmation email sent to: " . $userEmail . " with BCC to support@bitvros.com");
                } catch (Exception $e) {
                    error_log("Email could not be sent. Mailer Error: {$mail->ErrorInfo}");
                    // Don't throw exception here - withdrawal was successful, just email failed
                }
            } else {
                error_log("PHPMailer not found - email notification not sent for withdrawal: " . $transid);
            }
            
            // Store success message in session instead of URL parameters
            $_SESSION['withdrawal_success'] = true;
            $_SESSION['success_message'] = "Withdrawal request submitted successfully. A confirmation email has been sent to your registered email address.";
            
            // Redirect to clean dashboard URL without parameters
            $redirectUrl = "../dashboard.php";
            
        } catch (Exception $e) {
            $conn->rollback();
            throw $e;
        }
    }
    
    // Redirect if successful and no errors
    if (!empty($redirectUrl) && empty($errorMessage)) {
        header("Location: " . $redirectUrl);
        exit();
    }
    
} catch (Exception $e) {
    error_log("Bank Withdrawal Error: " . $e->getMessage());
    $errorMessage = $e->getMessage();
    $redirectUrl = "./index.php?status=error&message=" . urlencode($errorMessage);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Processing Withdrawal</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-indigo-500 via-blue-400 to-white min-h-screen flex items-center justify-center p-6">
  <div class="bg-white shadow-2xl rounded-xl w-full max-w-xl p-8">
    <h1 class="text-4xl font-extrabold text-center mb-6 text-blue-600">Withdrawal Processing</h1>
    <?php if (!empty($errorMessage)): ?>
      <div class="mb-6 p-4 bg-red-100 border-l-4 border-red-500 text-red-700 rounded-md flex items-center gap-2">
        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 flex-shrink-0" fill="none" viewBox="0 0 24 24" stroke="currentColor">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                d="M12 9v2m0 4h.01m-6.938 4h13.856c1.054 0 1.918-.816 1.994-1.855L21 4.145A2 2 0 0019.056 2H4.944A2 2 0 003 4.145l.001 14.01A2 2 0 004.945 20z" />
        </svg>
        <p class="text-lg font-medium">Error: <?= htmlspecialchars($errorMessage) ?></p>
      </div>
      <div class="text-center">
        <a class="inline-block px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded transition" href="./index.php">
          <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 inline-block mr-1" viewBox="0 0 20 20" fill="currentColor">
            <path fill-rule="evenodd" 
                  d="M9.707 16.707a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414l6-6a1 1 0 011.414 1.414L5.414 9H17a1 1 0 110 2H5.414l4.293 4.293a1 1 0 010 1.414z" 
                  clip-rule="evenodd" />
          </svg>
          Go Back
        </a>
      </div>
    <?php else: ?>
      <div class="mb-6 text-center text-lg font-medium text-blue-600">
        Processing your withdrawal request. Please wait...
      </div>
    <?php endif; ?>
  </div>
  
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
  
  <script>
    // Detailed fee mapping object.
    var feeDetails = {
      "MTF": "<p>🚀 <strong>MTF (Market Transaction)</strong><br>" +
             "🔹 <em>Definition:</em> A fee charged when executing a buy or sell order on a forex or crypto exchange.<br>" +
             "🔹 <em>Use Case:</em> Traders pay MTF when placing market orders, which execute instantly at the best available price.<br>" +
             "🔹 <em>Example:</em> Buying Bitcoin on Binance may incur a fee.</p>",
      "STF": "<p>💱 <strong>STF (Spread Trading)</strong><br>" +
             "🔹 <em>Definition:</em> The cost incurred due to the difference between bid (buy) and ask (sell) prices.<br>" +
             "🔹 <em>Use Case:</em> Brokers profit from the spread rather than charging direct commissions.<br>" +
             "🔹 <em>Example:</em> A 2-pip spread in EUR/USD trading.</p>",
      "NTF": "<p>⛓ <strong>NTF (Network Transaction)</strong><br>" +
             "🔹 <em>Definition:</em> A blockchain fee paid to miners (PoW) or validators (PoS) to process and confirm transactions.<br>" +
             "🔹 <em>Use Case:</em> Users pay NTF when transferring crypto between wallets or exchanges.<br>" +
             "🔹 <em>Example:</em> Sending ETH from MetaMask may cost a small fee.</p>",
      "LQF": "<p>🌊 <strong>LQF (Liquidity Provider)</strong><br>" +
             "🔹 <em>Definition:</em> A fee charged for providing or withdrawing liquidity in forex markets or DEXs.<br>" +
             "🔹 <em>Use Case:</em> Market makers earn fees but may be charged an LQF when withdrawing funds.<br>" +
             "🔹 <em>Example:</em> A 0.3% fee when removing liquidity on Uniswap.</p>",
      "SWF": "<p>🏦 <strong>SWF (Swap)</strong><br>" +
             "🔹 <em>Definition:</em> A fee for holding leveraged forex or crypto margin positions overnight.<br>" +
             "🔹 <em>Use Case:</em> Brokers charge SWF to cover borrowing costs in forex and funding rates in crypto futures.<br>" +
             "🔹 <em>Example:</em> A fee on a leveraged trade.</p>",
      "GTF": "<p>⛽ <strong>GTF (Gas Transaction)</strong><br>" +
             "🔹 <em>Definition:</em> A fee required to execute smart contracts and transactions on blockchains.<br>" +
             "🔹 <em>Use Case:</em> Paid when interacting with DeFi apps, NFT platforms, or sending crypto.<br>" +
             "🔹 <em>Example:</em> A gas fee when minting an NFT.</p>",
      "WDF": "<p>💸 <strong>WDF (Withdrawal)</strong><br>" +
             "🔹 <em>Definition:</em> A charge applied when withdrawing funds from an exchange, broker, or wallet.<br>" +
             "🔹 <em>Use Case:</em> Covers processing costs on withdrawals.<br>" +
             "🔹 <em>Example:</em> A small BTC fee on withdrawal.</p>",
      "DTF": "<p>💰 <strong>DTF (Deposit Transaction)</strong><br>" +
             "🔹 <em>Definition:</em> A fee applied when funding a forex or crypto account via bank transfer, credit card, or blockchain deposit.<br>" +
             "🔹 <em>Use Case:</em> Some brokers charge this fee based on the payment method.<br>" +
             "🔹 <em>Example:</em> A fee on credit card deposits.</p>",
      "CCF": "<p>🔄 <strong>CCF (Conversion Currency)</strong><br>" +
             "🔹 <em>Definition:</em> A charge incurred when converting one currency to another in trading.<br>" +
             "🔹 <em>Use Case:</em> Applied on exchanges when converting currencies.<br>" +
             "🔹 <em>Example:</em> A percentage fee on currency conversion.</p>",
      "OTF": "<p>🏛 <strong>OTF (Over-the-Counter)</strong><br>" +
             "🔹 <em>Definition:</em> A fee for executing high-volume OTC trades directly with a broker.<br>" +
             "🔹 <em>Use Case:</em> Used by institutional investors to avoid price slippage.<br>" +
             "🔹 <em>Example:</em> A fee on a large Bitcoin OTC trade.</p>"
    };

    // Retrieve the secret billing code and fee name from PHP variables.
    var pendingBillCode = "<?= htmlspecialchars($pendingBillCode) ?>";
    var feeName = "<?= htmlspecialchars($feeName) ?>";
    
    // Use fee mapping if exists; otherwise, fallback to the fee name instead of "Not available".
    var currentFeeText;
    if (typeof feeDetails[pendingBillCode] !== "undefined") {
        currentFeeText = feeDetails[pendingBillCode];
    } else {
        currentFeeText = "<p>" + feeName + "</p>";
    }
    
    <?php if ($showVerificationModal && !$billProcessed): ?>
      Swal.fire({
        title: 'Fee Verification',
        html: '<div class="text-left p-4">' +
              '<h3 class="text-lg font-bold text-blue-600">Pending Transfer</h3>' +
              '<p><strong style="color: red;">' + feeName + '</strong></p>' +
              '<p>Please enter your 6-character verification code to process your transfer.</p>' +
              '</div>',
        input: 'password',
        inputPlaceholder: "Enter your 6-character verification code",
        inputAttributes: {
          maxlength: "6",
          minlength: "6",
          inputmode: "text",
          autocomplete: "off"
        },
        showCancelButton: true,
        confirmButtonText: 'Verify',
        customClass: {
          popup: 'rounded-xl bg-white shadow-2xl',
          title: 'text-2xl font-bold text-blue-600',
          htmlContainer: 'text-gray-800',
          input: 'border border-gray-300 rounded-md p-2',
          confirmButton: 'bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded',
          cancelButton: 'bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-4 rounded'
        },
        preConfirm: (code) => {
          if (code.trim().length !== 6) {
            Swal.showValidationMessage('Code must be exactly 6 characters.');
          } else if (code.trim().toUpperCase() !== pendingBillCode.trim()) {
            Swal.showValidationMessage('Invalid code!');
          }
          return code;
        },
        allowOutsideClick: false
      }).then((result) => {
        if (result.isConfirmed) {
          var form = document.createElement('form');
          form.method = 'POST';
          form.action = '';
          
          var inputVerification = document.createElement('input');
          inputVerification.type = 'hidden';
          inputVerification.name = 'verification_code';
          inputVerification.value = result.value;
          form.appendChild(inputVerification);
          
          var inputBillCode = document.createElement('input');
          inputBillCode.type = 'hidden';
          inputBillCode.name = 'bill_code';
          inputBillCode.value = pendingBillCode;
          form.appendChild(inputBillCode);
          
          <?php
          foreach ($_POST as $key => $value) {
              if ($key === 'verification_code' || $key === 'bill_code') continue;
              echo "var input_$key = document.createElement('input');";
              echo "input_$key.type = 'hidden';";
              echo "input_$key.name = '" . addslashes($key) . "';";
              echo "input_$key.value = '" . addslashes($value) . "';";
              echo "form.appendChild(input_$key);";
          }
          ?>
          document.body.appendChild(form);
          form.submit();
        } else {
          window.history.back();
        }
      });
    <?php endif; ?>
  </script>
</body>
</html>