<?php
// Enable full error reporting (for development only)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in, if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);

if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$name    = htmlspecialchars($userDetails['name']);
$email   = htmlspecialchars($userDetails['email']);
$picture = htmlspecialchars($userDetails['picture']);
$pin     = htmlspecialchars($userDetails['pin']);
$country = htmlspecialchars($userDetails['country']);
$phone   = htmlspecialchars($userDetails['phone']);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Determine which form was submitted
    if (isset($_POST['update_profile'])) {
        // Profile update form
        $newName    = trim($_POST['name']);
        $newPin     = trim($_POST['pin']);
        $newCountry = trim($_POST['country']);
        $newPhone   = trim($_POST['phone']);

        // Validate inputs
        $errors = [];
        if (empty($newName)) {
            $errors[] = "Name is required.";
        }
        if (empty($newPin)) {
            $errors[] = "PIN is required.";
        } elseif (!is_numeric($newPin)) {
            $errors[] = "PIN must be a number.";
        }
        if (empty($newCountry)) {
            $errors[] = "Country is required.";
        }
        // Phone is optional, no validation needed

        // If no errors, update the database
        if (empty($errors)) {
            $userId = $_SESSION['user_id']; // Assuming user_id is stored in the session
            $sql = "UPDATE users SET name = ?, pin = ?, country = ?, phone = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("ssssi", $newName, $newPin, $newCountry, $newPhone, $userId);
                if ($stmt->execute()) {
                    $successMessage = "Profile updated successfully!";
                    // Refresh user details after update
                    $userDetails = getUserDetails($conn);
                    $name    = htmlspecialchars($userDetails['name']);
                    $pin     = htmlspecialchars($userDetails['pin']);
                    $country = htmlspecialchars($userDetails['country']);
                    $phone   = htmlspecialchars($userDetails['phone']);
                } else {
                    $errors[] = "Failed to update profile. Please try again.";
                }
                $stmt->close();
            } else {
                $errors[] = "Database error. Please try again.";
            }
        }
    } elseif (isset($_POST['change_password'])) {
        // Password change form
        $currentPassword = trim($_POST['current_password']);
        $newPassword = trim($_POST['new_password']);
        $confirmPassword = trim($_POST['confirm_password']);

        // Validate inputs
        $passwordErrors = [];
        if (empty($currentPassword)) {
            $passwordErrors[] = "Current password is required.";
        }
        if (empty($newPassword)) {
            $passwordErrors[] = "New password is required.";
        } elseif (strlen($newPassword) < 8) {
            $passwordErrors[] = "New password must be at least 8 characters.";
        }
        if ($newPassword !== $confirmPassword) {
            $passwordErrors[] = "New passwords do not match.";
        }

        // If no validation errors, verify current password and update
        if (empty($passwordErrors)) {
            $userId = $_SESSION['user_id'];
            
            // First, get the current hashed password
            $sql = "SELECT password FROM users WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("i", $userId);
                $stmt->execute();
                $result = $stmt->get_result();
                if ($row = $result->fetch_assoc()) {
                    $hashedPassword = $row['password'];
                    
                    // Verify the current password
                    if (password_verify($currentPassword, $hashedPassword)) {
                        // Password matches, now update with the new one
                        $newHashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                        
                        $updateSql = "UPDATE users SET password = ? WHERE id = ?";
                        $updateStmt = $conn->prepare($updateSql);
                        if ($updateStmt) {
                            $updateStmt->bind_param("si", $newHashedPassword, $userId);
                            if ($updateStmt->execute()) {
                                $passwordSuccessMessage = "Password changed successfully!";
                            } else {
                                $passwordErrors[] = "Failed to update password. Please try again.";
                            }
                            $updateStmt->close();
                        } else {
                            $passwordErrors[] = "Database error. Please try again.";
                        }
                    } else {
                        $passwordErrors[] = "Current password is incorrect.";
                    }
                } else {
                    $passwordErrors[] = "User not found.";
                }
                $stmt->close();
            } else {
                $passwordErrors[] = "Database error. Please try again.";
            }
        }
    }
}
?>
<?php include './includes/header.php'; ?>

<body class="bg-gray-50">
<?php include './includes/nav.php'; ?>



    <!-- Main Content -->
    <main class="container mx-auto px-4 py-8">
        <div class="flex items-center justify-between mb-6">
            <h1 class="text-2xl font-semibold text-gray-800">User Profile</h1>
            <div class="flex space-x-2">
                <a href="dashboard.php" class="bg-blue-100 text-blue-600 px-3 py-1 rounded-md hover:bg-blue-200">
                    <i class="fas fa-chevron-left mr-1"></i> Back to Dashboard
                </a>
            </div>
        </div>
        
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
            <!-- Profile Information Card -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <div class="text-center mb-6">
                    <img src="<?= $picture ?>" alt="User Picture" class="w-24 h-24 rounded-full mb-3 mx-auto">
                    <h2 class="text-xl font-semibold"><?= $name ?></h2>
                    <p class="text-gray-500"><?= $email ?></p>
                </div>

                <!-- Display success or error messages for profile update -->
                <?php if (!empty($successMessage)): ?>
                    <div class="mb-4 p-4 bg-green-100 text-green-800 rounded">
                        <?= $successMessage ?>
                    </div>
                <?php endif; ?>
                <?php if (!empty($errors)): ?>
                    <div class="mb-4 p-4 bg-red-100 text-red-800 rounded">
                        <?php foreach ($errors as $error): ?>
                            <p><?= $error ?></p>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <form method="POST" action="">
                    <div class="mb-4">
                        <label for="name" class="block font-semibold mb-1">Name:</label>
                        <div class="flex">
                            <input type="text" id="name" name="name" value="<?= $name ?>" readonly class="flex-1 p-2 border border-gray-300 rounded">
                            <button type="button" onclick="editField('name')" class="ml-2 p-2 bg-blue-500 text-white rounded hover:bg-blue-600">
                                <i class="fas fa-edit"></i>
                            </button>
                        </div>
                    </div>
                    <div class="mb-4">
                        <label for="email" class="block font-semibold mb-1">Email:</label>
                        <input type="text" id="email" name="email" value="<?= $email ?>" readonly class="w-full p-2 border border-gray-300 rounded bg-gray-100">
                    </div>
                    <div class="mb-4">
    <label for="pin" class="block font-semibold mb-1">PIN:</label>
    <div class="flex items-center">
       <input type="password" id="pin" name="pin" value="<?= $pin ?>" readonly maxlength="4" pattern="\d{4}" inputmode="numeric" onkeypress="if(event.charCode < 48 || event.charCode > 57){ event.preventDefault(); }" class="flex-1 p-2 border border-gray-300 rounded">

        <!-- Toggle show/hide button -->
        <button type="button" onclick="togglePinVisibility()" class="ml-2 p-2 bg-blue-500 text-white rounded hover:bg-blue-600">
            <i class="fas fa-eye"></i>
        </button>
        <!-- Edit button -->
        <button type="button" onclick="editField('pin')" class="ml-2 p-2 bg-blue-500 text-white rounded hover:bg-blue-600">
            <i class="fas fa-edit"></i>
        </button>
    </div>
</div>

<script>
function togglePinVisibility() {
    const pinInput = document.getElementById('pin');
    // Toggle between password and text input types
    pinInput.type = (pinInput.type === "password") ? "text" : "password";
}
</script>

                    <div class="mb-4">
    <label for="country" class="block font-semibold mb-1">Country:</label>
    <div class="flex relative">
        <input type="text" id="country" name="country" placeholder="Select Country" readonly class="flex-1 p-2 border border-gray-300 rounded">
        <button type="button" onclick="toggleDropdown()" class="ml-2 p-2 bg-blue-500 text-white rounded hover:bg-blue-600">
            <i class="fas fa-edit"></i>
        </button>
        <div id="dropdown" class="absolute w-full bg-white border border-gray-300 rounded mt-1 hidden max-h-40 overflow-y-auto z-10">
            <input type="text" id="search" class="p-2 w-full border-b border-gray-300" placeholder="Search country..." onkeyup="filterCountries()">
            <ul id="countryList" class="max-h-40 overflow-y-auto"></ul>
        </div>
    </div>
</div>

<script>
const countries = [
    "Afghanistan", "Albania", "Algeria", "Andorra", "Angola", "Antigua and Barbuda", "Argentina", "Armenia", "Australia", "Austria", 
    "Azerbaijan", "Bahamas", "Bahrain", "Bangladesh", "Barbados", "Belarus", "Belgium", "Belize", "Benin", "Bhutan", 
    "Bolivia", "Bosnia and Herzegovina", "Botswana", "Brazil", "Brunei", "Bulgaria", "Burkina Faso", "Burundi", "Cabo Verde", "Cambodia",
    "Cameroon", "Canada", "Central African Republic", "Chad", "Chile", "China", "Colombia", "Comoros", "Congo", "Costa Rica",
    "Croatia", "Cuba", "Cyprus", "Czech Republic", "Denmark", "Djibouti", "Dominica", "Dominican Republic", "Ecuador", "Egypt",
    "El Salvador", "Equatorial Guinea", "Eritrea", "Estonia", "Eswatini", "Ethiopia", "Fiji", "Finland", "France", "Gabon",
    "Gambia", "Georgia", "Germany", "Ghana", "Greece", "Grenada", "Guatemala", "Guinea", "Guinea-Bissau", "Guyana",
    "Haiti", "Honduras", "Hungary", "Iceland", "India", "Indonesia", "Iran", "Iraq", "Ireland", "Israel",
    "Italy", "Ivory Coast", "Jamaica", "Japan", "Jordan", "Kazakhstan", "Kenya", "Kiribati", "Kuwait", "Kyrgyzstan",
    "Laos", "Latvia", "Lebanon", "Lesotho", "Liberia", "Libya", "Liechtenstein", "Lithuania", "Luxembourg", "Madagascar",
    "Malawi", "Malaysia", "Maldives", "Mali", "Malta", "Marshall Islands", "Mauritania", "Mauritius", "Mexico", "Micronesia",
    "Moldova", "Monaco", "Mongolia", "Montenegro", "Morocco", "Mozambique", "Myanmar", "Namibia", "Nauru", "Nepal",
    "Netherlands", "New Zealand", "Nicaragua", "Niger", "Nigeria", "North Korea", "North Macedonia", "Norway", "Oman", "Pakistan",
    "Palau", "Panama", "Papua New Guinea", "Paraguay", "Peru", "Philippines", "Poland", "Portugal", "Qatar", "Romania",
    "Russia", "Rwanda", "Saint Kitts and Nevis", "Saint Lucia", "Saint Vincent and the Grenadines", "Samoa", "San Marino", "Sao Tome and Principe", "Saudi Arabia", "Senegal",
    "Serbia", "Seychelles", "Sierra Leone", "Singapore", "Slovakia", "Slovenia", "Solomon Islands", "Somalia", "South Africa", "South Korea",
    "South Sudan", "Spain", "Sri Lanka", "Sudan", "Suriname", "Sweden", "Switzerland", "Syria", "Tajikistan", "Tanzania",
    "Thailand", "Timor-Leste", "Togo", "Tonga", "Trinidad and Tobago", "Tunisia", "Turkey", "Turkmenistan", "Tuvalu", "Uganda",
    "Ukraine", "United Arab Emirates", "United Kingdom", "United States", "Uruguay", "Uzbekistan", "Vanuatu", "Vatican City", "Venezuela", "Vietnam",
    "Yemen", "Zambia", "Zimbabwe"
];

const dropdown = document.getElementById("dropdown");
const countryInput = document.getElementById("country");
const countryList = document.getElementById("countryList");

function toggleDropdown() {
    dropdown.classList.toggle("hidden");
    populateCountries();
}

function populateCountries() {
    countryList.innerHTML = "";
    countries.forEach(country => {
        const li = document.createElement("li");
        li.classList.add("p-2", "hover:bg-gray-200", "cursor-pointer");
        li.textContent = country;
        li.onclick = function() {
            countryInput.value = this.textContent;
            dropdown.classList.add("hidden");
        };
        countryList.appendChild(li);
    });
}

function filterCountries() {
    const searchValue = document.getElementById("search").value.toLowerCase();
    const items = countryList.getElementsByTagName("li");
    for (let i = 0; i < items.length; i++) {
        if (items[i].textContent.toLowerCase().includes(searchValue)) {
            items[i].style.display = "";
        } else {
            items[i].style.display = "none";
        }
    }
}
</script>

                    <div class="mb-4">
                        <label for="phone" class="block font-semibold mb-1">Phone:</label>
                        <div class="flex">
                            <input type="text" id="phone" name="phone" value="<?= $phone ?>" readonly class="flex-1 p-2 border border-gray-300 rounded">
                            <button type="button" onclick="editField('phone')" class="ml-2 p-2 bg-blue-500 text-white rounded hover:bg-blue-600">
                                <i class="fas fa-edit"></i>
                            </button>
                        </div>
                    </div>
                    <input type="hidden" name="update_profile" value="1">
                    <button type="submit" class="w-full p-2 bg-green-500 text-white rounded hover:bg-green-600">
                        <i class="fas fa-save mr-2"></i> Save Profile Changes
                    </button>
                </form>
            </div>

            <!-- Password Change Card -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <h2 class="text-xl font-semibold mb-4">Change Password</h2>
                
                <!-- Display password change messages -->
                <?php if (!empty($passwordSuccessMessage)): ?>
                    <div class="mb-4 p-4 bg-green-100 text-green-800 rounded">
                        <?= $passwordSuccessMessage ?>
                    </div>
                <?php endif; ?>
                <?php if (!empty($passwordErrors)): ?>
                    <div class="mb-4 p-4 bg-red-100 text-red-800 rounded">
                        <?php foreach ($passwordErrors as $error): ?>
                            <p><?= $error ?></p>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <form method="POST" action="">
                    <div class="mb-4">
                        <label for="current_password" class="block font-semibold mb-1">Current Password:</label>
                        <div class="relative">
                            <input type="password" id="current_password" name="current_password" class="w-full p-2 border border-gray-300 rounded pr-10">
                            <button type="button" class="absolute right-2 top-2 text-gray-500 hover:text-gray-700" onclick="togglePasswordVisibility('current_password')">
                                <i class="far fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    <div class="mb-4">
                        <label for="new_password" class="block font-semibold mb-1">New Password:</label>
                        <div class="relative">
                            <input type="password" id="new_password" name="new_password" class="w-full p-2 border border-gray-300 rounded pr-10">
                            <button type="button" class="absolute right-2 top-2 text-gray-500 hover:text-gray-700" onclick="togglePasswordVisibility('new_password')">
                                <i class="far fa-eye"></i>
                            </button>
                        </div>
                        <p class="text-xs text-gray-500 mt-1">Password must be at least 8 characters long</p>
                    </div>
                    <div class="mb-6">
                        <label for="confirm_password" class="block font-semibold mb-1">Confirm New Password:</label>
                        <div class="relative">
                            <input type="password" id="confirm_password" name="confirm_password" class="w-full p-2 border border-gray-300 rounded pr-10">
                            <button type="button" class="absolute right-2 top-2 text-gray-500 hover:text-gray-700" onclick="togglePasswordVisibility('confirm_password')">
                                <i class="far fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    <input type="hidden" name="change_password" value="1">
                    <button type="submit" class="w-full p-2 bg-blue-600 text-white rounded hover:bg-blue-700">
                        <i class="fas fa-lock mr-2"></i> Change Password
                    </button>
                </form>

                <div class="mt-6 border-t pt-4">
                    <h3 class="font-semibold mb-2">Security Tips</h3>
                    <ul class="text-sm text-gray-600 space-y-1 list-disc pl-5">
                        <li>Use a strong password with letters, numbers, and symbols</li>
                        <li>Never share your password or PIN with anyone</li>
                        <li>Change your password regularly</li>
                        <li>Use different passwords for different accounts</li>
                    </ul>
                </div>
            </div>
        </div>
    </main>



    <script>
        // Make field editable when clicked
        function editField(fieldId) {
            const inputElement = document.getElementById(fieldId);
            inputElement.removeAttribute('readonly');
            inputElement.classList.remove('bg-gray-100');
            inputElement.focus();
        }

        // Toggle password visibility
        function togglePasswordVisibility(inputId) {
            const input = document.getElementById(inputId);
            const eyeIcon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                eyeIcon.classList.remove('fa-eye');
                eyeIcon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                eyeIcon.classList.remove('fa-eye-slash');
                eyeIcon.classList.add('fa-eye');
            }
        }
    </script>
    
    
    
    
    <!-- JavaScript to hide the preloader once the DOM is loaded -->
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // Optionally stop the spinner animation
      var spinner = document.getElementById('spinner');
      if (spinner) {
        spinner.classList.remove('animate-spin');
      }
      
      // Hide the preloader container
      var preloader = document.getElementById('preloader');
      if (preloader) {
        preloader.style.display = 'none';
      }
    });
  </script>
    
    <?php include './includes/footer.php'; ?>

</body>
</html>