<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error_log.txt');

require_once '../db.php';
require_once '../session.php';

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method');
    }

    if (!isset($_SESSION['user_id'])) {
        throw new Exception('User not logged in');
    }

    $userId = (int)$_SESSION['user_id'];
    $transactionType = 'buy';

    // Sanitize amount
    $price = filter_var(
        $_POST['price'] ?? 0,
        FILTER_SANITIZE_NUMBER_FLOAT,
        FILTER_FLAG_ALLOW_FRACTION
    ) ?: 0;
    $price = (float)$price;
    if ($price < 250) {
        throw new Exception('Price must be at least $250');
    }

    // Sanitize percent
    $percentValue = (float)str_replace('%', '', $_POST['percentValue'] ?? '50%');

    // Sanitize and validate crypto
    $allowedCryptos = ['bitcoin','ethereum','tether','solana','cardano','dogecoin'];
    $selectedCrypto = $_POST['crypto'] ?? '';
    if (!in_array($selectedCrypto, $allowedCryptos, true)) {
        $selectedCrypto = 'bitcoin';
    }

    // Sanitize wallet address
    $walletAddress = filter_var(
        $_POST['walletAddress'] ?? '',
        FILTER_SANITIZE_STRING
    );
    if (!$walletAddress) {
        throw new Exception('Wallet address missing');
    }

    // Fetch name/email if not provided
    $name  = filter_var($_POST['name']  ?? '', FILTER_SANITIZE_STRING);
    $email = filter_var($_POST['email'] ?? '', FILTER_SANITIZE_EMAIL);
    if (empty($name) || empty($email)) {
        $stmt = $conn->prepare('SELECT name,email FROM users WHERE id = ?');
        $stmt->bind_param('i', $userId);
        $stmt->execute();
        $userData = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        if (!$userData) {
            throw new Exception('User data not found');
        }
        $name  = $userData['name'];
        $email = $userData['email'];
    }

    // Generate transaction ID
    $transid = uniqid('TX', true);
    $status = 'pending';
    $transactionDate = date('Y-m-d H:i:s');

    // Insert into buysignal
    $sql = "INSERT INTO buysignal
      (transaction_id, user_id, amount, crypto, percent_value, wallet_address, status, transaction_date, type)
     VALUES (?,?,?,?,?,?,?,?,?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param(
      'sidsdssss',
      $transid,
      $userId,
      $price,
      $selectedCrypto,
      $percentValue,
      $walletAddress,
      $status,
      $transactionDate,
      $transactionType
    );

    if (!$stmt->execute()) {
        throw new Exception('DB insert failed: ' . $stmt->error);
    }
    $stmt->close();

    // Redirect to signalf.php on success
    header('Location: ./signalf.php?' . http_build_query([
        'transid'         => $transid,
        'user_id'         => $userId,
        'name'            => $name,
        'email'           => $email,
        'amount'          => $price,
        'signa'           => $percentValue,
        'status'          => $status,
        'transactionDate' => $transactionDate,
        'crypto'          => $selectedCrypto,
        'walletAddress'   => $walletAddress
    ]));
    exit();

} catch (Exception $e) {
    error_log('Transaction Error: ' . $e->getMessage());
    // Redirect back to the form page with a Tailwind‐friendly error msg
    header('Location: ./signalbuy.php?status=error&message=' . urlencode($e->getMessage()));
    exit;
}
?>
