<?php
// Enable detailed error reporting for debugging purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Optionally, log errors to a file (error_log.txt in the current directory)
ini_set("log_errors", 1);
ini_set("error_log", __DIR__ . "/error_log.txt");

// Include necessary files (database connection and session management)
require_once 'db.php';
require_once '../session.php';

try {
    if ($_SERVER["REQUEST_METHOD"] === "POST") {
        // Ensure user is logged in
        if (!isset($_SESSION['user_id'])) {
            throw new Exception("User not logged in");
        }
        $userId = (int) $_SESSION['user_id'];
        
        // Validate and sanitize form data
        $transactionType     = isset($_POST['transactionType']) ? strtolower(trim($_POST['transactionType'])) : '';
        $transactionCurrency = isset($_POST['transactionCurrency']) ? trim($_POST['transactionCurrency']) : '';
        $transactionAmount   = isset($_POST['transactionAmount']) ? floatval($_POST['transactionAmount']) : 0.0;
        $convertedAmount     = isset($_POST['convertedAmount']) ? floatval($_POST['convertedAmount']) : 0.0;
        $transactionDate     = date("Y-m-d H:i:s");
        
        // Validate transaction type
        if ($transactionType !== 'buy' && $transactionType !== 'sell') {
            throw new Exception("Invalid transaction type");
        }
        
        // Get the user's current balances (columns: btc, eth, usdt, xrp)
        $stmt = $conn->prepare("SELECT btc, eth, usdt, xrp FROM users WHERE id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            throw new Exception("User not found");
        }
        
        $userBalances = $result->fetch_assoc();
        
        // Explicitly cast all balance values to float
        $userBalances['btc'] = (float)$userBalances['btc'];
        $userBalances['eth'] = (float)$userBalances['eth'];
        $userBalances['usdt'] = (float)$userBalances['usdt'];
        $userBalances['xrp'] = (float)$userBalances['xrp'];
        
        // Calculate total balance across all cryptocurrencies
        $totalBalance = $userBalances['btc'] + $userBalances['eth'] + $userBalances['usdt'] + $userBalances['xrp'];
        
        // Check if user has enough total balance for the transaction
        if ($totalBalance < $transactionAmount) {
            throw new Exception("Insufficient total balance for this transaction");
        }
        
        // Fetch user email and name for recording the transaction
        $userStmt = $conn->prepare("SELECT email, name FROM users WHERE id = ?");
        $userStmt->bind_param("i", $userId);
        $userStmt->execute();
        $userResult = $userStmt->get_result();
        $userData = $userResult->fetch_assoc();
        $userEmail = $userData['email'];
        $userName  = $userData['name'];
        
        // Prepare new balances from the current balances
        $newBalances = $userBalances;
        
        // Strategy for subtracting transaction amount:
        // Deduct funds from the user's balances in a set priority: USDT, BTC, ETH, then XRP.
        $remainingAmount = $transactionAmount;
        $currencyPriority = ['usdt', 'btc', 'eth', 'xrp'];
        
        foreach ($currencyPriority as $currency) {
            if ($remainingAmount <= 0) break;
            
            $amountToSubtract = min($newBalances[$currency], $remainingAmount);
            $newBalances[$currency] -= $amountToSubtract;
            $remainingAmount -= $amountToSubtract;
        }
        
        // Record the transaction in the crypto_stakes table
        $transStmt = $conn->prepare("INSERT INTO crypto_stakes (email, name, sessionType, sessionCurrency, sessionAmount, convertedAmount, sessionDate) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $transStmt->bind_param("ssssdds", $userEmail, $userName, $transactionType, $transactionCurrency, $transactionAmount, $convertedAmount, $transactionDate);
        $transStmt->execute();
        
        // Update user balances based on transaction type
        if ($transactionType === 'buy') {
            // For a 'buy' transaction, add the purchased (converted) amount to the corresponding currency balance.
            $currencyKey = strtolower($transactionCurrency);
            if (array_key_exists($currencyKey, $newBalances)) {
                $newBalances[$currencyKey] += $convertedAmount;
            }
        } elseif ($transactionType === 'sell') {
            // For a 'sell' transaction, add the received funds (transactionAmount) to the USDT balance.
            $newBalances['usdt'] += $transactionAmount;
        }
        
        // Update user balances in the database
        $updateStmt = $conn->prepare("UPDATE users SET btc = ?, eth = ?, usdt = ?, xrp = ? WHERE id = ?");
        $updateStmt->bind_param("ddddi", $newBalances['btc'], $newBalances['eth'], $newBalances['usdt'], $newBalances['xrp'], $userId);
        $updateStmt->execute();
        
        // Redirect to dashboard with success message
        header("Location: dashboard.php?status=success&message=" . urlencode("Transaction completed successfully"));
        exit();
    } else {
        // If not a POST request, simply redirect to dashboard
        header("Location: dashboard.php");
        exit();
    }
} catch (Exception $e) {
    // Log error details for debugging
    error_log("Transaction Error: " . $e->getMessage());
    // Redirect to dashboard with error details in URL parameters
    header("Location: dashboard.php?status=error&message=" . urlencode($e->getMessage()));
    exit();
}
?>
