<?php
// index.php

/**
 * Translate text using the LibreTranslate API.
 *
 * @param string $text The text to be translated.
 * @param string $targetLanguage The target language code.
 * @return string The translated text or an error message.
 */
function translate($text, $targetLanguage) {
    // No translation needed if text is empty or target language is English.
    if (empty($text) || $targetLanguage === 'en') {
        return $text;
    }
    
    $apiUrl = 'https://libretranslate.com/translate';
    // Optionally, add an API key if available:
    // $apiKey = 'your-api-key';
    
    $postData = [
        'q'      => $text,
        'source' => 'auto',       // Auto-detect source language.
        'target' => $targetLanguage,
        'format' => 'text',
        // 'api_key' => $apiKey,   // Uncomment if you have an API key.
    ];
    
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    // Send the POST data as JSON.
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    $response = curl_exec($ch);
    
    // Check for cURL errors.
    if (curl_errno($ch)) {
        $error_msg = curl_error($ch);
        curl_close($ch);
        return 'Translation error: ' . $error_msg;
    }
    
    // Verify that we received a 200 OK response.
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($httpCode !== 200) {
        curl_close($ch);
        return "Translation error: HTTP Code $httpCode";
    }
    
    curl_close($ch);
    $responseData = json_decode($response, true);
    
    if (!is_array($responseData) || !isset($responseData['translatedText'])) {
        // Return API error if available, else a generic error.
        $error = isset($responseData['error']) ? $responseData['error'] : 'Unknown error';
        return 'Translation error: ' . $error;
    }
    
    return $responseData['translatedText'];
}

// Define the page content for translation.
$pageContent = [
    'title'      => 'Welcome to Our Website',
    'heading'    => 'Hello, welcome to our website!',
    'paragraph1' => 'This is a multilingual website that supports multiple languages.',
    'paragraph2' => 'Select your preferred language from the dropdown menu below.',
    'about'      => 'About Us',
    'aboutText'  => 'We are a company dedicated to providing excellent service to customers around the world.',
    'contact'    => 'Contact Us',
    'contactText'=> 'Email us at info@example.com or call us at +1-234-567-8900.',
    'footer'     => 'Copyright © 2025. All rights reserved.'
];

// Define valid language codes.
$validLanguages = ['en', 'de', 'fr', 'es'];

// Determine current language from POST request or default to English.
$currentLanguage = 'en';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['language']) && in_array($_POST['language'], $validLanguages)) {
    $currentLanguage = $_POST['language'];
}

// Translate content if target language is not English.
$translatedContent = $pageContent;
if ($currentLanguage !== 'en') {
    foreach ($pageContent as $key => $text) {
        $translatedContent[$key] = translate($text, $currentLanguage);
    }
}

// Define flag image URLs.
$flagUrls = [
    'en' => 'https://flagcdn.com/16x12/us.png',
    'de' => 'https://flagcdn.com/16x12/de.png',
    'fr' => 'https://flagcdn.com/16x12/fr.png',
    'es' => 'https://flagcdn.com/16x12/es.png'
];

// Define language names for dropdown display.
$languageNames = [
    'en' => 'English',
    'de' => 'Deutsch',
    'fr' => 'Français',
    'es' => 'Español'
];
?>
<!DOCTYPE html>
<html lang="<?php echo htmlspecialchars($currentLanguage); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($translatedContent['title']); ?></title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 0;
            padding: 0;
            color: #333;
        }
        .container {
            width: 80%;
            margin: 0 auto;
            padding: 20px;
        }
        header {
            background-color: #f8f9fa;
            padding: 20px;
            margin-bottom: 20px;
            border-bottom: 1px solid #ddd;
        }
        section {
            margin-bottom: 30px;
        }
        footer {
            background-color: #f8f9fa;
            padding: 20px;
            margin-top: 20px;
            border-top: 1px solid #ddd;
            text-align: center;
        }
        .language-selector {
            position: fixed;
            bottom: 20px;
            left: 20px;
            z-index: 9999;
            background-color: #8a2be2;
            padding: 10px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.25);
            display: flex;
            gap: 8px;
            align-items: center;
            color: white;
        }
        .language-selector form {
            margin: 0;
        }
        .language-selector select {
            padding: 8px;
            border-radius: 4px;
            border: 1px solid #ccc;
            background-color: white;
            cursor: pointer;
            font-size: 14px;
            min-width: 150px;
        }
        .language-selector select:focus {
            outline: none;
            border-color: #7a29d7;
        }
        /* Optional styling for custom dropdown if enabled */
        .custom-select-wrapper {
            position: relative;
        }
        .selected-option {
            padding: 8px;
            border: 1px solid #ccc;
            border-radius: 4px;
            background-color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
        }
        .options-list {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background-color: white;
            border: 1px solid #ccc;
            border-radius: 0 0 4px 4px;
            display: none;
            z-index: 1000;
        }
        .option-item {
            padding: 8px;
            cursor: pointer;
            display: flex;
            align-items: center;
        }
        .option-item:hover {
            background-color: #f1f1f1;
        }
        .flag-img {
            margin-right: 8px;
            vertical-align: middle;
            width: 16px;
            height: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <h1><?php echo htmlspecialchars($translatedContent['heading']); ?></h1>
            <p><?php echo htmlspecialchars($translatedContent['paragraph1']); ?></p>
        </header>
        
        <section>
            <p><?php echo htmlspecialchars($translatedContent['paragraph2']); ?></p>
        </section>
        
        <section>
            <h2><?php echo htmlspecialchars($translatedContent['about']); ?></h2>
            <p><?php echo htmlspecialchars($translatedContent['aboutText']); ?></p>
        </section>
        
        <section>
            <h2><?php echo htmlspecialchars($translatedContent['contact']); ?></h2>
            <p><?php echo htmlspecialchars($translatedContent['contactText']); ?></p>
        </section>
        
        <footer>
            <p><?php echo htmlspecialchars($translatedContent['footer']); ?></p>
        </footer>
    </div>
    
    <div class="language-selector">
        <form method="post" id="langForm">
            <select name="language" onchange="this.form.submit()">
                <?php foreach ($languageNames as $code => $name): ?>
                    <option value="<?php echo htmlspecialchars($code); ?>" <?php if ($currentLanguage === $code) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($name); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </form>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Optional: Custom dropdown with flags.
            // Standard HTML <select> elements do not support images,
            // so the following function creates a custom dropdown.
            function createCustomDropdown() {
                const select = document.querySelector('select[name="language"]');
                const selectWrapper = document.createElement('div');
                selectWrapper.className = 'custom-select-wrapper';
                selectWrapper.style.position = 'relative';
                
                const selectedOption = document.createElement('div');
                selectedOption.className = 'selected-option';
                // Mapping flag URLs (passed from PHP) for current language.
                const flagUrls = {
                    'en': '<?php echo $flagUrls['en']; ?>',
                    'de': '<?php echo $flagUrls['de']; ?>',
                    'fr': '<?php echo $flagUrls['fr']; ?>',
                    'es': '<?php echo $flagUrls['es']; ?>'
                };
                const currentLang = select.value;
                selectedOption.innerHTML = `<img src="${flagUrls[currentLang]}" class="flag-img" alt="Flag"> <span>${select.options[select.selectedIndex].text}</span>`;
                
                const optionsList = document.createElement('div');
                optionsList.className = 'options-list';
                
                Array.from(select.options).forEach(option => {
                    const optionItem = document.createElement('div');
                    optionItem.className = 'option-item';
                    optionItem.innerHTML = `<img src="${flagUrls[option.value]}" class="flag-img" alt="Flag"> <span>${option.text}</span>`;
                    optionItem.addEventListener('click', function() {
                        select.value = option.value;
                        document.getElementById('langForm').submit();
                    });
                    optionsList.appendChild(optionItem);
                });
                
                selectedOption.addEventListener('click', function() {
                    optionsList.style.display = optionsList.style.display === 'none' || optionsList.style.display === '' ? 'block' : 'none';
                });
                
                document.addEventListener('click', function(e) {
                    if (!selectWrapper.contains(e.target)) {
                        optionsList.style.display = 'none';
                    }
                });
                
                selectWrapper.appendChild(selectedOption);
                selectWrapper.appendChild(optionsList);
                select.style.display = 'none';
                select.parentNode.insertBefore(selectWrapper, select);
            }
            
            // Uncomment the line below to activate the custom dropdown with flags:
            // createCustomDropdown();
        });
    </script>
</body>
</html>
