<?php
// Enable full error reporting (development only; disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include session and database management
include '../session.php';

// Initialize error array
$errors = [];

// Check if user is logged in; if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from session/database
$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    $errors[] = 'User not found or session expired.';
} else {
    // Extract user details (using null coalescing for safety)
    $google_id = $userDetails['google_id'] ?? '';
    $name      = $userDetails['name'] ?? '';
    $email     = $userDetails['email'] ?? '';
    $phone     = $userDetails['phone'] ?? '';
    $country   = $userDetails['country'] ?? '';

    // Check if the user already has a KYC submission
    $checkStmt = $conn->prepare("SELECT status FROM kyc_submissions WHERE email = ? ORDER BY submission_time DESC LIMIT 1");
    if (!$checkStmt) {
        $errors[] = "Failed to prepare check statement: " . $conn->error;
    } else {
        $checkStmt->bind_param("s", $email);
        $checkStmt->execute();
        $checkResult = $checkStmt->get_result();

        if ($checkResult->num_rows > 0) {
            // User already has a KYC submission
            $kycRecord = $checkResult->fetch_assoc();
            $status = $kycRecord['status'];
            $checkStmt->close();

            // Store the status in the session for SweetAlert display
            $_SESSION['kyc_status'] = $status;

            // Redirect to the dashboard
            header('Location: ./dashboard.php');
            exit();
        }
        $checkStmt->close();
    }
}

/**
 * Upload file function with error handling.
 *
 * @param string $fileInputName The name of the file input.
 * @param string $uploadDir The directory where the file will be uploaded.
 * @param array  &$errors Reference to errors array.
 * @return string|null Returns the file name on success or null on failure.
 */
function uploadFile($fileInputName, $uploadDir, &$errors) {
    // No file uploaded
    if (!isset($_FILES[$fileInputName]) || $_FILES[$fileInputName]['error'] === UPLOAD_ERR_NO_FILE) {
        return null;
    }
    
    // Handle file upload errors with detailed messages
    if ($_FILES[$fileInputName]['error'] !== UPLOAD_ERR_OK) {
        $errorCode = $_FILES[$fileInputName]['error'];
        switch ($errorCode) {
            case UPLOAD_ERR_INI_SIZE:
                $message = "The uploaded file exceeds the upload_max_filesize directive in php.ini.";
                break;
            case UPLOAD_ERR_FORM_SIZE:
                $message = "The uploaded file exceeds the MAX_FILE_SIZE directive specified in the HTML form.";
                break;
            case UPLOAD_ERR_PARTIAL:
                $message = "The uploaded file was only partially uploaded.";
                break;
            case UPLOAD_ERR_NO_FILE:
                $message = "No file was uploaded.";
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $message = "Missing a temporary folder.";
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $message = "Failed to write file to disk.";
                break;
            case UPLOAD_ERR_EXTENSION:
                $message = "A PHP extension stopped the file upload.";
                break;
            default:
                $message = "Unknown upload error.";
                break;
        }
        $errors[] = "File upload error for $fileInputName. $message (Error code: $errorCode)";
        return null;
    }
    
    // Validate MIME type and file size (max 16 MB)
    $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/bmp'];
    $maxFileSize = 16 * 1024 * 1024;
    
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $fileMimeType = finfo_file($finfo, $_FILES[$fileInputName]['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($fileMimeType, $allowedMimeTypes)) {
        $errors[] = "Invalid file type for $fileInputName. Allowed types: JPEG, PNG, GIF, WEBP, BMP.";
        return null;
    }
    
    if ($_FILES[$fileInputName]['size'] > $maxFileSize) {
        $errors[] = "File size for $fileInputName exceeds 16 MB.";
        return null;
    }
    
    // Generate a unique file name and build the file path
    $fileName = uniqid() . '_' . basename($_FILES[$fileInputName]['name']);
    $filePath = $uploadDir . $fileName;
    
    if (move_uploaded_file($_FILES[$fileInputName]['tmp_name'], $filePath)) {
        // Return only the file name so that the folder name is not stored in the database
        return $fileName;
    } else {
        $errors[] = "Failed to move uploaded file for $fileInputName.";
        return null;
    }
}

// Process form submission only when a POST request is made
$formSubmitted = false;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $formSubmitted = true;
    
    // Validate the required field: document type must be set
    if (empty($_POST['document_type'])) {
        $errors[] = "Document type is required.";
    } else {
        $allowedDocumentTypes = ['passport', 'id_card', 'drivers_license'];
        $documentType = $_POST['document_type'];
        if (!in_array($documentType, $allowedDocumentTypes)) {
            $errors[] = "Invalid document type.";
        } else {
            // Set up the upload directory and create it if necessary
            $uploadDir = 'uploads/';
            if (!is_dir($uploadDir)) {
                if (!mkdir($uploadDir, 0755, true)) {
                    $errors[] = "Unable to create upload directory.";
                }
            }
            
            // Only proceed with file uploads if there are no errors so far
            if (empty($errors)) {
                // Process file uploads based on the selected document type
                $documentFront = uploadFile($documentType . '_front', $uploadDir, $errors);
                // For passports, the back side is not required; use an empty string instead of null to avoid DB errors
                $documentBack = ($documentType !== 'passport') ? uploadFile($documentType . '_back', $uploadDir, $errors) : '';
                $selfie = uploadFile($documentType . '_selfie', $uploadDir, $errors);
                
                // Ensure that required files are provided
                if (!$documentFront) {
                    $errors[] = "Front side of document is required.";
                }
                if (!$selfie) {
                    $errors[] = "Selfie with document is required.";
                }
                
                // Continue with submission if no errors occurred
                if (empty($errors)) {
                    // Generate a unique transaction hash
                    $transhash = hash('sha256', uniqid($google_id, true));
                    
                    // Set default status to 'pending'
                    $status = 'pending';
                    
                    // Prepare the SQL statement to insert the KYC submission data
                    $stmt = $conn->prepare("INSERT INTO kyc_submissions (
                        google_id, name, email, phone, country, document_type,
                        document_front, document_back, selfie, transhash, status
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    
                    if (!$stmt) {
                        $errors[] = "Failed to prepare SQL statement: " . $conn->error;
                    } else {
                        $stmt->bind_param(
                            "sssssssssss",
                            $google_id,
                            $name,
                            $email,
                            $phone,
                            $country,
                            $documentType,
                            $documentFront,
                            $documentBack,
                            $selfie,
                            $transhash,
                            $status
                        );
                        
                        if ($stmt->execute()) {
                            // PHPMailer integration: send a confirmation email to the user.
                            try {
                                // First, check if Composer autoloader exists
                                if (file_exists('./vendor/autoload.php')) {
                                    require './vendor/autoload.php';
                                } else {
                                    // Direct inclusion of PHPMailer files if no autoloader is found
                                    require_once './PHPMailer/src/Exception.php';
                                    require_once './PHPMailer/src/PHPMailer.php';
                                    require_once './PHPMailer/src/SMTP.php';
                                }
                                
                                // Create a new PHPMailer instance
                                $mail = new PHPMailer\PHPMailer\PHPMailer(true); // true enables exceptions
                                
                                // Server settings
                                $mail->SMTPDebug = 0;  // Disable verbose debug output
                                $mail->isSMTP();       // Send using SMTP
                                $mail->Host       = 'bitvros.com';  // SMTP server
                                $mail->SMTPAuth   = true;                     // Enable SMTP authentication
                                $mail->Username   = 'support@bitvros.com'; // SMTP username
                                $mail->Password   = 'Bitv@9034';              // SMTP password
                                $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS; // Enable TLS encryption
                                $mail->Port       = 587;                      // TCP port to connect to
                                
                                // Set who the message is to be sent from
                                $mail->setFrom('support@bitvros.com', 'bitvros.com Support');
                                
                                // Add a recipient
                                $mail->addAddress($email, $name);
                                
                                // Set email format to HTML
                                $mail->isHTML(true);
                                
                                // Set the subject and body
                                $mail->Subject = 'KYC Submission Received';
                                $mail->Body = "
                                <!DOCTYPE html>
                                <html>
                                <head>
                                    <style>
                                        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                                        .container { max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #ddd; border-radius: 5px; }
                                        .header { background-color: #0066cc; color: white; padding: 15px; text-align: center; border-radius: 5px 5px 0 0; }
                                        .content { padding: 20px; }
                                        .footer { text-align: center; font-size: 12px; color: #777; margin-top: 20px; }
                                    </style>
                                </head>
                                <body>
                                    <div class='container'>
                                        <div class='header'>
                                            <h2>KYC Submission Received</h2>
                                        </div>
                                        <div class='content'>
                                            <p>Dear $name,</p>
                                            <p>Thank you for submitting your KYC documents. Your submission (ID: $transhash) is currently under review.</p>
                                            <p>We will notify you once the verification process is complete.</p>
                                            <p>If you have any questions, please contact our support team.</p>
                                            <p>Best regards,<br>bitvros.com Team</p>
                                        </div>
                                        <div class='footer'>
                                            <p>&copy; " . date('Y') . " bitvros.com. All rights reserved.</p>
                                            <p>This is an automated message; please do not reply.</p>
                                        </div>
                                    </div>
                                </body>
                                </html>
                                ";
                                
                                // Plain text version for non-HTML mail clients
                                $mail->AltBody = "Dear $name,\n\nThank you for submitting your KYC documents. Your submission (ID: $transhash) is currently under review. We will notify you once the verification process is complete.\n\nBest regards,\nAltrex Team";
                                
                                // Send the email
                                $mail->send();
                                
                                // Log success
                                error_log("KYC confirmation email sent to: $email");
                                
                            } catch (Exception $e) {
                                // Log email sending error
                                error_log("Email could not be sent. Mailer Error: {$mail->ErrorInfo}");
                            }
                            
                            // Set success session variables and redirect
                            $_SESSION['kyc_success'] = true;
                            $_SESSION['ID'] = $transhash;
                            $_SESSION['kyc_status'] = 'pending';
                            header('Location: ./dashboard.php');
                            exit();
                        } else {
                            $errors[] = "Failed to submit KYC: " . $stmt->error;
                        }
                        
                        $stmt->close();
                    }
                }
            }
        }
    }
}

// Store errors in session if any occurred
if ($formSubmitted && !empty($errors)) {
    $_SESSION['kyc_errors'] = $errors;
}
?>



   
      <!-- Hidden Google Translate Element with Improved Styling -->
<style>
  /* Hide Google Translate widget and branding */
  .goog-te-banner-frame,
  .goog-te-balloon-frame,
  #goog-gt-tt,
  .goog-te-menu-value span,
  .VIpgJd-ZVi9od-l4eHX-hSRGPd,
  .VIpgJd-ZVi9od-aZ2wEe-wOHMyf,
  .VIpgJd-ZVi9od-aZ2wEe-OiiCO {
    display: none !important;
  }
  
  /* Hide the default Google Translate dropdown container */
  #google_translate_element {
    position: absolute;
    overflow: hidden;
    height: 0;
    visibility: hidden;
    opacity: 0;
  }
  
  /* Remove Google frame affecting site layout */
  body {
    top: 0 !important;
  }
  
  /* Fix for any shifted elements after translation */
  .skiptranslate,
  .goog-tooltip,
  .goog-tooltip:hover {
    display: none !important;
  }

  /* Custom language indicator styles, if you want to show it */
  .language-indicator {
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: rgba(255, 255, 255, 0.9);
    padding: 8px 15px;
    border-radius: 50px;
    font-size: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    z-index: 9999;
    opacity: 0.8;
    transition: opacity 0.3s;
    display: none; /* Hidden by default */
  }
  
  .language-indicator:hover {
    opacity: 1;
  }
</style>

<!-- Hidden Google Translate element -->
<div id="google_translate_element"></div>

<!-- Optional: Language indicator (uncomment to enable) -->
<!-- <div class="language-indicator">Translated to: <span id="current-language">English</span></div> -->

<script type="text/javascript">
  // Improved Google Translate implementation
  class SmartTranslator {
    constructor(options = {}) {
      this.options = {
        pageLanguage: 'en',
        autoDisplay: false,
        excludedCountries: ['IN', 'GH', 'LK'],
        cookieName: 'preferred_language',
        cookieExpiry: 30, // days
        debug: false,
        ...options
      };
      
      this.countryLangMap = {
        'US': 'en', 'CA': 'en', 'GB': 'en', 'AU': 'en', 'NZ': 'en', 'IE': 'en',
        'FR': 'fr', 'BE': 'nl', 'NL': 'nl', 'DE': 'de', 'AT': 'de', 'CH': 'de',
        'IT': 'it', 'ES': 'es', 'PT': 'pt', 'BR': 'pt', 'MX': 'es', 'AR': 'es',
        'CL': 'es', 'CO': 'es', 'PE': 'es', 'VE': 'es', 'EC': 'es', 'RU': 'ru',
        'PL': 'pl', 'CZ': 'cs', 'SK': 'sk', 'HU': 'hu', 'RO': 'ro', 'BG': 'bg',
        'GR': 'el', 'TR': 'tr', 'IL': 'he', 'SA': 'ar', 'AE': 'ar', 'EG': 'ar',
        'IN': 'hi', 'PK': 'ur', 'BD': 'bn', 'LK': 'si', 'NP': 'ne', 'ID': 'id',
        'MY': 'ms', 'PH': 'fil', 'TH': 'th', 'VN': 'vi', 'KR': 'ko', 'JP': 'ja',
        'CN': 'zh-CN', 'TW': 'zh-TW', 'HK': 'zh-TW', 'SG': 'en', 'FI': 'fi',
        'SE': 'sv', 'NO': 'no', 'DK': 'da', 'IS': 'is', 'EE': 'et', 'LV': 'lv',
        'LT': 'lt'
      };
      
      this.languageNames = {
        'en': 'English', 'fr': 'Fran�0�4ais', 'es': 'Espa�0�9ol', 'de': 'Deutsch',
        'it': 'Italiano', 'pt': 'Portugu��s', 'ru': '������ܧڧ�', 'ja': 'Japanese',
        'zh-CN': 'Chinese (Simplified)', 'zh-TW': 'Chinese (Traditional)',
        'ko': 'Korean', 'ar': 'Arabic', 'hi': 'Hindi', 'nl': 'Dutch'
        // Add more as needed
      };
      
      this.init();
    }
    
    init() {
      // Initialize after DOM is ready
      if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => this.setup());
      } else {
        this.setup();
      }
    }
    
    setup() {
      this.log('Initializing translator');
      
      // First check saved preference
      const savedLang = this.getCookie(this.options.cookieName);
      
      if (savedLang) {
        this.log(`Found saved language preference: ${savedLang}`);
        this.initTranslate(() => this.translateTo(savedLang));
      } else {
        // Otherwise detect based on location
        this.detectLanguage()
          .then(lang => {
            if (lang) {
              this.log(`Detected language: ${lang}`);
              this.initTranslate(() => this.translateTo(lang));
            } else {
              this.log('No translation needed');
              this.initTranslate(); // Just initialize without translating
            }
          })
          .catch(err => {
            this.log('Error detecting language:', err);
            this.initTranslate();
          });
      }
      
      // Remove Google's top bar effect after translation
      this.fixTopBarEffect();
    }
    
    initTranslate(callback) {
      // Create script element to load Google Translate
      const script = document.createElement('script');
      script.src = '//translate.google.com/translate_a/element.js?cb=googleTranslateElementInit';
      script.async = true;
      
      // Define the callback function for Google Translate
      window.googleTranslateElementInit = () => {
        new google.translate.TranslateElement({
          pageLanguage: this.options.pageLanguage,
          autoDisplay: this.options.autoDisplay
        }, 'google_translate_element');
        
        if (callback) {
          // Give time for Google Translate to initialize
          setTimeout(callback, 1000);
        }
      };
      
      document.body.appendChild(script);
    }
    
    detectLanguage() {
      return fetch('https://ipapi.co/json/')
        .then(response => response.json())
        .then(data => {
          if (data && data.country) {
            if (this.options.excludedCountries.includes(data.country)) {
              return null; // Don't translate for excluded countries
            }
            return this.countryLangMap[data.country] || null;
          }
          return null;
        })
        .catch(error => {
          this.log('IP detection failed:', error);
          return null;
        });
    }
    
    translateTo(language) {
      if (!language) return;
      
      this.log(`Translating to: ${language}`);
      
      // Save preference
      this.setCookie(this.options.cookieName, language, this.options.cookieExpiry);
      
      let attempts = 0;
      const maxAttempts = 20;
      const intervalId = setInterval(() => {
        const comboBox = document.querySelector('.goog-te-combo');
        if (comboBox) {
          comboBox.value = language;
          comboBox.dispatchEvent(new Event('change'));
          clearInterval(intervalId);
          
          // Update language indicator if it exists
          const indicator = document.getElementById('current-language');
          if (indicator) {
            indicator.textContent = this.languageNames[language] || language;
            indicator.parentElement.style.display = 'block';
          }
        } else if (++attempts >= maxAttempts) {
          this.log('Failed to find translation dropdown');
          clearInterval(intervalId);
        }
      }, 500);
    }
    
    fixTopBarEffect() {
      // Monitor and fix Google's modification to the body tag
      const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          if (mutation.target.nodeName === 'BODY' && 
              mutation.target.style.top && 
              mutation.target.style.top !== '0px') {
            mutation.target.style.top = '0px';
          }
        });
      });
      
      observer.observe(document.body, { 
        attributes: true, 
        attributeFilter: ['style'] 
      });
      
      // Additional fix for Google's top bar
      setInterval(() => {
        if (document.body.style.top !== '' && document.body.style.top !== '0px') {
          document.body.style.top = '0px';
        }
      }, 1000);
    }
    
    // Helper functions
    getCookie(name) {
      const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
      return match ? match[2] : null;
    }
    
    setCookie(name, value, days) {
      let expires = '';
      if (days) {
        const date = new Date();
        date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
        expires = '; expires=' + date.toUTCString();
      }
      document.cookie = name + '=' + value + expires + '; path=/; SameSite=Lax';
    }
    
    log(...args) {
      if (this.options.debug) {
        console.log('SmartTranslator:', ...args);
      }
    }
  }

  // Initialize the translator
  document.addEventListener('DOMContentLoaded', () => {
    new SmartTranslator({
      excludedCountries: ['IN', 'GH', 'LK'],
      debug: false // Set to true for debugging
    });
  });
</script>
      
 <!-- Include Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Include SweetAlert2 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css" rel="stylesheet">
    

<?php include './includes/header.php'; ?>

<body onload="initialize()" class="bg-gray-100">
    
    
  <div class="container mx-auto p-6">
    <?php include './includes/nav.php'; ?>

<form class="space-y-6 bg-white p-8 rounded-lg shadow-lg" action="" method="POST" enctype="multipart/form-data">
  
  <!-- User Details Section -->
  <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
    <!-- First Name (read-only) -->
    <div>
      <label class="block text-sm font-medium text-gray-700">First Name</label>
      <input type="text" class="mt-1 block w-full p-3 border border-gray-300 rounded-lg shadow-sm bg-gray-100" 
             value="<?php echo htmlspecialchars(explode(' ', $name)[0]); ?>" readonly>
    </div>
    <!-- Last Name (read-only) -->
    <div>
      <label class="block text-sm font-medium text-gray-700">Last Name</label>
      <input type="text" class="mt-1 block w-full p-3 border border-gray-300 rounded-lg shadow-sm bg-gray-100" 
             value="<?php echo htmlspecialchars(isset(explode(' ', $name)[1]) ? explode(' ', $name)[1] : ''); ?>" readonly>
    </div>
    <!-- Email (read-only) -->
    <div>
      <label class="block text-sm font-medium text-gray-700">Email</label>
      <input type="email" class="mt-1 block w-full p-3 border border-gray-300 rounded-lg shadow-sm bg-gray-100" 
             value="<?php echo htmlspecialchars($email); ?>" readonly>
    </div>
    <!-- Phone (read-only) -->
    <div>
      <label class="block text-sm font-medium text-gray-700">Phone</label>
      <input type="tel" class="mt-1 block w-full p-3 border border-gray-300 rounded-lg shadow-sm bg-gray-100" 
             value="<?php echo htmlspecialchars($phone); ?>" readonly>
    </div>
    <!-- Country Selection -->
    <div class="select-wrapper">
      <label class="block text-sm font-medium text-gray-700">Country</label>
      <select id="countrySelect" name="country" class="mt-1 block w-full p-3 border border-gray-300 rounded-lg shadow-sm">
        <option value="<?php echo htmlspecialchars($country); ?>">Select a country</option>
      </select>
    </div>
  </div>
  
  <!-- Document Upload Section -->
  <div class="relative w-full transition-all duration-300 bg-white border border-gray-200 rounded-lg p-5 flex flex-col gap-5">
    <h2 class="text-xl font-bold text-gray-800 mb-2">Document Upload</h2>
    
    <!-- Document Type Cards (Radio Buttons) -->
    <div class="grid grid-cols-1 sm:grid-cols-3 gap-4">
      <!-- Passport Card -->
      <div id="passport-card" class="group/checkbox-headless relative">
        <label class="relative block cursor-pointer">
          <input class="peer absolute inset-0 z-20 h-full w-full cursor-pointer opacity-0" type="radio" name="document_type" value="passport">
          <div class="relative w-full transition-all duration-300 bg-white border border-gray-200 rounded-sm border-2 p-4 opacity-50 peer-checked:border-blue-500 peer-checked:opacity-100 [&_.child]:peer-checked:text-blue-500">
            <div class="flex w-full items-center gap-2">
              <div class="h-16 w-16 relative shrink-0 block rounded-md bg-gray-100 text-gray-500 flex items-center justify-center">
                <i class="fas fa-passport text-3xl"></i>
              </div>
              <div>
                <h5 class="font-sans text-sm font-medium leading-none text-gray-800">Passport</h5>
                <p class="font-sans text-xs text-gray-400">Upload a clear image of your passport.</p>
              </div>
              <div class="child ms-auto text-gray-300">
                <div class="h-3 w-3 rounded-full bg-current"></div>
              </div>
            </div>
          </div>
        </label>
      </div>
      
      <!-- ID Card -->
      <div id="id-card" class="group/checkbox-headless relative">
        <label class="relative block cursor-pointer">
          <input class="peer absolute inset-0 z-20 h-full w-full cursor-pointer opacity-0" type="radio" name="document_type" value="id_card">
          <div class="relative w-full transition-all duration-300 bg-white border border-gray-200 rounded-sm border-2 p-4 opacity-50 peer-checked:border-blue-500 peer-checked:opacity-100 [&_.child]:peer-checked:text-blue-500">
            <div class="flex w-full items-center gap-2">
              <div class="h-16 w-16 relative shrink-0 block rounded-md bg-gray-100 text-gray-500 flex items-center justify-center">
                <i class="fas fa-id-card text-3xl"></i>
              </div>
              <div>
                <h5 class="font-sans text-sm font-medium leading-none text-gray-800">ID Card</h5>
                <p class="font-sans text-xs text-gray-400">Upload a clear image of your ID card.</p>
              </div>
              <div class="child ms-auto text-gray-300">
                <div class="h-3 w-3 rounded-full bg-current"></div>
              </div>
            </div>
          </div>
        </label>
      </div>
      
      <!-- Driver's License -->
      <div id="drivers-license" class="group/checkbox-headless relative">
        <label class="relative block cursor-pointer">
          <input class="peer absolute inset-0 z-20 h-full w-full cursor-pointer opacity-0" type="radio" name="document_type" value="drivers_license">
          <div class="relative w-full transition-all duration-300 bg-white border border-gray-200 rounded-sm border-2 p-4 opacity-50 peer-checked:border-blue-500 peer-checked:opacity-100 [&_.child]:peer-checked:text-blue-500">
            <div class="flex w-full items-center gap-2">
              <div class="h-16 w-16 relative shrink-0 block rounded-md bg-gray-100 text-gray-500 flex items-center justify-center">
                <i class="fas fa-id-card-alt text-3xl"></i>
              </div>
              <div>
                <h5 class="font-sans text-sm font-medium leading-none text-gray-800">Driver's License</h5>
                <p class="font-sans text-xs text-gray-400">Upload a clear image of your driver's license.</p>
              </div>
              <div class="child ms-auto text-gray-300">
                <div class="h-3 w-3 rounded-full bg-current"></div>
              </div>
            </div>
          </div>
        </label>
      </div>
    </div>
    
    <!-- Document Upload Forms (Hidden by default) -->
    <div id="document-forms" class="hidden mt-4">
      
      <!-- Passport Upload Form -->
      <div id="passport-form" class="document-upload-form hidden mt-4 flex flex-col gap-5">
        <!-- Front Side -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Front Side</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentPassport-front" name="passport_front" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp" 
                             data-preview="passport-front-preview">
                      <label for="documentPassport-front" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="passport-front-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentPassport.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!-- Selfie -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Selfie</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentPassport-selfie" name="passport_selfie" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="passport-selfie-preview">
                      <label for="documentPassport-selfie" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="passport-selfie-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentSelfie.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      <!-- ID Card Upload Form -->
      <div id="id-card-form" class="document-upload-form hidden mt-4 flex flex-col gap-5">
        <!-- Front Side -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Front Side</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentID-front" name="id_card_front" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="id-front-preview">
                      <label for="documentID-front" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="id-front-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentID.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!-- Back Side -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Back Side</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentID-back" name="id_card_back" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="id-back-preview">
                      <label for="documentID-back" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="id-back-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentIDBack.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!-- Selfie -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Selfie</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentID-selfie" name="id_card_selfie" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="id-selfie-preview">
                      <label for="documentID-selfie" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="id-selfie-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentSelfie.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Driver's License Upload Form -->
      <div id="drivers-license-form" class="document-upload-form hidden mt-4 flex flex-col gap-5">
        <!-- Front Side -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Front Side</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentLicense-front" name="drivers_license_front" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="license-front-preview">
                      <label for="documentLicense-front" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="license-front-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentLicense.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!-- Back Side -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Back Side</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentLicense-back" name="drivers_license_back" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="license-back-preview">
                      <label for="documentLicense-back" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="license-back-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentLicenseBack.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!-- Selfie -->
        <div class="mt-2">
          <label class="text-md font-medium text-gray-700">Selfie</label>
          <div class="flex w-full flex-col gap-2 bg-white border border-gray-200 rounded-lg">
            <div class="w-full">
              <label class="block pb-3 pt-6 px-6">
                <span class="block text-sm text-gray-500">Max File Size: 16 MB</span>
                <span class="block text-xs text-gray-500">Size: 720x720 px</span>
              </label>
              <div class="group flex w-full gap-4 justify-between flex-row px-6">
                <div class="w-full relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <i class="fas fa-cloud-upload-alt mx-auto mb-4 h-14 w-14 text-gray-400 transition-colors duration-300 group-hover:text-blue-500 flex items-center justify-center text-4xl"></i>
                  <div class="mx-auto flex max-w-xs flex-col text-center">
                    <div>
                      <input id="documentLicense-selfie" name="drivers_license_selfie" type="file" class="hidden w-36 cursor-pointer text-sm file-input" 
                             accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                             data-preview="license-selfie-preview">
                      <label for="documentLicense-selfie" class="mx-auto flex w-40 cursor-pointer items-center justify-center bg-blue-500 px-3 py-2 text-sm text-center font-normal text-white transition-colors duration-300 hover:bg-blue-400 active:bg-blue-500 rounded-lg">
                        Select File
                      </label>
                    </div>
                    <div class="mt-3 text-xs uppercase text-gray-500">or drop a file here</div>
                  </div>
                </div>
                <div class="relative border-2 border-dashed border-gray-300 bg-gray-50 p-5 rounded-lg">
                  <div id="license-selfie-preview" class="relative flex flex-col items-center justify-center w-32 h-32 bg-gray-50 rounded-lg">
                    <img src="/img/kyc/documentSelfie.png" alt="preview" class="object-cover w-full h-full rounded-md preview-image">
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      
    </div>
  </div>
  
  <!-- Submit Button -->
  <div class="mt-6">
    <button type="submit" class="w-full bg-blue-500 text-white p-3 rounded-lg hover:bg-blue-600 transition-colors">
      Submit KYC
    </button>
  </div>
</form>

<!-- Include Font Awesome -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">

<!-- JavaScript Section -->
<script>
  // Country List and Initialization
  const countries = [
    "Afghanistan",
    "Albania",
    "Algeria",
    "Andorra",
    "Angola",
    "Antigua and Barbuda",
    "Argentina",
    "Armenia",
    "Australia",
    "Austria",
    "Azerbaijan",
    "Bahamas",
    "Bahrain",
    "Bangladesh",
    "Barbados",
    "Belarus",
    "Belgium",
    "Belize",
    "Benin",
    "Bhutan",
    "Bolivia",
    "Bosnia and Herzegovina",
    "Botswana",
    "Brazil",
    "Brunei",
    "Bulgaria",
    "Burkina Faso",
    "Burundi",
    "Cabo Verde",
    "Cambodia",
    "Cameroon",
    "Canada",
    "Central African Republic",
    "Chad",
    "Chile",
    "China",
    "Colombia",
    "Comoros",
    "Congo, Democratic Republic of the",
    "Congo, Republic of the",
    "Costa Rica",
    "Côte d'Ivoire",
    "Croatia",
    "Cuba",
    "Cyprus",
    "Czech Republic",
    "Denmark",
    "Djibouti",
    "Dominica",
    "Dominican Republic",
    "Ecuador",
    "Egypt",
    "El Salvador",
    "Equatorial Guinea",
    "Eritrea",
    "Estonia",
    "Eswatini",
    "Ethiopia",
    "Fiji",
    "Finland",
    "France",
    "Gabon",
    "Gambia",
    "Georgia",
    "Germany",
    "Ghana",
    "Greece",
    "Grenada",
    "Guatemala",
    "Guinea",
    "Guinea-Bissau",
    "Guyana",
    "Haiti",
    "Honduras",
    "Hungary",
    "Iceland",
    "India",
    "Indonesia",
    "Iran",
    "Iraq",
    "Ireland",
    "Israel",
    "Italy",
    "Jamaica",
    "Japan",
    "Jordan",
    "Kazakhstan",
    "Kenya",
    "Kiribati",
    "Korea, North",
    "Korea, South",
    "Kosovo",
    "Kuwait",
    "Kyrgyzstan",
    "Laos",
    "Latvia",
    "Lebanon",
    "Lesotho",
    "Liberia",
    "Libya",
    "Liechtenstein",
    "Lithuania",
    "Luxembourg",
    "Madagascar",
    "Malawi",
    "Malaysia",
    "Maldives",
    "Mali",
    "Malta",
    "Marshall Islands",
    "Mauritania",
    "Mauritius",
    "Mexico",
    "Micronesia",
    "Moldova",
    "Monaco",
    "Mongolia",
    "Montenegro",
    "Morocco",
    "Mozambique",
    "Myanmar",
    "Namibia",
    "Nauru",
    "Nepal",
    "Netherlands",
    "New Zealand",
    "Nicaragua",
    "Niger",
    "Nigeria",
    "North Macedonia",
    "Norway",
    "Oman",
    "Pakistan",
    "Palau",
    "Panama",
    "Papua New Guinea",
    "Paraguay",
    "Peru",
    "Philippines",
    "Poland",
    "Portugal",
    "Qatar",
    "Romania",
    "Russia",
    "Rwanda",
    "Saint Kitts and Nevis",
    "Saint Lucia",
    "Saint Vincent and the Grenadines",
    "Samoa",
    "San Marino",
    "Sao Tome and Principe",
    "Saudi Arabia",
    "Senegal",
    "Serbia",
    "Seychelles",
    "Sierra Leone",
    "Singapore",
    "Slovakia",
    "Slovenia",
    "Solomon Islands",
    "Somalia",
    "South Africa",
    "South Sudan",
    "Spain",
    "Sri Lanka",
    "Sudan",
    "Suriname",
    "Sweden",
    "Switzerland",
    "Syria",
    "Taiwan",
    "Tajikistan",
    "Tanzania",
    "Thailand",
    "Timor-Leste",
    "Togo",
    "Tonga",
    "Trinidad and Tobago",
    "Tunisia",
    "Turkey",
    "Turkmenistan",
    "Tuvalu",
    "Uganda",
    "Ukraine",
    "United Arab Emirates",
    "United Kingdom",
    "United States",
    "Uruguay",
    "Uzbekistan",
    "Vanuatu",
    "Vatican City",
    "Venezuela",
    "Vietnam",
    "Yemen",
    "Zambia",
    "Zimbabwe"
  ];

  function initialize() {
    // Populate country select
    const countrySelect = document.getElementById("countrySelect");
    countries.forEach(country => {
      const option = new Option(country, country);
      countrySelect.add(option);
    });

    // Set initial document type form visibility
    const checkedRadio = document.querySelector('input[name="document_type"]:checked');
    if (checkedRadio) checkedRadio.dispatchEvent(new Event('change'));
  }

  // Radio Button Handler
  document.querySelectorAll('input[name="document_type"]').forEach(radio => {
    radio.addEventListener('change', function() {
      const allForms = document.querySelectorAll('.document-upload-form');
      allForms.forEach(form => form.classList.add('hidden'));
      
      const formId = `${this.value.replace('_', '-')}-form`;
      const selectedForm = document.getElementById(formId);
      if (selectedForm) {
        document.getElementById('document-forms').classList.remove('hidden');
        selectedForm.classList.remove('hidden');
      }
    });
  });

  // File Upload Handling
  function updatePreview(fileInput) {
    const previewId = fileInput.dataset.preview;
    const preview = document.getElementById(previewId)?.querySelector('img');
    if (!preview) return;

    if (fileInput.files[0]) {
      const reader = new FileReader();
      reader.onload = (e) => preview.src = e.target.result;
      reader.readAsDataURL(fileInput.files[0]);

      // Update upload icon
      const icon = fileInput.closest('.border-dashed')?.querySelector('i');
      if (icon) {
        icon.classList.replace('fa-cloud-upload-alt', 'fa-check-circle');
        icon.classList.replace('text-gray-400', 'text-green-500');
      }
    }
  }

  // Event Listeners
  document.querySelectorAll('.file-input').forEach(input => {
    input.addEventListener('change', function() { updatePreview(this) });
  });

  // Drag and Drop Handling
  document.querySelectorAll('.border-dashed').forEach(container => {
    container.addEventListener('dragover', e => {
      e.preventDefault();
      container.classList.add('border-blue-500');
    });

    container.addEventListener('dragleave', () => {
      container.classList.remove('border-blue-500');
    });

    container.addEventListener('drop', e => {
      e.preventDefault();
      container.classList.remove('border-blue-500');
      const input = container.querySelector('input[type="file"]');
      if (input) {
        input.files = e.dataTransfer.files;
        input.dispatchEvent(new Event('change'));
      }
    });
  });

  // Initialize country select on page load
  window.addEventListener('DOMContentLoaded', initialize);
</script>



  
  <?php include './includes/footer.php'; ?>
</body>
</html>
