
<?php
/**
 * User Profile Management System
 * Enhanced version with improved security, error handling, and code organization
 */

// Security Configuration
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(0);

// Enable error logging for debugging
ini_set('log_errors', 1);
ini_set('error_log', '../logs/error.log');

// Start session with secure settings
if (session_status() === PHP_SESSION_NONE) {
    session_start([
        'cookie_httponly' => true,
        'cookie_secure' => isset($_SERVER['HTTPS']),
        'cookie_samesite' => 'Strict'
    ]);
}

/**
 * Configuration and Dependencies
 */
require_once 'db.php';
require_once '../session.php';

/**
 * Constants and Configuration
 */
class Config {
    const ALLOWED_STATUSES = ['active', 'suspend', 'pending'];
    const PHONE_MIN_LENGTH = 10;
    const PHONE_MAX_LENGTH = 15;
    const REDIRECT_MAPPINGS = [
        'COT' => 'cot.php',
        'insure' => 'insurance.php',
        'suspend' => 'suspend.php',
        'verify' => 'verify.php'
    ];
}

/**
 * Country Data Management
 */
class CountryManager {
    private static $allCountries = [
        ["name" => "Afghanistan", "code" => "af", "dial_code" => "+93"],
        // ... [rest of countries array remains the same] ...
        ["name" => "Zimbabwe", "code" => "zw", "dial_code" => "+263"]
    ];

    /**
     * Get all countries data
     */
    public static function getAllCountries() {
        return self::$allCountries;
    }

    /**
     * Get country by dial code
     */
    public static function getCountryByDialCode($dialCode) {
        foreach (self::$allCountries as $country) {
            if ($country['dial_code'] === $dialCode) {
                return $country;
            }
        }
        return null;
    }

    /**
     * Get country by country code
     */
    public static function getCountryByCode($code) {
        foreach (self::$allCountries as $country) {
            if ($country['code'] === $code) {
                return $country;
            }
        }
        return null;
    }

    /**
     * Get country by name
     */
    public static function getCountryByName($name) {
        foreach (self::$allCountries as $country) {
            if (strcasecmp($country['name'], $name) === 0) {
                return $country;
            }
        }
        return null;
    }

    /**
     * Validate dial code
     */
    public static function isValidDialCode($dialCode) {
        return self::getCountryByDialCode($dialCode) !== null;
    }
}

/**
 * Security and Validation Helper
 */
class SecurityHelper {
    /**
     * Sanitize input data
     */
    public static function sanitizeInput($input) {
        return trim(htmlspecialchars($input, ENT_QUOTES, 'UTF-8'));
    }

    /**
     * Validate email format
     */
    public static function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Validate phone number
     */
    public static function validatePhone($phone) {
        $cleanPhone = preg_replace('/[^0-9+\-\s]/', '', $phone);
        $phoneLength = strlen(preg_replace('/[^0-9]/', '', $cleanPhone));
        
        return $phoneLength >= Config::PHONE_MIN_LENGTH && 
               $phoneLength <= Config::PHONE_MAX_LENGTH;
    }

    /**
     * Generate CSRF token
     */
    public static function generateCSRFToken() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    /**
     * Validate CSRF token
     */
    public static function validateCSRFToken($token) {
        return isset($_SESSION['csrf_token']) && 
               hash_equals($_SESSION['csrf_token'], $token);
    }
}

/**
 * User Management Class
 */
class UserManager {
    private $conn;

    public function __construct($connection) {
        $this->conn = $connection;
    }

    /**
     * Check user restrictions and redirect if necessary
     */
    public function checkUserRestrictions($email) {
        try {
            // FIX: Added extra column to SELECT statement
            $stmt = mysqli_prepare($this->conn, 
                "SELECT status, reset_token, extra FROM users WHERE email = ? LIMIT 1");
            
            if (!$stmt) {
                throw new Exception("Database prepare failed: " . mysqli_error($this->conn));
            }

            mysqli_stmt_bind_param($stmt, "s", $email);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);

            if ($result && mysqli_num_rows($result) > 0) {
                $row = mysqli_fetch_assoc($result);
                mysqli_stmt_close($stmt);
                
                // Check for inactive Extra - FIX: Now properly checks extra field
                if ($row['extra'] === 'suspend') {
                    $this->redirectToPage('suspend.php');
                }
                
                // Check for specific restrictions
                if (!empty($row['reset_token']) && 
                    isset(Config::REDIRECT_MAPPINGS[$row['reset_token']])) {
                    $this->redirectToPage(Config::REDIRECT_MAPPINGS[$row['reset_token']]);
                }
                
                return $row;
            }
            
            mysqli_stmt_close($stmt);
            return null;
            
        } catch (Exception $e) {
            error_log("User restriction check failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Update user profile
     */
    public function updateUserProfile($googleId, $country, $phone) {
        try {
            // Validate inputs
            if (empty($country) || empty($phone)) {
                throw new Exception("Country and phone number are required");
            }

            if (!SecurityHelper::validatePhone($phone)) {
                throw new Exception("Invalid phone number format");
            }

            // Prepare update query
            $stmt = mysqli_prepare($this->conn, 
                "UPDATE users SET country = ?, phone = ?, updated_at = NOW() WHERE google_id = ?");
            
            if (!$stmt) {
                throw new Exception("Database prepare failed: " . mysqli_error($this->conn));
            }

            mysqli_stmt_bind_param($stmt, "sss", $country, $phone, $googleId);
            
            if (!mysqli_stmt_execute($stmt)) {
                throw new Exception("Failed to update profile: " . mysqli_stmt_error($stmt));
            }

            $affected = mysqli_stmt_affected_rows($stmt);
            mysqli_stmt_close($stmt);

            if ($affected === 0) {
                throw new Exception("No rows updated. User may not exist.");
            }

            return true;
            
        } catch (Exception $e) {
            error_log("Profile update failed: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get country statistics
     */
    public function getCountryStatistics() {
        try {
            $query = "SELECT country, COUNT(*) as customers 
                     FROM users 
                     WHERE country IS NOT NULL AND country != '' 
                     GROUP BY country 
                     ORDER BY customers DESC";
            
            $result = mysqli_query($this->conn, $query);
            
            if (!$result) {
                throw new Exception("Query failed: " . mysqli_error($this->conn));
            }

            $dbCountries = [];
            $totalCustomers = 0;

            while ($row = mysqli_fetch_assoc($result)) {
                $dbCountries[$row['country']] = (int)$row['customers'];
                $totalCustomers += (int)$row['customers'];
            }

            // Merge with all countries data
            $countriesMap = [];
            foreach (CountryManager::getAllCountries() as $country) {
                $custCount = $dbCountries[$country['name']] ?? 0;
                $percentage = $totalCustomers > 0 ? 
                    round(($custCount / $totalCustomers) * 100, 2) : 0;
                
                $countriesMap[] = array_merge($country, [
                    'customers' => $custCount,
                    'percentage' => $percentage
                ]);
            }

            return [
                'countries' => $countriesMap,
                'total_customers' => $totalCustomers
            ];
            
        } catch (Exception $e) {
            error_log("Country statistics failed: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Safe redirect function
     */
    private function redirectToPage($page) {
        $allowedPages = [
            'suspend.php', 'cot.php', 'insurance.php', 'verify.php'
        ];
        
        if (in_array($page, $allowedPages)) {
            header("Location: $page");
            exit();
        }
    }
}

/**
 * Main Application Logic
 */
try {
    // Authentication check
    if (!isLoggedIn()) {
        header('Location: ../login.php');
        exit();
    }

    // Initialize user manager
    $userManager = new UserManager($conn);

    // Get user details
    $userDetails = getUserDetails($conn);
    if ($userDetails === null) {
        throw new Exception('User session expired or invalid');
    }

    // Extract user data
    $userData = [
        'google_id' => $userDetails['google_id'] ?? '',
        'name' => $userDetails['name'] ?? '',
        'email' => $userDetails['email'] ?? '',
        'picture' => $userDetails['picture'] ?? '',
        'country' => $userDetails['country'] ?? '',
        'phone' => $userDetails['phone'] ?? '',
        'status' => $userDetails['status'] ?? '',
        'balance' => $userDetails['balance'] ?? 0,
        'percent' => $userDetails['percent'] ?? 0,
        'created_at' => $userDetails['created_at'] ?? '',
        'email_verified' => $userDetails['email_verified'] ?? false,
        'pin' => $userDetails['pin'] ?? '',
        'btc' => $userDetails['btc'] ?? 0,
        'eth' => $userDetails['eth'] ?? 0,
        'usdt' => $userDetails['usdt'] ?? 0,
        'xrp' => $userDetails['xrp'] ?? 0,
        'doge' => $userDetails['doge'] ?? 0,
        'ada' => $userDetails['ada'] ?? 0,
        'pepe' => $userDetails['pepe'] ?? 0,
        'sol' => $userDetails['sol'] ?? 0,
        'bnb' => $userDetails['bnb'] ?? 0,
        'message' => $userDetails['message'] ?? '',
        'extra' => $userDetails['extra'] ?? '', // Fixed: Only one 'extra' key
        'msgstatus' => $userDetails['msgstatus'] ?? '',
        'bonus' => $userDetails['bonus'] ?? 0,
        'profit' => $userDetails['profit'] ?? 0,
        'update_time' => $userDetails['update_time'] ?? ''
    ];

    // Check user restrictions
    if (!empty($userData['email'])) {
        $userManager->checkUserRestrictions($userData['email']);
    }

    // Check if profile needs completion
    $showProfileAlert = empty($userData['country']) || empty($userData['phone']);

    // Get country statistics
    $countryStats = $userManager->getCountryStatistics();
    $countriesMap = $countryStats['countries'];
    $totalCustomers = $countryStats['total_customers'];

    // Initialize message variables
    $updateMessage = '';
    $updateError = '';

    // Process form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
        try {
            // CSRF protection
            if (!SecurityHelper::validateCSRFToken($_POST['csrf_token'] ?? '')) {
                throw new Exception('Invalid security token. Please try again.');
            }

            // Sanitize input
            $newDialCode = SecurityHelper::sanitizeInput($_POST['country'] ?? '');
            $newPhone = SecurityHelper::sanitizeInput($_POST['full_phone'] ?? '');

            // Validate inputs
            if (empty($newDialCode) || empty($newPhone)) {
                throw new Exception('Both country and phone number are required.');
            }

            // Validate dial code
            if (!CountryManager::isValidDialCode($newDialCode)) {
                throw new Exception('Invalid country selected.');
            }

            // Get country name from dial code
            $countryData = CountryManager::getCountryByDialCode($newDialCode);
            $newCountry = $countryData['name'];

            // Update user profile
            $userManager->updateUserProfile($userData['google_id'], $newCountry, $newPhone);

            // Update local variables
            $userData['country'] = $newCountry;
            $userData['phone'] = $newPhone;
            $showProfileAlert = false;

            $updateMessage = 'Profile updated successfully!';

        } catch (Exception $e) {
            $updateError = $e->getMessage();
            error_log("Profile update error: " . $e->getMessage());
        }
    }

    // Generate CSRF token for form
    $csrfToken = SecurityHelper::generateCSRFToken();

    // Prepare JSON data for frontend
    $jsonCountries = json_encode($countriesMap, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP);

} catch (Exception $e) {
    error_log("Application error: " . $e->getMessage());
    die('An error occurred. Please try again later.');
}

/**
 * Helper Functions (Legacy support)
 */
function getCountryCode($countryName) {
    $country = CountryManager::getCountryByName($countryName);
    return $country ? $country['code'] : '';
}

// Legacy variable assignments for backward compatibility
$google_id = $userData['google_id'];
$name = $userData['name'];
$email = $userData['email'];
$picture = $userData['picture'];
$country = $userData['country'];
$phone = $userData['phone'];
$status = $userData['status'];
$extra = $userData['extra']; // Fixed variable name
$balance = $userData['balance'];
$percent = $userData['percent'];
$created_at = $userData['created_at'];
$email_verified = $userData['email_verified'];
$pin = $userData['pin'];
$btc = $userData['btc'];
$eth = $userData['eth'];
$usdt = $userData['usdt'];
$xrp = $userData['xrp'];
$doge = $userData['doge'];
$ada = $userData['ada'];
$pepe = $userData['pepe'];
$sol = $userData['sol'];
$bnb = $userData['bnb'];
$message = $userData['message'];
$msgstatus = $userData['msgstatus'];
$bonus = $userData['bonus'];
$profit = $userData['profit'];
$update_time = $userData['update_time'];

// Legacy country data
$allCountries = CountryManager::getAllCountries();

?>



<!DOCTYPE html>
<html lang="en">
<?php include './includes/header.php'; ?>

<body onload="initialize()" style="overflow-x: hidden;">

  <?php include './includes/nav.php'; ?>



  <?php
if (isset($_SESSION['success_alert'])) {
    echo '<div id="sessionAlert" class="bg-green-100 text-green-800 border border-green-400 p-4 rounded shadow text-center">';
    echo $_SESSION['success_alert'];
    echo '</div>';
    unset($_SESSION['success_alert']);
}
?>
  <script>
    // After 3 seconds, fade out the alert smoothly over 1 second
    setTimeout(function () {
      var alertEl = document.getElementById("sessionAlert");
      if (alertEl) {
        alertEl.classList.add("transition-opacity", "duration-1000", "opacity-0");
        // Remove the element from the DOM after the fade-out transition completes
        setTimeout(function () {
          alertEl.remove();
        }, 1000);
      }
    }, 3000);
  </script>





  <?php
// Retrieve the user email from session
$user_email = getSessionEmail();
if (!$user_email) {
    echo '<div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded relative" role="alert">
            <strong class="font-bold">Not Logged In</strong>
            <span class="block sm:inline"> Please log in to access your dashboard.</span>
          </div>';
    exit;
}

// Fetch user details (msgstatus, extra)
$query = "SELECT msgstatus, message FROM users WHERE email = :email";
$stmt = $conn->prepare($query);
$stmt->bindParam(':email', $user_email, PDO::PARAM_STR);
$stmt->execute();
$user = $stmt->fetch(PDO::FETCH_ASSOC);
$show_modal = false;
$popup_message = '';

if ($user) {
    $msgstatus = strtolower(trim($user['msgstatus'] ?? ''));
    $message = trim($user['message'] ?? '');
    
    // Show popup only if msgstatus is "yes"
    if ($msgstatus === 'yes') {
        $show_modal = true;
        $popup_message = $message; // Fetching 'message' value from the database
    }
}

// Display appropriate SweetAlert based on verification status
if (isset($_SESSION['kyc_status'])) {
    $status = $_SESSION['kyc_status'];
    
    // Clear the session variable after using it
    unset($_SESSION['kyc_status']);
}
?>



  <!-- JavaScript to show SweetAlert -->
 <script>
  document.addEventListener('DOMContentLoaded', function () {
    var showModal = <?php echo json_encode($show_modal); ?>;
    var popupMessage = <?php echo json_encode($popup_message); ?>;

    if (showModal) {
      Swal.fire({
        title: 'NOTICE!',
        html: popupMessage,
        icon: 'warning',
        confirmButtonText: 'OK',
        background: '#343a40',
        backdrop: 'rgba(0, 0, 0, 0.7)',
        customClass: {
          title: 'swal-dark-title',
          htmlContainer: 'swal-dark-content',
          confirmButton: 'swal-dark-button',
          popup: 'swal-dark-popup'
        }
      });
    }
  });
</script>


  <?php


// Check if KYC submission was successful
if (isset($_SESSION['kyc_success']) && $_SESSION['kyc_success']) {
    $transhash = $_SESSION['transhash'];
    // Clear the session variable
    unset($_SESSION['kyc_success']);
    unset($_SESSION['transhash']);
}
?>



  <?php if (isset($transhash)): ?>
  <script>
    // Show SweetAlert notification
    Swal.fire({
      icon: 'success',
      title: 'KYC Submitted Successfully!',
      text: 'Your transaction hash is: <?php echo $transhash; ?>',
      confirmButtonText: 'OK'
    });
  </script>
  <?php endif; ?>



  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/flag-icons/6.6.6/css/flag-icons.min.css">
 



  <hr>
  &nbsp;
  <!-- Portfolio Balance Section -->

  
  
  
  <!-- Trade Modal -->
  <div id="tradeModal"
    style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(255, 255, 255, 0.1); backdrop-filter: blur(10px); z-index: 9999;">
    <div
      style="background: rgba(255, 255, 255, 0.95); backdrop-filter: blur(20px); margin: 8% auto; padding: 40px 30px; width: 85%; max-width: 480px; border-radius: 24px; position: relative; box-shadow: 0 25px 50px rgba(0, 0, 0, 0.1); border: 1px solid rgba(255, 255, 255, 0.2);">
      <!-- Close button -->
      <span
        style="position: absolute; top: 20px; right: 25px; cursor: pointer; font-size: 24px; color: #6b7280; width: 32px; height: 32px; display: flex; align-items: center; justify-content: center; border-radius: 50%; transition: all 0.2s ease;"
        onclick="document.getElementById('tradeModal').style.display='none';"
        onmouseover="this.style.background='rgba(107, 114, 128, 0.1)'; this.style.transform='scale(1.1)'"
        onmouseout="this.style.background='transparent'; this.style.transform='scale(1)'">&times;</span>

      <!-- Modal Header -->
      <div style="margin-bottom: 30px; text-align: center;">
        <h2
          style="font-size: 28px; font-weight: 700; color: #1f2937; margin: 0; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); -webkit-background-clip: text; -webkit-text-fill-color: transparent; background-clip: text;">
          Choose Trade Type</h2>
        <p style="font-size: 16px; color: #6b7280; margin: 8px 0 0 0; font-weight: 400;">Select your preferred trading
          market</p>
      </div>

      <!-- Trade Options -->
      <div style="display: flex; flex-direction: column; gap: 16px;">
        <a href="forex.php"
          style="display: block; padding: 20px 24px; text-align: center; background: linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%); border: 2px solid rgba(5, 150, 105, 0.2); border-radius: 16px; text-decoration: none; color: #059669; font-weight: 600; font-size: 18px; transition: all 0.3s ease; position: relative; overflow: hidden;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 15px 35px rgba(0, 0, 0, 0.1)'; this.style.borderColor='#059669'; this.style.background='linear-gradient(135deg, rgba(5, 150, 105, 0.1) 0%, rgba(5, 150, 105, 0.05) 100%)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'; this.style.borderColor='rgba(5, 150, 105, 0.2)'; this.style.background='linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%)'">
          <span style="display: inline-block; margin-right: 12px; font-size: 20px;">💱</span>
          Forex Trading
        </a>
        <a href="forex.php"
          style="display: block; padding: 20px 24px; text-align: center; background: linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%); border: 2px solid rgba(220, 38, 38, 0.2); border-radius: 16px; text-decoration: none; color: #dc2626; font-weight: 600; font-size: 18px; transition: all 0.3s ease; position: relative; overflow: hidden;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 15px 35px rgba(0, 0, 0, 0.1)'; this.style.borderColor='#dc2626'; this.style.background='linear-gradient(135deg, rgba(220, 38, 38, 0.1) 0%, rgba(220, 38, 38, 0.05) 100%)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'; this.style.borderColor='rgba(220, 38, 38, 0.2)'; this.style.background='linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%)'">
          <span style="display: inline-block; margin-right: 12px; font-size: 20px;">📈</span>
          Stock Market
        </a>
        <a href="crypto.php"
          style="display: block; padding: 20px 24px; text-align: center; background: linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%); border: 2px solid rgba(37, 99, 235, 0.2); border-radius: 16px; text-decoration: none; color: #2563eb; font-weight: 600; font-size: 18px; transition: all 0.3s ease; position: relative; overflow: hidden;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 15px 35px rgba(0, 0, 0, 0.1)'; this.style.borderColor='#2563eb'; this.style.background='linear-gradient(135deg, rgba(37, 99, 235, 0.1) 0%, rgba(37, 99, 235, 0.05) 100%)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'; this.style.borderColor='rgba(37, 99, 235, 0.2)'; this.style.background='linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%)'">
          <span style="display: inline-block; margin-right: 12px; font-size: 20px;">₿</span>
          Cryptocurrency
        </a>
        <a href="copytrade.php"
          style="display: block; padding: 20px 24px; text-align: center; background: linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%); border: 2px solid rgba(37, 99, 235, 0.2); border-radius: 16px; text-decoration: none; color: #2563eb; font-weight: 600; font-size: 18px; transition: all 0.3s ease; position: relative; overflow: hidden;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 15px 35px rgba(0, 0, 0, 0.1)'; this.style.borderColor='#2563eb'; this.style.background='linear-gradient(135deg, rgba(37, 99, 235, 0.1) 0%, rgba(37, 99, 235, 0.05) 100%)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'; this.style.borderColor='rgba(37, 99, 235, 0.2)'; this.style.background='linear-gradient(135deg, rgba(255, 255, 255, 0.8) 0%, rgba(255, 255, 255, 0.4) 100%)'">
          <span style="display: inline-block; vertical-align: middle; margin-right: 12px; font-size: 20px;">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.8"
              stroke="currentColor" style="width: 24px; height: 24px;">
              <path stroke-linecap="round" stroke-linejoin="round"
                d="M9 12l2 2 4-4m2-2.5V5a2 2 0 00-2-2h-1.5a1.5 1.5 0 01-3 0H8a2 2 0 00-2 2v2.5M5 9v10a2 2 0 002 2h10a2 2 0 002-2V9H5z" />
            </svg>
          </span>
          Copy Trading
        </a>

      </div>
    </div>
  </div>


<section style="margin-top: 34px; padding: 0 16px;">

  <!-- Responsive grid: stacked by default, 3-columns on desktop -->
  <style>
    @media (min-width: 1024px) {
      .desktop-grid { display: grid !important; grid-template-columns: repeat(3, 1fr) !important; }
    }
    @keyframes boil   { 0%,100%{transform:scale(1);opacity:.6} 50%{transform:scale(1.2);opacity:.3} }
    @keyframes boil2  { 0%,100%{transform:scale(1);opacity:.5} 50%{transform:scale(1.3);opacity:.2} }
    @keyframes boil3  { 0%,100%{transform:scale(1);opacity:.3} 50%{transform:scale(1.4);opacity:.1} }
    @keyframes blink  { 0%,100%{opacity:1} 50%{opacity:.2} }
  </style>

  <div class="desktop-grid" style="display: block; gap: 16px; max-width: 1200px; margin: 0 auto;">

    <!-- Profile Card -->
    <div style="background:white; border-radius:16px; padding:16px; box-shadow:0 3px 15px rgba(0,0,0,0.06); position:relative; overflow:hidden;">
      <div id="cover-section" style="position:relative; height:100px; border-top-left-radius:16px; border-top-right-radius:16px; overflow:hidden; background-size:cover; background-position:center;">
        <div style="position:absolute; inset:0; background:linear-gradient(135deg, rgba(0,0,0,0.4), rgba(0,0,0,0.6));"></div>
        <div style="position:absolute; bottom:8px; left:12px; right:12px; text-align:center; color:#fff;">
          <div id="greeting" style="font-size:16px; font-weight:600; text-shadow:0 1px 3px rgba(0,0,0,0.3); margin-bottom:8px;"></div>
          <div id="quote" style="font-size:12px; font-weight:500; line-height:1.4; font-style:italic; text-shadow:0 1px 2px rgba(0,0,0,0.4);"></div>
        </div>
      </div>
      <div style="padding:14px;">
        <div style="display:flex; align-items:center; gap:10px; position:relative;">
          <div style="position:relative; width:48px; height:48px; margin-top:-32px;">
            <div style="position:absolute; top:-4px; left:-4px; width:56px; height:56px; border-radius:50%; background:rgba(0,200,83,0.2); animation:boil 2s infinite ease-in-out;"></div>
            <div style="position:absolute; top:-4px; left:-4px; width:56px; height:56px; border-radius:50%; background:rgba(0,200,83,0.3); animation:boil2 2.5s infinite ease-in-out;"></div>
            <div style="position:absolute; top:-4px; left:-4px; width:56px; height:56px; border-radius:50%; background:rgba(0,200,83,0.15); animation:boil3 3s infinite ease-in-out;"></div>
            <img src="<?= htmlspecialchars($picture ?? 'https://primemarketspro.com/session/faviconb.png') ?>" alt="Profile" style="width:48px; height:48px; border-radius:50%; position:relative; z-index:2; object-fit:cover;">
            <div style="position:absolute; bottom:1px; right:1px; width:8px; height:8px; background:#00c853; border-radius:50%; z-index:3; animation:blink 1s infinite;"></div>
          </div>
          <div style="flex-grow:1;">
            <div style="display:flex; align-items:center; gap:5px;">
<h2 style="margin:0; font-size:15px; font-weight:bold; color:#111;">
  <?= htmlspecialchars($name ?? 'name') ?>
</h2>
            <?php
require('db.php'); // This file sets up the PDO connection in $conn and includes your session functions

// Retrieve the user email from session using your helper function
$user_email = getSessionEmail();
if (!$user_email) {
    // Don't show anything if not logged in
    exit;
}

// Prepare the query using PDO with a named parameter
$query = "SELECT * FROM kyc_submissions WHERE email = :email";
$stmt = $conn->prepare($query);
$stmt->bindParam(':email', $user_email, PDO::PARAM_STR);
$stmt->execute();

// Fetch the user data as an associative array
$user = $stmt->fetch(PDO::FETCH_ASSOC);

// Only show verified badge if status is "verified"
if ($user && $user['status'] == 'verified') {
    echo '
    <span style="display:inline-block;" 
          oncontextmenu="return false;" 
          ondragstart="return false;" 
          onmousedown="return false;">
        <img src="https://upload.wikimedia.org/wikipedia/commons/e/e4/Twitter_Verified_Badge.svg" 
             alt="Verified" 
             style="width:14px; height:14px; pointer-events: none; user-select: none;">
    </span>';
}
// If not verified or no user found, show nothing
?>
        </div>
           <p style="margin:0; font-size:12px;">
  <a href="mailto:<?= htmlspecialchars($email ?? 'user123@example.com') ?>"
     style="color:#1e90ff; text-decoration:none;">
    <?= htmlspecialchars($email ?? 'user123@example.com') ?>
  </a>
</p>

          </div>
        </div>
        <div style="display:flex; justify-content:space-between; align-items:center; margin-top:10px; font-size:12px; color:#333;">
          <span>ID: <?= htmlspecialchars($google_id ?? 'USER123') ?></span>
         <!-- HTML -->
<div id="locationSection" style="display:flex; align-items:center; gap:5px;">
  <img
    src="https://flagcdn.com/us.svg"
    alt="Flag"
    style="width:18px; height:12px; border-radius:2px;"
  >
  <span id="countryName">Loading...</span>
<?php
$status_value = htmlspecialchars($status ?? 'Active');

switch (strtolower($status_value)) {
    case 'active':
        $bg = '#d1fae5';      // green background
        $color = '#065f46';   // dark green text
        break;
    case 'pending':
        $bg = '#fef9c3';      // yellow background
        $color = '#92400e';   // dark yellow text
        break;
    default:
        $bg = '#fee2e2';      // red background
        $color = '#991b1b';   // dark red text
        break;
}
?>
<div style="display:flex; align-items:center; gap:3px; margin-left:6px; background:<?= $bg ?>; color:<?= $color ?>; padding:4px 10px; border-radius:12px; font-size:13px; font-weight:600;">
    <?= $status_value ?>
</div>
</div>

<script>
  async function fetchUserLocation() {
    // Try ipapi.co first
    try {
      const res = await fetch('https://ipapi.co/json/', { mode: 'cors' });
      if (!res.ok) throw new Error(`ipapi.co ${res.status}`);
      const data = await res.json();
      if (data.country_name && data.country_code) {
        document.getElementById('countryName').textContent = data.country_name;
        // select the <img> inside #locationSection
        const img = document.querySelector('#locationSection img');
        img.src = `https://flagcdn.com/${data.country_code.toLowerCase()}.svg`;
        img.alt = `${data.country_name} Flag`;
        return;
      }
      throw new Error('Incomplete data from ipapi.co');
    } catch (e) {
      console.warn('ipapi.co failed:', e);
    }

    // Fallback to geolocation-db.com
    try {
      const res2 = await fetch('https://geolocation-db.com/json/', { mode: 'cors' });
      if (!res2.ok) throw new Error(`geolocation-db ${res2.status}`);
      const data2 = await res2.json();
      if (data2.country_name && data2.country_code) {
        document.getElementById('countryName').textContent = data2.country_name;
        const img2 = document.querySelector('#locationSection img');
        img2.src = `https://flagcdn.com/${data2.country_code.toLowerCase()}.svg`;
        img2.alt = `${data2.country_name} Flag`;
        return;
      }
      throw new Error('Incomplete data from geolocation-db');
    } catch (e) {
      console.error('Fallback failed:', e);
    }

    // Final fallback
    document.getElementById('countryName').textContent = 'United States';
    const imgFallback = document.querySelector('#locationSection img');
    imgFallback.src = 'https://flagcdn.com/us.svg';
    imgFallback.alt = 'US Flag';
  }

  // Kick it off
  fetchUserLocation();

  // If you also want to update the badge here:
  const verificationStatuses = ['verified','unverified','pending'];
  function updateVerificationBadge() {
    const s = verificationStatuses[Math.floor(Math.random()*3)];
    const badges = {
      verified:   { icon: '✔️', text: 'Verified',   color: '#22c55e' },
      unverified: { icon: '✖️', text: 'Unverified', color: '#ef4444' },
      pending:    { icon: '⏳', text: 'Pending',    color: '#f59e0b' }
    };
    const b = badges[s];
    const badgeEl = document.getElementById('verificationBadge');
    badgeEl.innerHTML = `<span style="color:${b.color}; font-size:10px; font-weight:600;">${b.icon} ${b.text}</span>`;
  }

  // Example: call once
  updateVerificationBadge();
</script>



        </div>
       <?php
// Database configuration
$host     = 'localhost';
$dbname   = 'wptlgmnb_dbp';
$username = 'wptlgmnb_userp';
$password = 'Qliox4ITygj6Vh4V';
$charset  = 'utf8mb4';
$dsn      = "mysql:host=$host;dbname=$dbname;charset=$charset";
$options  = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES   => false,
];

try {
    $pdo = new PDO($dsn, $username, $password, $options);
    $stmt    = $pdo->query("SELECT id, name, image_url FROM copy_traders ORDER BY id DESC LIMIT 9");
    $traders = $stmt->fetchAll();
    $activeCount = count($traders);
} catch (Exception $e) {
    die("DB Error: " . $e->getMessage());
}

// Fallback SVG generator
function svgFallback($initial) {
    $svg = '<svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">'
         . '<circle cx="20" cy="20" r="20" fill="#4F46E5"/>'
         . '<text x="20" y="26" text-anchor="middle" fill="white" font-size="14" font-family="Arial" font-weight="bold">'
         . htmlspecialchars($initial)
         . '</text></svg>';
    return 'data:image/svg+xml;base64,' . base64_encode($svg);
}

$basePath = __DIR__ . '/admin/copy_trader_images/';
$webPath  = '/session/auth/user/admin/copy_trader_images/';
$formats  = ['png','jpg','jpeg','gif','webp','svg'];
?>

<div style="margin-top:16px; padding-top:12px; border-top:1px solid #e2e8f0;">
  <div style="display:flex; align-items:center; justify-content:space-between; margin-bottom:8px;">
    <span style="font-size:12px; color:#64748b; font-weight:500;">Copy Traders</span>
    <span style="font-size:11px; color:#94a3b8;"><?php echo $activeCount; ?> active</span>
  </div>
  <div style="display:flex; align-items:center; gap:-8px;">
   <?php
    $zIndex = $activeCount;
    foreach (array_slice($traders, 0, 5) as $idx => $t) {
        $cname    = htmlspecialchars($t['name']);
        $rawImg  = trim($t['image_url']);
        $initial = strtoupper(substr($cname, 0, 1));
        $src     = '';
        
        // Direct match
        if ($rawImg !== '') {
            $path = $basePath . $rawImg;
            if (file_exists($path)) {
                $src = $webPath . $rawImg;
            }
        }
        // Extension loop
        if (!$src && $rawImg !== '') {
            $baseName = pathinfo($rawImg, PATHINFO_FILENAME);
            foreach ($formats as $ext) {
                $file = $basePath . $baseName . '.' . $ext;
                if (file_exists($file)) {
                    $src = $webPath . $baseName . '.' . $ext;
                    break;
                }
            }
        }
        // Fallback
        if (!$src) {
            $src = svgFallback($initial);
        }
        ?>
        <img src="<?php echo $src; ?>"
             alt="<?php echo $cname; ?>"
             style="width:32px; height:32px; border-radius:50%; border:2px solid #fff; object-fit:cover;
                    margin-left:<?php echo $idx > 0 ? '-8px' : '0'; ?>; z-index:<?php echo $zIndex--; ?>;">
    <?php }
    if ($activeCount > 5): ?>
        <div style="width:32px; height:32px; border-radius:50%; background:#22c55e; border:2px solid #fff; display:flex; align-items:center; justify-content:center; margin-left:-8px; z-index:1;">
            <span style="font-size:11px; font-weight:600; color:#fff;">+<?php echo $activeCount - 5; ?></span>
        </div>
    <?php endif; ?>
    
  </div>
  
  
             <div class="progress-container" id="progressContainer">
            <!-- Progress bars will be generated by JavaScript -->
        </div>
</div>

 <div class="signal-info">
            <div class="signal-label">
                <i class="fas fa-bolt"></i> Active Copy traders Signal
            </div>
            <div id="percentDisplay" class="signal-percent" style="color: #10b981;">78%</div>
        </div>
        
         
          
           
<hr style="
  border: none;
  height: 4px;
  background: linear-gradient(to right, #ef4444, #10b981);
  border-radius: 4px;
  margin: 16px 0;
">
        
     
     
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
      
        .trade-card {
            max-width: 720px;
            width: 100%;
            margin: 10px auto;
            border-radius: 16px;
            padding: 32px;
           
           
            position: relative;
            overflow: hidden;
        }
        
       .trade-card {
    position: relative;
    border-radius: 12px; /* Makes the corners round */
    overflow: hidden; /* Ensures child elements don’t overflow the rounded edges */

    /* Drop shadow */
    box-shadow:
        0 4px 8px rgba(0, 0, 0, 0.15), /* soft outer shadow */
        inset 0 1px 0 rgba(255, 255, 255, 0.4), /* inner top light */
        inset 0 -1px 0 rgba(0, 0, 0, 0.1); /* inner bottom dark (emboss effect) */

    background: #fff; /* Optional: for a clean contrast */
    padding: 16px; /* Optional: inner spacing */
    z-index: 1;
}

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 28px;
            position: relative;
            z-index: 2;
        }
        
        .title {
            font-size: 20px;
            font-weight: 700;
            color: #111827;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .date {
            font-size: 14px;
            color: #4b5563;
            background: rgba(229, 231, 235, 0.6);
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 500;
        }
        
        .signal-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 18px;
        }
        
        .signal-label {
            font-size: 15px;
            color: #4b5563;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .signal-percent {
            font-weight: 800;
            font-size: 18px;
            transition: all 0.3s ease;
        }
        
        .progress-container {
            display: flex;
            justify-content: center;
            align-items: flex-end;
            gap: 4px;
            height: 48px;
            margin-top: 8px;
        }
        
        .progress-bar {
            width: 6px;
            border-radius: 3px 3px 0 0;
            transition: all 0.4s ease;
        }
        
        .stats-container {
            display: flex;
            justify-content: space-between;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid rgba(229, 231, 235, 0.8);
        }
        
        .stat-box {
            text-align: center;
            padding: 12px;
            background: rgba(241, 245, 249, 0.6);
            border-radius: 12px;
            flex: 1;
            margin: 0 6px;
            box-shadow: inset 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .stat-value {
            font-size: 20px;
            font-weight: 700;
            margin-bottom: 4px;
        }
        
        .stat-label {
            font-size: 13px;
            color: #4b5563;
        }
        
        .confidence-meter {
            margin-top: 24px;
            background: rgba(241, 245, 249, 0.8);
            padding: 16px;
            border-radius: 12px;
            position: relative;
        }
        
        .confidence-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 12px;
        }
        
        .confidence-title {
            font-weight: 600;
            color: #1f2937;
        }
        
        .confidence-value {
            font-weight: 700;
            color: #10b981;
        }
        
        .meter-bar {
            height: 10px;
            background: #e5e7eb;
            border-radius: 5px;
            overflow: hidden;
            position: relative;
        }
        
        .meter-fill {
            height: 100%;
            width: 75%;
            background: linear-gradient(90deg, #3b82f6, #60a5fa);
            border-radius: 5px;
        }
        
        .action-buttons {
            display: flex;
            gap: 12px;
            margin-top: 24px;
        }
        
        .btn {
            flex: 1;
            padding: 14px;
            border-radius: 12px;
            font-weight: 600;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .btn-primary {
            background: linear-gradient(to right, #3b82f6, #2563eb);
            color: white;
            box-shadow: 0 4px 6px rgba(37, 99, 235, 0.2);
        }
        
        .btn-secondary {
            background: white;
            color: #4b5563;
            border: 1px solid #e5e7eb;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .btn:hover {
            transform: translateY(-2px);
        }
        
        .btn:active {
            transform: translateY(1px);
        }
        
        .strength-indicator {
            display: flex;
            justify-content: space-between;
            margin-top: 24px;
            padding: 0 8px;
        }
        
        .indicator-label {
            font-size: 12px;
            color: #6b7280;
            position: relative;
            text-align: center;
            flex: 1;
        }
        
        .indicator-label::before {
            content: '';
            position: absolute;
            top: -8px;
            left: 50%;
            transform: translateX(-50%);
            width: 6px;
            height: 6px;
            border-radius: 50%;
        }
        
        .indicator-label.low::before { background: #ef4444; }
        .indicator-label.medium::before { background: #f59e0b; }
        .indicator-label.high::before { background: #10b981; }
        
        .signal-icon {
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 10px;
            color: white;
            font-size: 18px;
        }
    </style>

  
  <?php
$percent = isset($percent) ? (float)$percent : 0;

// Determine the signal level and colors
if ($percent <= 10) {
    $signalLevel = 'Low';
    $barColor = '#ef4444'; // Red
    $gradient = 'linear-gradient(270deg, #dc2626, #ef4444, #f87171)';
    $textColor = '#ef4444';
} elseif ($percent <= 50) {
    $signalLevel = 'Medium';
    $barColor = '#f59e0b'; // Amber/Warm
    $gradient = 'linear-gradient(270deg, #f59e0b, #fbbf24, #fde68a)';
    $textColor = '#f59e0b';
} else {
    $signalLevel = 'High';
    $barColor = '#10b981'; // Green
    $gradient = 'linear-gradient(270deg, #10b981, #34d399, #6ee7b7)';
    $textColor = '#10b981';
}
?>

<div class="trade-card" style="max-width: 420px; margin: auto; padding: 12px; background: ; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); font-family: Arial, sans-serif; color: #1f2937;">
    
    <!-- Strength Indicator -->
    <div class="strength-indicator" style="display: flex; justify-content: space-between; font-size: 12px; margin-bottom: 10px;">
        <div class="indicator-label low" style="color: #ef4444;">Low (1-10%)</div>
        <div class="indicator-label medium" style="color: #f59e0b;">Medium (10-50%)</div>
        <div class="indicator-label high" style="color: #10b981;">High (50-100%)</div>
    </div>

    <!-- Confidence Meter -->
    <div class="confidence-meter" style="width: 100%; background: #ffffff11; border-radius: 10px; padding: 10px; box-shadow: inset 0 0 6px rgba(0,0,0,0.05);">
        <div class="confidence-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
            <div class="confidence-title" style="font-weight: bold; font-size: 14px; display: flex; align-items: center;">
                <i class="fas fa-shield-alt" style="margin-right: 6px; color: #1e90ff;"></i> 
                <span style="color:#1e90ff;">Trade Signal Level</span>
            </div>
            <div class="confidence-value" style="font-size: 13px; font-weight: bold; color: <?= $textColor ?>;">
                <?= $signalLevel ?>
            </div>
        </div>

        <!-- Meter Bar -->
        <div class="meter-bar" style="width: 100%; height: 10px; background: #f0f0f0; border-radius: 20px; overflow: hidden; position: relative;">
            <div class="meter-fill" style="
                width: <?= $percent ?>%;
                height: 100%;
                background: <?= $gradient ?>;
                background-size: 400% 400%;
                animation: shimmer 2s infinite linear;
                border-radius: 20px;">
            </div>
        </div>

        <!-- Percentage Text -->
        <div style="
            margin-top: 6px;
            text-align: right;
            font-weight: bold;
            font-size: 13px;
            color: <?= $percent > 0 ? $barColor : '#9ca3af' ?>;
            background: <?= $percent > 0 ? $gradient : 'none' ?>;
            -webkit-background-clip: <?= $percent > 0 ? 'text' : 'none' ?>;
            -webkit-text-fill-color: <?= $percent > 0 ? 'transparent' : 'inherit' ?>;
            animation: <?= $percent > 0 ? 'shimmerText 2s infinite linear' : 'none' ?>;">
            <?= $percent > 0 ? htmlspecialchars($percent) . '%' : 'No Signal' ?>
        </div>
         <!-- Action Buttons -->
<div style="display: flex; gap: 12px; margin-top: 16px;">
  <a href="signalbuy.php" style="text-decoration: none;">
    <button style="
      background-color: #3b82f6;
      color: white;
      border: none;
      padding: 6px 14px;
      font-size: 13px;
      font-weight: 500;
      border-radius: 6px;
      display: flex;
      align-items: center;
      gap: 6px;
      box-shadow: 0 1px 2px rgba(0,0,0,0.05);
      cursor: pointer;
      transition: background-color 0.3s ease;
    ">
      <i class="fas fa-play-circle"></i> Boost Signal
    </button>
  </a>

  <a href="analytic.php" style="text-decoration: none;">
    <button style="
      background-color: #6b7280;
      color: white;
      border: none;
      padding: 6px 14px;
      font-size: 13px;
      font-weight: 500;
      border-radius: 6px;
      display: flex;
      align-items: center;
      gap: 6px;
      box-shadow: 0 1px 2px rgba(0,0,0,0.05);
      cursor: pointer;
      transition: background-color 0.3s ease;
    ">
      <i class="fas fa-chart-bar"></i> View Analysis
    </button>
  </a>
</div>

    </div>
</div>

<!-- Shimmer Keyframes -->
<style>
@keyframes shimmer {
    0% {
        background-position: 0% 50%;
    }
    100% {
        background-position: 100% 50%;
    }
}
@keyframes shimmerText {
    0% {
        background-position: 0% 50%;
    }
    100% {
        background-position: 100% 50%;
    }
}
</style>

        
       
    </div>
    
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
    const progressContainer = document.getElementById('progressContainer');
    const percentDisplay = document.getElementById('percentDisplay');
    
    // Initialize with 78%
    let currentPercent = 78;
    updateProgress(currentPercent);
    
    // Function to update progress display
    function updateProgress(percent) {
        // Clear container
        progressContainer.innerHTML = '';
        
        // Update percent display with trading text (smaller font)
        if (percent <= 50) {
            percentDisplay.innerHTML = '<span style="font-size: 0.6em;">PLACE TRADE</span><br>' + percent + '%';
        } else {
            percentDisplay.innerHTML = '<span style="font-size: 0.6em;">TP</span><br>' + percent + '%';
        }
        
        // Determine color based on percent (saturated colors)
        let color;
        if (percent <= 50) {
            color = '#dc2626'; // Saturated red
        } else {
            color = '#16a34a'; // Saturated green
        }
        percentDisplay.style.color = color;
        
        // Generate candlesticks
        const totalCandlesticks = 25;
        const filledCandlesticks = Math.ceil((percent / 100) * totalCandlesticks);
        
        for (let i = 0; i < totalCandlesticks; i++) {
            const candlestick = document.createElement('div');
            candlestick.style.position = 'relative';
            candlestick.style.width = '8px';
            candlestick.style.display = 'flex';
            candlestick.style.flexDirection = 'column';
            candlestick.style.alignItems = 'center';
            candlestick.style.justifyContent = 'end';
            candlestick.style.animation = 'fadeInUp 0.8s ease-out forwards';
            candlestick.style.opacity = '0';
            
            // Create upper wick (thin line above body)
            const upperWick = document.createElement('div');
            upperWick.style.width = '1px';
            upperWick.style.marginBottom = '0px';
            const upperWickHeight = 3 + Math.floor(Math.random() * 12);
            upperWick.style.height = upperWickHeight + 'px';
            
            // Create body (thick rectangle)
            const body = document.createElement('div');
            body.style.width = '8px';
            body.style.borderRadius = '0px'; // Sharp edges
            body.style.transition = 'all 0.3s ease';
            const bodyHeight = 8 + Math.floor(Math.random() * 25);
            body.style.height = bodyHeight + 'px';
            
            // Create lower wick (thin line below body)
            const lowerWick = document.createElement('div');
            lowerWick.style.width = '1px';
            lowerWick.style.marginTop = '0px';
            const lowerWickHeight = 3 + Math.floor(Math.random() * 12);
            lowerWick.style.height = lowerWickHeight + 'px';
            
            if (i < filledCandlesticks) {
                // Alternate between red and green candlesticks
                const isGreen = i % 2 === 0;
                
                if (isGreen) {
                    // Saturated green candlestick
                    body.style.background = '#16a34a';
                    body.style.border = '1px solid #15803d';
                    body.style.boxShadow = '0 0 8px rgba(22, 163, 74, 0.4)';
                    upperWick.style.background = '#16a34a';
                    lowerWick.style.background = '#16a34a';
                } else {
                    // Saturated red candlestick
                    body.style.background = '#dc2626';
                    body.style.border = '1px solid #b91c1c';
                    body.style.boxShadow = '0 0 8px rgba(220, 38, 38, 0.4)';
                    upperWick.style.background = '#dc2626';
                    lowerWick.style.background = '#dc2626';
                }
                
                // Add animation delay for progressive appearance
                candlestick.style.animationDelay = (i * 0.05) + 's';
            } else {
                // Unfilled candlesticks are red and green too
                const isGreen = i % 2 === 0;
                
                if (isGreen) {
                    body.style.background = '#16a34a';
                    body.style.border = '1px solid #15803d';
                    body.style.boxShadow = '0 0 8px rgba(22, 163, 74, 0.4)';
                    upperWick.style.background = '#16a34a';
                    lowerWick.style.background = '#16a34a';
                } else {
                    body.style.background = '#dc2626';
                    body.style.border = '1px solid #b91c1c';
                    body.style.boxShadow = '0 0 8px rgba(220, 38, 38, 0.4)';
                    upperWick.style.background = '#dc2626';
                    lowerWick.style.background = '#dc2626';
                }
                candlestick.style.opacity = '0.3';
            }
            
            candlestick.appendChild(upperWick);
            candlestick.appendChild(body);
            candlestick.appendChild(lowerWick);
            progressContainer.appendChild(candlestick);
        }
    }
    
    // Demo functionality to change percentage
    let demoInterval;
    
    function startDemo() {
        stopDemo(); // Clear any existing interval
        
        let demoPercent = 10;
        let increasing = true;
        
        demoInterval = setInterval(() => {
            if (increasing) {
                demoPercent += 8;
                if (demoPercent >= 100) increasing = false;
            } else {
                demoPercent -= 8;
                if (demoPercent <= 10) increasing = true;
            }
            
            updateProgress(demoPercent);
        }, 1200);
    }
    
    function stopDemo() {
        if (demoInterval) {
            clearInterval(demoInterval);
            demoInterval = null;
        }
    }
    
    // Auto-start demo
    startDemo();
    
    // Make functions globally accessible
    window.startDemo = startDemo;
    window.stopDemo = stopDemo;
    window.updateProgress = updateProgress;
});
    </script>
    

</div>

<!-- Inline CSS Animation -->
<style>
@keyframes pulse {
  0% { transform: scale(1); opacity: 1; }
  50% { transform: scale(1.2); opacity: 0.6; }
  100% { transform: scale(1); opacity: 1; }
}
</style>

      </div>
    </div>

<div id="dailyImageDiv" style="border-radius:16px; padding:16px; overflow:hidden; position:relative;">
  <!-- Portfolio Statistics Card -->
  <div >
    <div style="position:absolute; top:16px; right:16px; text-align:right;">
      <div style="font-size:11px; color:#718096; margin-bottom:3px;">24h Change</div>
      <div id="portfolioChange" style="font-size:14px; font-weight:600;">-0.87%</div>
    </div>
    
    
     <?php
// Get the user email from the already included session
$user_email = $email;

// Get current total portfolio value
$query = "SELECT SUM(btc + eth + usdt + xrp + doge + ada + pepe + profit + bonus + sol + bnb) AS total FROM users WHERE email = :email";
$stmt = $conn->prepare($query);
$stmt->bindParam(':email', $user_email);
$stmt->execute();
$result = $stmt->fetch(PDO::FETCH_ASSOC);
$currentTotal = $result['total'] ?? 0;

// Check if portfolio_snapshots table exists, if not create it
$createTableQuery = "CREATE TABLE IF NOT EXISTS portfolio_snapshots (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_email VARCHAR(255) NOT NULL,
    portfolio_value DECIMAL(15,2) NOT NULL,
    snapshot_date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_user_date (user_email, snapshot_date)
)";
$conn->exec($createTableQuery);

// Get today's date
$today = date('Y-m-d');

// Check if we have a snapshot for today
$snapshotQuery = "SELECT portfolio_value FROM portfolio_snapshots WHERE user_email = :email AND snapshot_date = :today";
$snapshotStmt = $conn->prepare($snapshotQuery);
$snapshotStmt->bindParam(':email', $user_email);
$snapshotStmt->bindParam(':today', $today);
$snapshotStmt->execute();
$todaySnapshot = $snapshotStmt->fetch(PDO::FETCH_ASSOC);

// Get yesterday's snapshot for comparison
$yesterday = date('Y-m-d', strtotime('-1 day'));
$yesterdayQuery = "SELECT portfolio_value FROM portfolio_snapshots WHERE user_email = :email AND snapshot_date = :yesterday";
$yesterdayStmt = $conn->prepare($yesterdayQuery);
$yesterdayStmt->bindParam(':email', $user_email);
$yesterdayStmt->bindParam(':yesterday', $yesterday);
$yesterdayStmt->execute();
$yesterdaySnapshot = $yesterdayStmt->fetch(PDO::FETCH_ASSOC);

// If no snapshot exists for today, create one
if (!$todaySnapshot) {
    $insertQuery = "INSERT INTO portfolio_snapshots (user_email, portfolio_value, snapshot_date) 
                    VALUES (:email, :value, :date) 
                    ON DUPLICATE KEY UPDATE portfolio_value = :value";
    $insertStmt = $conn->prepare($insertQuery);
    $insertStmt->bindParam(':email', $user_email);
    $insertStmt->bindParam(':value', $currentTotal);
    $insertStmt->bindParam(':date', $today);
    $insertStmt->execute();
    
    $startOfDayValue = $currentTotal;
} else {
    $startOfDayValue = $todaySnapshot['portfolio_value'];
}

// Calculate daily change
$dailyChange = 0;
$dailyChangePercent = 0;
$changeColor = '#12bb61'; // Green by default
$changeIcon = 'fa-arrow-up';

if ($yesterdaySnapshot) {
    $previousValue = $yesterdaySnapshot['portfolio_value'];
    $dailyChange = $currentTotal - $previousValue;
    
    if ($previousValue > 0) {
        $dailyChangePercent = ($dailyChange / $previousValue) * 100;
    }
    
    // Set color and icon based on change
    if ($dailyChange < 0) {
        $changeColor = '#ef4444'; // Red
        $changeIcon = 'fa-arrow-down';
    } elseif ($dailyChange == 0) {
        $changeColor = '#6b7280'; // Gray
        $changeIcon = 'fa-minus';
    }
} else {
    // If no previous data, show as neutral
    $changeColor = '#6b7280';
    $changeIcon = 'fa-minus';
    $dailyChange = 0;
    $dailyChangePercent = 0;
}

// Format values for display
$formattedTotal = '$' . number_format($currentTotal, 2);
$formattedChange = '$' . number_format(abs($dailyChange), 2);
$formattedPercent = number_format(abs($dailyChangePercent), 2);

// Update today's snapshot with current value (in case it changed during the day)
$updateQuery = "UPDATE portfolio_snapshots SET portfolio_value = :value WHERE user_email = :email AND snapshot_date = :date";
$updateStmt = $conn->prepare($updateQuery);
$updateStmt->bindParam(':value', $currentTotal);
$updateStmt->bindParam(':email', $user_email);
$updateStmt->bindParam(':date', $today);
$updateStmt->execute();
?>


     <!-- Portfolio Balance & Toggle Button -->
              <div style="display: flex; align-items: center; gap: 10px; margin-top: 5px;">
    <!-- Balance Display -->
    <h1 id="portfolioBalance" style="font-size: 30px; font-weight: 800; color: #F1E3FF; margin: 0;">
        <?= $formattedTotal ?>
    </h1>
    
    <!-- Eye Toggle Button -->
    <button id="toggleBalance" style="background-color: #eef2f7; border: none; border-radius: 8px; padding: 8px; cursor: pointer;">
        <svg id="eyeIcon" xmlns="http://www.w3.org/2000/svg" style="height: 1.25rem; width: 1.25rem;" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                  d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                  d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 
                     9.542 7-1.274 4.057-5.065 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
        </svg>
    </button>
</div>

<script>
document.addEventListener("DOMContentLoaded", function() {
    // Portfolio balance toggle functionality
    let isBalanceVisible = true;
    const originalBalance = <?= json_encode($formattedTotal) ?>;

    const balanceEl = document.getElementById("portfolioBalance");
    const toggleBtn = document.getElementById("toggleBalance");
    const eyeIcon = document.getElementById("eyeIcon");

    toggleBtn.addEventListener("click", function() {
        if (isBalanceVisible) {
            balanceEl.textContent = "••••••••";
            eyeIcon.innerHTML = `
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                      d="M13.875 18.825A10.05 10.05 0 0112 19
                         c-4.477 0-8.268-2.943-9.542-7
                         a9.959 9.959 0 012.21-3.458" />
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                      d="M4.598 4.598l15 15" />
            `;
            isBalanceVisible = false;
        } else {
            balanceEl.textContent = originalBalance;
            eyeIcon.innerHTML = `
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                      d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                      d="M2.458 12C3.732 7.943 7.523 5 12 5
                         c4.477 0 8.268 2.943 9.542 7
                         -1.274 4.057-5.065 7-9.542 7
                         -4.477 0-8.268-2.943-9.542-7z" />
            `;
            isBalanceVisible = true;
        }
    });

    // Logout functionality
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', function(e) {
            e.preventDefault();

            Swal.fire({
                title: 'Logout Confirmation',
                text: 'Are you sure you want to logout from your trading account?',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#dc2626',
                cancelButtonColor: '#64748b',
                confirmButtonText: 'Yes, Logout',
                cancelButtonText: 'Cancel',
                customClass: {
                    popup: 'swal-popup-custom'
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    Swal.fire({
                        title: 'Logged Out',
                        text: 'You have been successfully logged out.',
                        icon: 'success',
                        timer: 1500,
                        showConfirmButton: false,
                        customClass: {
                            popup: 'swal-popup-custom'
                        }
                    }).then(() => {
                        // Redirect to logout.php
                        window.location.href = 'logout.php';
                    });
                }
            });
        });
    }

    // Send modal function
    window.openSendModal = function() {
        Swal.fire({
            title: 'Send Assets',
            html: `
                <style>
                    .crypto-grid {
                        display: grid;
                        grid-template-columns: repeat(4, 1fr);
                        gap: 10px;
                        margin-top: 15px;
                    }

                    .crypto-item {
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        text-decoration: none;
                        color: #333;
                        background: #f4f4f4;
                        padding: 6px;
                        border-radius: 10px;
                        transition: all 0.2s ease-in-out;
                    }

                    .crypto-item:hover {
                        background: #ddd;
                        transform: translateY(-1px);
                    }

                    .crypto-item img {
                        width: 34px;
                        height: 34px;
                        object-fit: cover;
                        border-radius: 50%;
                        margin-bottom: 4px;
                    }

                    .crypto-item span {
                        font-size: 10px;
                        font-weight: 500;
                        text-align: center;
                    }

                    .swal2-popup.swal-popup-custom {
                        padding: 20px 15px 15px;
                    }

                    @media (max-width: 500px) {
                        .crypto-grid {
                            grid-template-columns: repeat(3, 1fr);
                            gap: 8px;
                        }

                        .crypto-item img {
                            width: 30px;
                            height: 30px;
                        }

                        .crypto-item span {
                            font-size: 9px;
                        }
                    }
                </style>

                <div class="crypto-grid">
                    <a href="bitcoins.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/bitcoin.png" alt="Bitcoin">
                        <span>Bitcoin</span>
                    </a>
                    <a href="eths.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/eth.png" alt="Ethereum">
                        <span>Ethereum</span>
                    </a>
                    <a href="usdtd.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/usdt.png" alt="USDT">
                        <span>USDT</span>
                    </a>
                    <a href="xrps.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/xrp.png" alt="XRP">
                        <span>XRP</span>
                    </a>
                    <a href="bnbs.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/bnb.png" alt="BNB">
                        <span>BNB</span>
                    </a>
                    <a href="sols.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/sol.png" alt="Solana">
                        <span>Solana</span>
                    </a>
                    <a href="doges.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/doge.png" alt="Dogecoin">
                        <span>Dogecoin</span>
                    </a>
                    <a href="adas.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/ada.png" alt="Cardano">
                        <span>Cardano</span>
                    </a>
                </div>
            `,
            showCancelButton: true,
            confirmButtonText: 'Continue',
            cancelButtonText: 'Close',
            customClass: {
                popup: 'swal-popup-custom'
            },
            width: '600px'
        });
    };

    // Receive modal function
    window.openReceiveModal = function() {
        Swal.fire({
            title: 'Receive Assets',
            html: `
                <style>
                    .crypto-grid {
                        display: grid;
                        grid-template-columns: repeat(4, 1fr);
                        gap: 10px;
                        margin-top: 15px;
                    }

                    .crypto-item {
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        text-decoration: none;
                        color: #222;
                        background: #f9f9f9;
                        padding: 6px;
                        border-radius: 10px;
                        transition: all 0.25s ease;
                        box-shadow: 0 1px 2px rgba(0, 0, 0, 0.06);
                    }

                    .crypto-item:hover {
                        background: #e6e6e6;
                        transform: translateY(-2px);
                    }

                    .crypto-item img {
                        width: 30px;
                        height: 30px;
                        object-fit: cover;
                        border-radius: 8px;
                        margin-bottom: 4px;
                    }

                    .crypto-item span {
                        font-size: 10px;
                        font-weight: 500;
                        text-align: center;
                    }

                    .swal2-popup.swal-popup-custom {
                        padding: 20px 12px 12px;
                    }

                    @media (max-width: 500px) {
                        .crypto-grid {
                            grid-template-columns: repeat(3, 1fr);
                        }

                        .crypto-item img {
                            width: 26px;
                            height: 26px;
                        }

                        .crypto-item span {
                            font-size: 9px;
                        }
                    }
                </style>

                <div class="crypto-grid">
                    <a href="bitcoinr.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/bitcoin.png" alt="Bitcoin">
                        <span>Bitcoin</span>
                    </a>
                    <a href="ethr.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/eth.png" alt="Ethereum">
                        <span>Ethereum</span>
                    </a>
                    <a href="usdtr.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/usdt.png" alt="USDT">
                        <span>USDT</span>
                    </a>
                    <a href="xrpr.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/xrp.png" alt="XRP">
                        <span>XRP</span>
                    </a>
                    <a href="bnbr.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/bnb.png" alt="BNB">
                        <span>BNB</span>
                    </a>
                    <a href="solr.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/sol.png" alt="Solana">
                        <span>Solana</span>
                    </a>
                    <a href="doger.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/doge.png" alt="Dogecoin">
                        <span>Dogecoin</span>
                    </a>
                    <a href="adar.php" class="crypto-item">
                        <img src="https://primemarketspro.com/cryptos/ada.png" alt="Cardano">
                        <span>Cardano</span>
                    </a>
                </div>
            `,
            showCancelButton: true,
            confirmButtonText: 'Continue',
            cancelButtonText: 'Close',
            customClass: {
                popup: 'swal-popup-custom'
            },
            width: '600px'
        });
    };

    // Scroll to stake section
    window.scrollToStakeSection = function() {
        const stakeSection = document.getElementById('stakeSection');
        if (stakeSection) {
            stakeSection.scrollIntoView({ behavior: 'smooth' });
        } else {
            Swal.fire({
                title: 'Staking Coming Soon',
                text: 'Our staking features will be available soon!',
                icon: 'info',
                customClass: {
                    popup: 'swal-popup-custom'
                }
            });
        }
    };
});
</script>

            

              <!-- Add this style block once in your page or in your CSS file -->
              <style>
                /* Pulsing green dot animation */
                @keyframes pulse {
                  0% {
                    transform: scale(1);
                    opacity: 1;
                  }

                  50% {
                    transform: scale(1.3);
                    opacity: 0.7;
                  }

                  100% {
                    transform: scale(1);
                    opacity: 1;
                  }
                }

                .pulsing-dot {
                  display: inline-block;
                  width: 10px;
                  height: 10px;
                  border-radius: 50%;
                  background-color: #28a745;
                  /* solid green */
                  animation: pulse 1.5s infinite ease-in-out;
                  margin-right: 8px;
                }

                .daily-performance {
                  font-size: 14px;
                  color: white;
                  /* fixed white color */
                  font-weight: 500;
                  display: flex;
                  align-items: center;
                }

                .daily-performance i {
                  margin-right: 6px;
                }
              </style>

              <!-- Daily Performance -->
              <div class="daily-performance">
                <!-- Pulsing green dot -->
                <span class="pulsing-dot"></span>

                <i class="fas <?= $changeIcon ?>"></i>

                <span>
                  <?php if ($dailyChange >= 0): ?>
                  +
                  <?= $formattedPercent ?>% (+
                  <?= $formattedChange ?>) Today
                  <?php else: ?>
                  -
                  <?= $formattedPercent ?>% (-
                  <?= $formattedChange ?>) Today
                  <?php endif; ?>
                </span>
              </div>



<?php
// PHP: get profit and total deposit
$user_email = $email;
$query = "SELECT SUM(btc + eth + usdt + xrp + doge + ada + pepe + sol + bnb) AS total FROM users WHERE email = :email"; 
$stmt = $conn->prepare($query);
$stmt->bindParam(':email', $user_email);
$stmt->execute();
$result = $stmt->fetch(PDO::FETCH_ASSOC);
$currentTotal = $result['total'] ?? 0;

// Generate more realistic data points for curves
$profitPoints = [];
$depositPoints = [];
$timeLabels = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

// Create curved progression for profit (with ups and downs)
$finalProfit = $profit;
$finalDeposit = $currentTotal;

for ($i = 0; $i < 12; $i++) {
    // Profit with realistic fluctuations
    $baseProgressProfit = ($finalProfit / 11) * $i;
    $fluctuation = sin($i * 0.8) * ($finalProfit * 0.15); // Add sine wave for natural curves
    $randomVariation = (rand(-100, 100) / 100) * ($finalProfit * 0.08); // Add some randomness
    $profitPoints[] = max(0, round($baseProgressProfit + $fluctuation + $randomVariation));
    
    // Deposit with more steady growth but some variations
    $baseProgressDeposit = ($finalDeposit / 11) * $i;
    $depositFluctuation = cos($i * 0.6) * ($finalDeposit * 0.05); // Smaller fluctuations for deposits
    $depositPoints[] = max(0, round($baseProgressDeposit + $depositFluctuation));
}

// Ensure final values are correct
$profitPoints[11] = $finalProfit;
$depositPoints[11] = $finalDeposit;
?>

<style>
    /* Style wrapper to control canvas height */
    #chartWrapper {
        position: relative;
        width: 100%;
        height: 300px; /* default height */
    }

    /* On desktop screens, reduce height */
    @media screen and (min-width: 992px) {
        #chartWrapper {
            height: 180px; /* reduced height for desktop */
        }
    }

    canvas#profitDepositChart {
        width: 100% !important;
        height: 100% !important;
    }
</style>

<!-- Wrapper with responsive height -->
<div id="chartWrapper">
    <canvas id="profitDepositChart"></canvas>
</div>

<!-- Chart.js library -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    const ctx = document.getElementById('profitDepositChart').getContext('2d');
    
    const profitGradient = ctx.createLinearGradient(0, 0, 0, 300);
    profitGradient.addColorStop(0, 'rgba(0, 255, 0, 0.4)');
    profitGradient.addColorStop(1, 'rgba(0, 255, 0, 0)');
    
    const depositGradient = ctx.createLinearGradient(0, 0, 0, 300);
    depositGradient.addColorStop(0, 'rgba(255, 0, 0, 0.4)');
    depositGradient.addColorStop(1, 'rgba(255, 0, 0, 0)');
    
    const chart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: <?= json_encode($timeLabels) ?>,
            datasets: [
                {
                    label: 'Accumulative Profit',
                    data: <?= json_encode($profitPoints) ?>,
                    fill: true,
                    backgroundColor: profitGradient,
                    borderColor: 'green',
                    borderWidth: 2,
                    tension: 0.4, // This creates the curved lines
                    pointBackgroundColor: 'green',
                    pointBorderColor: 'green',
                    pointRadius: 3
                },
                {
                    label: 'Total Deposit in USD',
                    data: <?= json_encode($depositPoints) ?>,
                    fill: true,
                    backgroundColor: depositGradient,
                    borderColor: 'red',
                    borderWidth: 2,
                    tension: 0.4,
                    pointBackgroundColor: 'red',
                    pointBorderColor: 'red',
                    pointRadius: 3
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            interaction: {
                intersect: false,
                mode: 'index'
            },
            plugins: {
                legend: {
                    labels: {
                        color: '#fff',
                        font: { weight: '300' }
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#fff',
                    bodyColor: '#fff',
                    borderColor: '#333',
                    borderWidth: 1
                }
            },
            scales: {
                x: {
                    ticks: { 
                        color: '#ccc',
                        font: { size: 11 }
                    },
                    grid: { 
                        color: 'rgba(255, 255, 255, 0.05)',
                        drawBorder: false
                    }
                },
                y: {
                    ticks: { 
                        color: '#ccc',
                        font: { size: 11 },
                        callback: function(value) {
                            return '$' + value.toLocaleString();
                        }
                    },
                    grid: { 
                        color: 'rgba(255, 255, 255, 0.05)',
                        drawBorder: false
                    },
                    beginAtZero: true
                }
            },
            elements: {
                point: {
                    hoverRadius: 6
                }
            }
        }
    });
</script>

            <!-- Action Buttons -->
          <style>
    .action-buttons {
        display: flex;
        justify-content: center; /* Optional: centers the buttons */
        gap: 20px; /* space between buttons */
        flex-wrap: nowrap; /* keep in one line */
    }

    .action-btn {
        color: white;
        border: none;
        padding: 10px 20px;
        border-radius: 8px;
        cursor: pointer;
        display: flex;
        flex-direction: column;
        align-items: center;
        text-decoration: none; /* removes underline from <a> */
    }

    .action-icon {
        font-size: 20px;
        margin-bottom: 5px;
    }

    .action-label {
        font-size: 14px;
        font-weight: bold;
    }

    /* Optional: Add hover effect */
    .action-btn:hover {
    }
</style>

    
    <div style="display:flex; justify-content:center; margin-top:20px;">
  <div style="display:flex; gap:12px; background:white; box-shadow:0 4px 12px rgba(0,0,0,0.1); border-radius:28px; padding:10px 16px; transition:all 0.3s ease;">

    <!-- Receive Button -->
    <button onclick="openReceiveModal()" style="display:flex; flex-direction:column; align-items:center; justify-content:center; background:#D1FAE5; border:none; border-radius:14px; padding:10px 6px; min-width:52px; height:60px; gap:4px; cursor:pointer; transition:all 0.3s ease;" onmouseover="this.style.transform='scale(1.05)';" onmouseout="this.style.transform='scale(1)'">
      <i class="fas fa-download" style="font-size:14px; color:#059669;"></i>
      <span style="font-size:8px; font-weight:700; color:#047857; text-transform:uppercase;">Recv</span>
    </button>

    <!-- Send Button -->
    <button onclick="openSendModal()" style="display:flex; flex-direction:column; align-items:center; justify-content:center; background:#FECACA; border:none; border-radius:14px; padding:10px 6px; min-width:52px; height:60px; gap:4px; cursor:pointer; transition:all 0.3s ease;" onmouseover="this.style.transform='scale(1.05)';" onmouseout="this.style.transform='scale(1)'">
      <i class="fas fa-paper-plane" style="font-size:14px; color:#B91C1C;"></i>
      <span style="font-size:8px; font-weight:700; color:#991B1B; text-transform:uppercase;">Send</span>
    </button>

    <!-- Trade Button -->
    <button onclick="document.getElementById('tradeModal').style.display='block'; return false;" style="display:flex; flex-direction:column; align-items:center; justify-content:center; background:#DBEAFE; border:none; border-radius:14px; padding:10px 6px; min-width:52px; height:60px; gap:4px; cursor:pointer; transition:all 0.3s ease;" onmouseover="this.style.transform='scale(1.05)';" onmouseout="this.style.transform='scale(1)'">
      <i class="fas fa-exchange-alt" style="font-size:14px; color:#2563EB;"></i>
      <span style="font-size:8px; font-weight:700; color:#1E40AF; text-transform:uppercase;">Trade</span>
    </button>

    <!-- Stake Button -->
    <button onclick="scrollToStakeSection()"  style="display:flex; flex-direction:column; align-items:center; justify-content:center; background:#EDE9FE; border:none; border-radius:14px; padding:10px 6px; min-width:52px; height:60px; gap:4px; cursor:pointer; transition:all 0.3s ease;" onmouseover="this.style.transform='scale(1.05)';" onmouseout="this.style.transform='scale(1)'">
      <i class="fas fa-coins" style="font-size:14px; color:#7C3AED;"></i>
      <span style="font-size:8px; font-weight:700; color:#5B21B6; text-transform:uppercase;">Stake</span>
    </button>

  </div>
</div>

    
    
    
    


    
    </div>
  </div>
</div>

<script>
  const images = [
    "url('https://images.freeimages.com/images/large-previews/55c/forest-sunrise-1334147.jpg')",
    "url('https://img.freepik.com/free-photo/beautiful-nature-landscape-with-mountains-lake_23-2150706049.jpg?semt=ais_hybrid&w=740')",
    "url('https://images.unsplash.com/photo-1612441804231-77a36b284856?fm=jpg&q=60&w=3000&ixlib=rb-4.1.0&ixid=M3wxMjA3fDB8MHxzZWFyY2h8M3x8bW91bnRhaW4lMjBsYW5kc2NhcGV8ZW58MHx8MHx8fDA%3D')",
    "url('https://worldexpeditions.com/croppedImages/North-America/USA/Denali-Alaska-USA-1535125-950px.jpg')",
    "url('https://plus.unsplash.com/premium_photo-1675198764473-30434364c8b6?fm=jpg&q=60&w=3000&ixlib=rb-4.1.0&ixid=M3wxMjA3fDB8MHxzZWFyY2h8MXx8bmF0dXJlJTIwYmFja2dyb3VuZHxlbnwwfHwwfHx8MA%3D%3D')",
    "url('https://media.istockphoto.com/id/1208290498/photo/closeup-nature-view-of-green-leaf-background.jpg?s=612x612&w=0&k=20&c=jD24fB7u7QoWJmeHynZ8RGZD7A1mcOPfsL8X47M_sjM=')",
    "url('https://img.freepik.com/premium-vector/dark-blue-abstract-low-poly-design-background_1048-20593.jpg')"
  ];

  const today = new Date().getDay(); // Sunday = 0
  const dailyImage = images[today];

  document.getElementById('dailyImageDiv').style.background = `
    linear-gradient(135deg, rgba(0, 0, 0, 0.8) 0%, rgba(255, 255, 255, 0.2) 88%),
    ${dailyImage} no-repeat center center
  `;
  document.getElementById('dailyImageDiv').style.backgroundSize = "cover";
</script>




<style>
  @keyframes blink {
    0%,100% { opacity:1; }
    50% { opacity:0; }
  }
  @keyframes boil {
    0%,100% { transform:scale(1); opacity:0.6; }
    50% { transform:scale(1.2); opacity:0.1; }
  }
  @keyframes boil2 {
    0%,100% { transform:scale(1); opacity:0.4; }
    50% { transform:scale(1.3); opacity:0.05; }
  }
  @keyframes boil3 {
    0%,100% { transform:scale(1); opacity:0.3; }
    50% { transform:scale(1.4); opacity:0.08; }
  }
</style>





  <script>
    class FinancialDashboard {
      constructor() {
        this.quotes = {
          morning: [
            "Trade the open.",
            "Start your chart.",
            "Morning moves matter.",
            "Crypto never sleeps.",
            "Plan the breakout.",
            "Stack sats early.",
            "Forex flow begins.",
            "New candles, new gains."
          ],
          afternoon: [
            "Stay on trend.",
            "Volume drives profits.",
            "Watch the news candle.",
            "Midday consolidation.",
            "Let your trades run.",
            "Adjust stop-loss.",
            "Check economic calendar.",
            "Ride the wave."
          ],
          evening: [
            "Recap and reflect.",
            "Set tomorrow’s alerts.",
            "Risk review time.",
            "Analyze your trades.",
            "Crypto grind continues.",
            "Forex resets daily.",
            "Review earnings.",
            "Price action never ends."
          ]
        };

        this.coverImages = {
          morning: [
            'https://miro.medium.com/v2/resize:fit:1400/1*YKiYk6B3Tr0LkPgkKmvOcQ.jpeg',
            'https://static.vecteezy.com/system/resources/thumbnails/001/625/300/small_2x/stock-market-chart-free-video.jpg',
            'https://img.freepik.com/free-photo/3d-rendering-financial-neon-bull_23-2151691899.jpg?semt=ais_items_boosted&w=740',
            'https://img.freepik.com/free-photo/3d-rendering-financial-neon-bull_23-2151691886.jpg?semt=ais_hybrid&w=740',
            'https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcRGKtRhfAfVbQ0LBYF3skpZYznxe7eFLjnE0ARZZZ-hAGf0vGN02PpcemhzOPzRtKInYe0&usqp=CAU'
          ],
          afternoon: [
            'https://e0.pxfuel.com/wallpapers/499/366/desktop-wallpaper-trading-chart-stock-market-crash.jpg',
            'https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcRo9olS4zTHK8brWL9lkvh9zWh2hJj7Ks0YJUbN727Tb1Y_cv0ijxKjS1qb13IsFQLlyJg&usqp=CAU',
            'https://www.thewallstreetschool.com/blog/wp-content/uploads/2024/12/Untitled-design-2024-12-02T112621.343.jpg',
            'https://www.forex.com/en/-/media/project/gain-capital/forex/heroes/markets/shares-and-equities/fxca_stock_market_header-mobile.jpg?h=960&iar=0&w=1440&hash=924555108F19F88C03AC8AFA383E8E9E',
            'https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcSzUNv1N1Ys7MLf24jMTztDEGtNTcbPn--qVB9GtKB5v0NISaMoI6prGicwHNX6K65tJ-Y&usqp=CAU'
          ],
          evening: [
           'https://e0.pxfuel.com/wallpapers/499/366/desktop-wallpaper-trading-chart-stock-market-crash.jpg',
            'https://www.thewallstreetschool.com/blog/wp-content/uploads/2024/12/Untitled-design-2024-12-02T112621.343.jpg',
            'https://www.thewallstreetschool.com/blog/wp-content/uploads/2024/12/Untitled-design-2024-12-02T112621.343.jpg',
            'https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcSJZ1UnCC71wSC2f-bz5hXnDcKoo5Xb5CF_dw&s',
          ]
        };
      }

      getTimeOfDay() {
        const hour = new Date().getHours();
        if (hour < 12) return 'morning';
        else if (hour < 18) return 'afternoon';
        else return 'evening';
      }

      getGreetingWithIcon() {
        const time = this.getTimeOfDay();
        let icon = '', greeting = '';
        switch (time) {
          case 'morning':
            icon = '☀️';
            greeting = 'Good Morning';
            break;
          case 'afternoon':
            icon = '🌤️';
            greeting = 'Good Afternoon';
            break;
          case 'evening':
            icon = '🌙';
            greeting = 'Good Evening';
            break;
        }
        return `<span>${icon}</span> <span>${greeting}</span>`;
      }

      getRandomQuote() {
        const time = this.getTimeOfDay();
        const quotes = this.quotes[time];
        return quotes[Math.floor(Math.random() * quotes.length)];
      }

      getRandomImage() {
        const time = this.getTimeOfDay();
        const images = this.coverImages[time];
        return images[Math.floor(Math.random() * images.length)];
      }

      update() {
        const greetingEl = document.getElementById('greeting');
        const quoteEl = document.getElementById('quote');
        const cover = document.getElementById('cover-section');

        greetingEl.innerHTML = this.getGreetingWithIcon();
        quoteEl.textContent = `"${this.getRandomQuote()}"`;

        const bgImage = this.getRandomImage();
        cover.style.backgroundImage = `url('${bgImage}')`;
      }
    }

    const dashboard = new FinancialDashboard();
    dashboard.update();

    // Optional: Update every hour
    setInterval(() => dashboard.update(), 60 * 60 * 1000);
  </script>






<script>
  
  // Enhanced Portfolio Management
  updatePortfolioData() {
    try {
      const baseValue = 250000;
      const marketHours = this.isMarketHours();
      const volatility = marketHours ? 0.02 : 0.005;
      
      const variation = (Math.random() - 0.5) * baseValue * volatility;
      const portfolioValue = baseValue + variation;
      
      const portfolioElement = this.getElement('portfolioValue');
      if (portfolioElement) {
        portfolioElement.innerHTML = `
          <div style="display: flex; align-items: center; gap: 8px; position: relative; z-index: 10;">
            <span style="font-size: 1.8em; font-weight: 700; color: #ffffff; text-shadow: 3px 3px 8px rgba(0,0,0,0.6);">
              ${this.formatCurrency(portfolioValue)}
            </span>
            <span style="
              font-size: 0.7em; 
              color: rgba(255,255,255,0.9);
              background: rgba(0,0,0,0.4);
              padding: 4px 8px;
              border-radius: 10px;
              font-weight: 700;
              backdrop-filter: blur(10px);
              border: 1px solid rgba(255,255,255,0.2);
              ${marketHours ? 'animation: pulse 2s infinite;' : ''}
            ">
              ${marketHours ? 'LIVE' : 'CLOSED'}
            </span>
          </div>
        `;
      }

      const changePercent = (Math.random() - 0.4) * 8;
      const dollarChange = (portfolioValue * changePercent) / 100;
      const changeElement = this.getElement('portfolioChange');
      
      if (changeElement) {
        const isPositive = changePercent >= 0;
        const arrow = isPositive ? '↗' : '↘';
        const sign = isPositive ? '+' : '';
        
        changeElement.innerHTML = `
          <div style="
            display: flex; 
            align-items: center; 
            gap: 6px;
            padding: 8px 16px;
            border-radius: 20px;
            background: ${isPositive ? 'rgba(34, 197, 94, 0.3)' : 'rgba(239, 68, 68, 0.3)'};
            border: 1px solid ${isPositive ? 'rgba(34, 197, 94, 0.6)' : 'rgba(239, 68, 68, 0.6)'};
            backdrop-filter: blur(15px);
            position: relative;
            z-index: 10;
            box-shadow: 0 8px 25px rgba(0,0,0,0.2);
          ">
            <span style="font-size: 1.2em; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">${arrow}</span>
            <div>
              <div style="color: ${isPositive ? '#22c55e' : '#ef4444'}; font-weight: 700; font-size: 1.1em; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
                ${sign}${changePercent.toFixed(2)}%
              </div>
              <div style="color: ${isPositive ? '#22c55e' : '#ef4444'}; font-size: 0.85em; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
                ${sign}${this.formatCurrency(Math.abs(dollarChange))}
              </div>
            </div>
          </div>
        `;
      }
    } catch (error) {
      console.error('Error updating portfolio data:', error);
    }
  }

  isMarketHours() {
    const now = new Date();
    const hour = now.getHours();
    const day = now.getDay();
    return day >= 1 && day <= 5 && hour >= 9 && hour < 16;
  }

  // Compact Verification Badge
  updateVerificationBadge() {
    try {
      const statuses = ['verified', 'unverified', 'pending', 'premium', 'gold'];
      const randomStatus = statuses[Math.floor(Math.random() * statuses.length)];

      const badges = {
        verified: {
          icon: `<svg width="12" height="12" viewBox="0 0 24 24" fill="#22c55e">
            <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"/>
          </svg>`,
          text: 'Verified',
          color: '#22c55e',
          bgColor: 'rgba(34, 197, 94, 0.2)',
          borderColor: '#22c55e'
        },
        unverified: {
          icon: `<svg width="12" height="12" viewBox="0 0 24 24" fill="#ef4444">
            <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
          </svg>`,
          text: 'Unverified',
          color: '#ef4444',
          bgColor: 'rgba(239, 68, 68, 0.2)',
          borderColor: '#ef4444'
        },
        pending: {
          icon: `<svg width="12" height="12" viewBox="0 0 24 24" fill="#f59e0b">
            <circle cx="12" cy="12" r="10" stroke="#f59e0b" stroke-width="2" fill="none"/>
            <path d="M12 6v6l4 2" stroke="#f59e0b" stroke-width="2"/>
          </svg>`,
          text: 'Pending',
          color: '#f59e0b',
          bgColor: 'rgba(245, 158, 11, 0.2)',
          borderColor: '#f59e0b'
        },
        premium: {
          icon: `<svg width="12" height="12" viewBox="0 0 24 24" fill="#8b5cf6">
            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
          </svg>`,
          text: 'Premium',
          color: '#8b5cf6',
          bgColor: 'rgba(139, 92, 246, 0.2)',
          borderColor: '#8b5cf6'
        },
        gold: {
          icon: `<svg width="12" height="12" viewBox="0 0 24 24" fill="#fbbf24">
            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
          </svg>`,
          text: 'Gold',
          color: '#fbbf24',
          bgColor: 'rgba(251, 191, 36, 0.2)',
          borderColor: '#fbbf24'
        }
      };

      const badge = badges[randomStatus];
      const badgeElement = this.getElement('verificationBadge');
      
      if (badgeElement) {
        badgeElement.innerHTML = `
          <div style="
            display: flex; 
            align-items: center; 
            gap: 4px; 
            padding: 6px 10px; 
            border-radius: 15px; 
            background: ${badge.bgColor};
            border: 1px solid ${badge.borderColor}60;
            backdrop-filter: blur(15px);
            transition: all 0.3s ease;
            cursor: pointer;
            position: relative;
            z-index: 10;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
          " onmouseover="this.style.transform='scale(1.05)'" onmouseout="this.style.transform='scale(1)'">
            ${badge.icon}
            <span style="
              color: ${badge.color}; 
              font-size: 10px; 
              font-weight: 700;
              text-transform: uppercase;
              letter-spacing: 0.5px;
              text-shadow: 1px 1px 2px rgba(0,0,0,0.3);
            ">
              ${badge.text}
            </span>
          </div>
        `;
      }
    } catch (error) {
      console.error('Error updating verification badge:', error);
    }
  }

  // Enhanced Location and IP Management with Multiple APIs
  async fetchUserLocation() {
    if (this.locationCache && this.ipCache) {
      this.updateLocationDisplay(this.locationCache, this.ipCache);
      return;
    }

    const apis = [
      {
        url: 'https://ipapi.co/json/',
        parser: (data) => ({
          country: data.country_name,
          city: data.city,
          region: data.region,
          ip: data.ip,
          timezone: data.timezone,
          isp: data.org,
          latitude: data.latitude,
          longitude: data.longitude,
          countryCode: data.country_code
        })
      },
      {
        url: 'https://ipgeolocation.abstractapi.com/v1/?api_key=free',
        parser: (data) => ({
          country: data.country,
          city: data.city,
          region: data.region,
          ip: data.ip_address,
          timezone: data.timezone?.name || 'Unknown',
          isp: data.connection?.isp_name || 'Unknown',
          latitude: data.latitude,
          longitude: data.longitude,
          countryCode: data.country_code
        })
      },
      {
        url: 'https://geolocation-db.com/json/',
        parser: (data) => ({
          country: data.country_name,
          city: data.city,
          region: data.state,
          ip: data.IPv4,
          timezone: 'Unknown',
          isp: 'Unknown',
          latitude: data.latitude,
          longitude: data.longitude,
          countryCode: data.country_code
        })
      },
      {
        url: 'https://api.ipify.org?format=json',
        parser: async (data) => {
          // Get IP first, then use another service for location
          const ip = data.ip;
          try {
            const locationResponse = await fetch(`https://ipapi.co/${ip}/json/`);
            const locationData = await locationResponse.json();
            return {
              country: locationData.country_name,
              city: locationData.city,
              region: locationData.region,
              ip: ip,
              timezone: locationData.timezone,
              isp: locationData.org,
              latitude: locationData.latitude,
              longitude: locationData.longitude,
              countryCode: locationData.country_code
            };
          } catch (error) {
            return null;
          }
        }
      },
      {
        url: 'https://api.db-ip.com/v2/free/self',
        parser: (data) => ({
          country: data.countryName,
          city: data.city,
          region: data.stateProv,
          ip: data.ipAddress,
          timezone: 'Unknown',
          isp: 'Unknown',
          latitude: data.latitude,
          longitude: data.longitude,
          countryCode: data.countryCode
        })
      }
    ];

    for (const api of apis) {
      try {
        console.log(`Trying API: ${api.url}`);
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000);
        
        const response = await fetch(api.url, { 
          mode: 'cors',
          signal: controller.signal,
          headers: {
            'Accept': 'application/json',
          }
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        console.log('API Response:', data);
        
        let location;
        if (typeof api.parser === 'function') {
          location = await api.parser(data);
        } else {
          location = api.parser(data);
        }
        
        if (location && location.country && location.ip) {
          console.log('Location found:', location);
          this.locationCache = location;
          this.ipCache = location;
          this.updateLocationDisplay(location, location);
          return;
        } else {
          console.warn('Incomplete location data from API:', api.url);
        }
      } catch (error) {
        console.warn(`Location API failed: ${api.url}`, error.message);
        continue;
      }
    }

    console.warn('All location APIs failed, using fallback');
    const fallback = { 
      country: 'Unknown Location', 
      city: 'Unknown City',
      region: 'Unknown Region',
      ip: '0.0.0.0',
      timezone: 'Unknown',
      isp: 'Unknown ISP',
      countryCode: 'XX'
    };
    this.updateLocationDisplay(fallback, fallback);
  }

  updateLocationDisplay(location, ipData) {
    
    try {
      const countryElement = this.getElement('countryName');
      const flagElement = document.querySelector('#locationSection img');
      const ipElement = this.getElement('userIP');
      const locationDetails = this.getElement('locationDetails');

      if (countryElement) {
        countryElement.textContent = location.country || 'Unknown Location';
      }

      // Update flag if country code is available
      if (flagElement && location.countryCode) {
        flagElement.src = `https://flagcdn.com/w40/${location.countryCode.toLowerCase()}.png`;
        flagElement.alt = `${location.country} flag`;
        flagElement.style.cssText = `
          width: 24px;
          height: auto;
          border-radius: 4px;
          box-shadow: 0 2px 8px rgba(0,0,0,0.3);
        `;
      }

      if (ipElement) {
        ipElement.innerHTML = `
          <div style="
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 12px;
            background: rgba(255,255,255,0.15);
            backdrop-filter: blur(15px);
            border-radius: 12px;
            border: 1px solid rgba(255,255,255,0.2);
            position: relative;
            z-index: 10;
          ">
            <svg width="14" height="14" viewBox="0 0 24 24" fill="#ffffff">
              <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
            </svg>
            <span style="color: #ffffff; font-weight: 600; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
              ${ipData.ip}
            </span>
          </div>
        `;
      }

      if (locationDetails) {
        locationDetails.innerHTML = `
          <div style="
            display: grid;
            gap: 8px;
            padding: 16px;
            background: rgba(255,255,255,0.1);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            border: 1px solid rgba(255,255,255,0.2);
            position: relative;
            z-index: 10;
            box-shadow: 0 8px 32px rgba(0,0,0,0.2);
          ">
            <div style="display: flex; justify-content: space-between; align-items: center;">
              <span style="color: rgba(255,255,255,0.8); font-size: 0.9em;">City:</span>
              <span style="color: #ffffff; font-weight: 600; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
                ${location.city || 'Unknown'}
              </span>
            </div>
            <div style="display: flex; justify-content: space-between; align-items: center;">
              <span style="color: rgba(255,255,255,0.8); font-size: 0.9em;">Region:</span>
              <span style="color: #ffffff; font-weight: 600; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
                ${location.region || 'Unknown'}
              </span>
            </div>
            <div style="display: flex; justify-content: space-between; align-items: center;">
              <span style="color: rgba(255,255,255,0.8); font-size: 0.9em;">Timezone:</span>
              <span style="color: #ffffff; font-weight: 600; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
                ${location.timezone || 'Unknown'}
              </span>
            </div>
            <div style="display: flex; justify-content: space-between; align-items: center;">
              <span style="color: rgba(255,255,255,0.8); font-size: 0.9em;">ISP:</span>
              <span style="color: #ffffff; font-weight: 600; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
                ${location.isp || 'Unknown'}
              </span>
            </div>
          </div>
        `;
      }
    } catch (error) {
      console.error('Error updating location display:', error);
    }
  }

  // Market Status and Stock Updates
  updateMarketStatus() {
    try {
      const marketElement = this.getElement('marketStatus');
      if (!marketElement) return;

      const isOpen = this.isMarketHours();
      const nextOpen = this.getNextMarketOpen();
      
      marketElement.innerHTML = `
        <div style="
          display: flex;
          align-items: center;
          gap: 12px;
          padding: 12px 16px;
          background: rgba(255,255,255,0.15);
          backdrop-filter: blur(20px);
          border-radius: 16px;
          border: 1px solid rgba(255,255,255,0.2);
          position: relative;
          z-index: 10;
          box-shadow: 0 8px 32px rgba(0,0,0,0.2);
        ">
          <div style="
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: ${isOpen ? '#22c55e' : '#ef4444'};
            animation: ${isOpen ? 'pulse 2s infinite' : 'none'};
            box-shadow: 0 0 0 0 ${isOpen ? 'rgba(34, 197, 94, 0.7)' : 'rgba(239, 68, 68, 0.7)'};
          "></div>
          <div>
            <div style="color: #ffffff; font-weight: 700; font-size: 1.1em; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
              ${isOpen ? 'Market Open' : 'Market Closed'}
            </div>
            <div style="color: rgba(255,255,255,0.8); font-size: 0.85em;">
              ${isOpen ? 'Trading active' : `Opens ${nextOpen}`}
            </div>
          </div>
        </div>
      `;
    } catch (error) {
      console.error('Error updating market status:', error);
    }
  }

  getNextMarketOpen() {
    const now = new Date();
    const tomorrow = new Date(now);
    tomorrow.setDate(tomorrow.getDate() + 1);
    tomorrow.setHours(9, 30, 0, 0);
    
    // If it's weekend, go to Monday
    if (tomorrow.getDay() === 0) { // Sunday
      tomorrow.setDate(tomorrow.getDate() + 1);
    } else if (tomorrow.getDay() === 6) { // Saturday
      tomorrow.setDate(tomorrow.getDate() + 2);
    }
    
    return tomorrow.toLocaleDateString('en-US', { 
      weekday: 'short', 
      month: 'short', 
      day: 'numeric',
      hour: 'numeric',
      minute: '2-digit'
    });
  }

  // Performance Metrics
  updatePerformanceMetrics() {
    try {
      const metricsElement = this.getElement('performanceMetrics');
      if (!metricsElement) return;

      const metrics = {
        dailyReturn: (Math.random() - 0.3) * 5,
        weeklyReturn: (Math.random() - 0.3) * 15,
        monthlyReturn: (Math.random() - 0.2) * 25,
        yearlyReturn: (Math.random() + 0.2) * 45
      };

      metricsElement.innerHTML = `
        <div style="
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
          gap: 12px;
          padding: 16px;
          background: rgba(255,255,255,0.1);
          backdrop-filter: blur(20px);
          border-radius: 16px;
          border: 1px solid rgba(255,255,255,0.2);
          position: relative;
          z-index: 10;
          box-shadow: 0 8px 32px rgba(0,0,0,0.2);
        ">
          ${Object.entries(metrics).map(([period, value]) => {
            const isPositive = value >= 0;
            const periodLabel = period.replace('Return', '').replace(/([A-Z])/g, ' $1').trim();
            return `
              <div style="text-align: center;">
                <div style="color: rgba(255,255,255,0.8); font-size: 0.8em; text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 4px;">
                  ${periodLabel}
                </div>
                <div style="
                  color: ${isPositive ? '#22c55e' : '#ef4444'};
                  font-weight: 700;
                  font-size: 1.2em;
                  text-shadow: 1px 1px 3px rgba(0,0,0,0.6);
                ">
                  ${isPositive ? '+' : ''}${value.toFixed(2)}%
                </div>
              </div>
            `;
          }).join('')}
        </div>
      `;
    } catch (error) {
      console.error('Error updating performance metrics:', error);
    }
  }

  // News Feed Simulation
  updateNewsFeed() {
    try {
      const newsElement = this.getElement('newsFeed');
      if (!newsElement) return;

      const newsItems = [
        "Tech stocks rally as AI sector shows strong growth",
        "Federal Reserve maintains interest rates at current levels",
        "Cryptocurrency market sees increased institutional adoption",
        "Energy sector outperforms amid rising oil prices",
        "Market volatility expected as earnings season approaches",
        "ESG investing trends continue to shape portfolio strategies",
        "Emerging markets show resilience despite global headwinds",
        "Real estate investment trusts gain momentum"
      ];

      const randomNews = newsItems.sort(() => 0.5 - Math.random()).slice(0, 3);

      newsElement.innerHTML = `
        <div style="
          background: rgba(255,255,255,0.1);
          backdrop-filter: blur(20px);
          border-radius: 16px;
          border: 1px solid rgba(255,255,255,0.2);
          position: relative;
          z-index: 10;
          box-shadow: 0 8px 32px rgba(0,0,0,0.2);
          overflow: hidden;
        ">
          <div style="
            padding: 16px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            background: rgba(255,255,255,0.05);
          ">
            <h3 style="color: #ffffff; margin: 0; font-size: 1.1em; font-weight: 700; text-shadow: 1px 1px 3px rgba(0,0,0,0.6);">
              Market News
            </h3>
          </div>
          <div style="padding: 16px;">
            ${randomNews.map((news, index) => `
              <div style="
                padding: 12px 0;
                ${index < randomNews.length - 1 ? 'border-bottom: 1px solid rgba(255,255,255,0.1);' : ''}
                animation: fadeInUp 0.8s ease ${index * 0.2}s both;
              ">
                <div style="
                  color: #ffffff;
                  font-size: 0.95em;
                  line-height: 1.5;
                  text-shadow: 1px 1px 3px rgba(0,0,0,0.6);
                ">
                  ${news}
                </div>
                <div style="
                  color: rgba(255,255,255,0.6);
                  font-size: 0.75em;
                  margin-top: 4px;
                ">
                  ${Math.floor(Math.random() * 60)} minutes ago
                </div>
              </div>
            `).join('')}
          </div>
        </div>
      `;
    } catch (error) {
      console.error('Error updating news feed:', error);
    }
  }

  // Clock Updates
  updateClock() {
    try {
      const clockElement = this.getElement('currentTime');
      if (!clockElement) return;

      const now = new Date();
      const timeString = now.toLocaleTimeString('en-US', {
        hour12: true,
        hour: 'numeric',
        minute: '2-digit',
        second: '2-digit'
      });

      const dateString = now.toLocaleDateString('en-US', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
      });

      clockElement.innerHTML = `
        <div style="
          text-align: center;
          padding: 16px;
          background: rgba(255,255,255,0.15);
          backdrop-filter: blur(20px);
          border-radius: 16px;
          border: 1px solid rgba(255,255,255,0.2);
          position: relative;
          z-index: 10;
          box-shadow: 0 8px 32px rgba(0,0,0,0.2);
        ">
          <div style="
            color: #ffffff;
            font-size: 1.8em;
            font-weight: 700;
            margin-bottom: 8px;
            text-shadow: 2px 2px 6px rgba(0,0,0,0.6);
            font-family: 'Courier New', monospace;
          ">
            ${timeString}
          </div>
          <div style="
            color: rgba(255,255,255,0.8);
            font-size: 0.9em;
            text-shadow: 1px 1px 3px rgba(0,0,0,0.6);
          ">
            ${dateString}
          </div>
        </div>
      `;
    } catch (error) {
      console.error('Error updating clock:', error);
    }
  }

  // Cleanup method
  cleanup() {
    this.intervals.forEach(interval => clearInterval(interval));
    this.intervals = [];
  }

  // Main initialization
  init() {
    try {
      console.log('Initializing Financial Dashboard...');
      
      // Initial updates
      this.updateGreetingAndQuote();
      this.updatePortfolioData();
      this.updateVerificationBadge();
      this.updateMarketStatus();
      this.updatePerformanceMetrics();
      this.updateNewsFeed();
      this.updateClock();
      
      // Fetch location data
      this.fetchUserLocation();

      // Set up intervals
      this.intervals.push(
        setInterval(() => this.updatePortfolioData(), 5000),
        setInterval(() => this.updateGreetingAndQuote(), 300000), // 5 minutes
        setInterval(() => this.updateVerificationBadge(), 30000),
        setInterval(() => this.updateMarketStatus(), 60000),
        setInterval(() => this.updatePerformanceMetrics(), 15000),
        setInterval(() => this.updateNewsFeed(), 120000), // 2 minutes
        setInterval(() => this.updateClock(), 1000)
      );

      // Add CSS animations
      this.addAnimations();
      
      console.log('Financial Dashboard initialized successfully!');
    } catch (error) {
      console.error('Error initializing Financial Dashboard:', error);
    }
  }

  // Add CSS animations
  addAnimations() {
    const style = document.createElement('style');
    style.textContent = `
      @keyframes fadeInUp {
        from {
          opacity: 0;
          transform: translateY(30px);
        }
        to {
          opacity: 1;
          transform: translateY(0);
        }
      }
      
      @keyframes slideInRight {
        from {
          opacity: 0;
          transform: translateX(50px);
        }
        to {
          opacity: 1;
          transform: translateX(0);
        }
      }
      
      @keyframes pulse {
        0% {
          box-shadow: 0 0 0 0 rgba(34, 197, 94, 0.7);
        }
        70% {
          box-shadow: 0 0 0 10px rgba(34, 197, 94, 0);
        }
        100% {
          box-shadow: 0 0 0 0 rgba(34, 197, 94, 0);
        }
      }
      
      @keyframes shimmer {
        0%, 100% {
          opacity: 0.5;
        }
        50% {
          opacity: 0.8;
        }
      }
    `;
    document.head.appendChild(style);
  }
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  window.financialDashboard = new FinancialDashboard();
});

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
  if (window.financialDashboard) {
    window.financialDashboard.cleanup();
  }
});
</script>









 
 




</div>


  &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;

  <!-- Main container -->
  <!-- Navigation Buttons -->
  <div class="w-full flex flex-col items-center p-4">

    
    
    <div class="flex gap-2 border-b border-gray-400 dark:border-gray-700 overflow-x-auto p-2 rounded-lg justify-center">





      <!-- Button to trigger the "Fiat" modal -->
      <button type="button" onclick="showFiatModal()" style="
    display: inline-flex;
    align-items: center;
    gap: 6px; /* Reduced gap */
    background: linear-gradient(135deg, #7F00FF, #E100FF);
    padding: 8px 16px; /* Reduced padding */
    font-size: 13px; /* Slightly smaller font size */
    border-radius: 6px;
    border: none;
    cursor: pointer;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
    box-shadow: 0 3px 10px rgba(0,0,0,0.1);
  " onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 5px 14px rgba(0,0,0,0.15)'"
        onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 10px rgba(0,0,0,0.1)'">
        <!-- Inline SVG Icon (Dollar Sign in Circle) -->
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="14" height="14" <!-- Reduced size -->
          fill="#FFFFFF"
          style="flex-shrink: 0;"
          >
          <path
            d="M12 1C5.925 1 1 5.925 1 12s4.925 11 11 11 11-4.925 11-11S18.075 1 12 1zm.25 16.75c-1.629 0-3.25-.875-3.25-2.5h1.5c0 .552.694 1 .25 1 .732 0 1.5-.518 1.5-1.25s-.758-1.25-1.5-1.25c-1.629 0-3.25-.875-3.25-2.5s1.621-2.5 3.25-2.5 3.25.875 3.25 2.5h-1.5c0-.552-.694-1-.25-1-.732 0-1.5.518-1.5 1.25s.758 1.25 1.5 1.25c1.629 0 3.25.875 3.25 2.5s-1.621 2.5-3.25 2.5z" />
        </svg>
        <span style="
    background-color: rgba(255, 255, 255, 0.2);
    color: #FFFFFF;
    font-size: 11px; /* Slightly smaller text */
    font-weight: 500;
    padding: 3px 8px; /* Reduced padding */
    border-radius: 9999px;
  ">Cashout</span>
      </button>


     <script>
  function showFiatModal() {
    Swal.fire({
      title: 'Choose a Payment Method',
      showConfirmButton: false,
      showCloseButton: true,
      html: `
        <div style="display: flex; flex-direction: column; gap: 20px; margin-top: 20px;">
          <a href="./raying/index.php" style="display: flex; align-items: center; gap: 10px; background-color: #e0f2fe; padding: 10px 20px; border-radius: 8px; text-decoration: none; color: #000; font-weight: 500;">
            <i class="fas fa-university" style="color: #0284c7;"></i> Bank Transfer
          </a>
          <a href="./crypto.php" style="display: flex; align-items: center; gap: 10px; background-color: #fef9c3; padding: 10px 20px; border-radius: 8px; text-decoration: none; color: #000; font-weight: 500;">
            <i class="fab fa-bitcoin" style="color: #f59e0b;"></i> Crypto Payment
          </a>
          <a href="#" id="paypalSupportBtn" style="display: flex; align-items: center; gap: 10px; background-color: #e0f7fa; padding: 10px 20px; border-radius: 8px; text-decoration: none; color: #000; font-weight: 500;">
            <i class="fab fa-paypal" style="color: #0070ba;"></i> PayPal
          </a>
        </div>
      `,
      background: '#fff',
      didOpen: () => {
        // Attach click event when the modal opens
        const paypalBtn = document.getElementById('paypalSupportBtn');
        if (paypalBtn) {
          paypalBtn.addEventListener('click', function (e) {
            e.preventDefault(); // prevent link default

            Swal.fire({
              title: 'Contact Support',
              html: 'Please contact support at:<br><strong><a href="mailto:support@primemarketspro.com">support@primemarketspro.com</a></strong>',
              icon: 'info',
              confirmButtonText: 'OK',
              confirmButtonColor: '#0070ba'
            });
          });
        }
      },
      customClass: {
        popup: 'swal2-rounded'
      }
    });
  }
</script>





      <?php
// Start session and enable error reporting (disable in production)
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
ini_set('display_errors', 1);
error_reporting(E_ALL);

/**
 * Logs messages with a timestamp to a log file.
 *
 * @param string $message  The message to log.
 * @param string $severity The severity level (e.g., INFO, WARNING, ERROR).
 */
function logError($message, $severity = 'ERROR') {
    $logFile = __DIR__ . '/error_log.txt';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$severity] $message" . PHP_EOL;
    error_log($logMessage, 3, $logFile);
}

// Check if the user is logged in by verifying the session variable
if (!isset($_SESSION['user_email'])) {
    echo "<div class='alert alert-warning'>User is not logged in or email is not set in session.</div>";
    logError("User attempted to access dashboard without login", "WARNING");
    exit();
}

$userEmail = trim($_SESSION['user_email']);
logError("User accessing dashboard: $userEmail", "INFO");

// Database connection details
$servername = "localhost";
$username   = "wptlgmnb_userp";
$password   = "Qliox4ITygj6Vh4V";
$dbname     = "wptlgmnb_dbp";

// Create a new MySQLi connection
$conn = new mysqli($servername, $username, $password, $dbname);
if ($conn->connect_error) {
    logError("Database connection failed: " . $conn->connect_error, "ERROR");
    die("<div class='alert alert-danger'>Database connection failed.</div>");
}

// Fetch user data from the `users` table
$userData = [];
$userQuery = "SELECT * FROM users WHERE email = ?";
if ($stmt = $conn->prepare($userQuery)) {
    $stmt->bind_param("s", $userEmail);
    $stmt->execute();
    $resultUser = $stmt->get_result();
    if ($row = $resultUser->fetch_assoc()) {
        $userData = $row;
        logError("Fetched user data for: $userEmail", "INFO");
    } else {
        logError("No user data found for: $userEmail", "WARNING");
    }
    $stmt->close();
} else {
    logError("Failed to prepare user query: " . $conn->error, "ERROR");
}

// Fetch wallet data from the `wallets` table
$walletData = [];
$walletQuery = "SELECT * FROM wallets WHERE user_email = ?";
if ($stmt = $conn->prepare($walletQuery)) {
    $stmt->bind_param("s", $userEmail);
    $stmt->execute();
    $resultWallet = $stmt->get_result();
    if ($row = $resultWallet->fetch_assoc()) {
        $walletData = $row;
        logError("Fetched wallet data for: $userEmail", "INFO");
    } else {
        logError("No wallet data found for: $userEmail", "WARNING");
    }
    $stmt->close();
} else {
    logError("Failed to prepare wallet query: " . $conn->error, "ERROR");
}

$conn->close();

// Extract user percent value from $userData (default to 0 if not available)
$userPercent = 0;
if (!empty($userData) && isset($userData['percent'])) {
    $userPercent = (float)$userData['percent'];
}

// Extract user plan from $userData (default to 'Free' if not set)
$userPlan = isset($userData['plan']) ? $userData['plan'] : 'Free';
?>

      <!-- Plan Button -->
      <button type="button" onclick="showPaidPlanModal()" style="
    display: inline-flex;
    align-items: center;
    gap: 5px;
    background: linear-gradient(135deg, #10b981, #06d6a0);
    padding: 6px 14px;
    font-size: 12px;
    border-radius: 6px;
    border: none;
    cursor: pointer;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
    box-shadow: 0 3px 10px rgba(0,0,0,0.1);
  " onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 5px 14px rgba(0,0,0,0.15)'"
        onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 10px rgba(0,0,0,0.1)'">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="14" height="14" fill="#FFFFFF"
          style="flex-shrink: 0;">
          <path d="M12 2l3.09 6.26L22 9.27l-5 4.87L18.18 22 12 18.27 5.82 22 7 14.14l-5-4.87 6.91-1.01L12 2z" />
        </svg>
        <span style="
    background-color: rgba(255, 255, 255, 0.2);
    color: #FFFFFF;
    font-size: 11px;
    font-weight: 500;
    padding: 3px 8px;
    border-radius: 9999px;
  ">Plan</span>
      </button>

      <!-- Signal Button -->
      <button type="button" onclick="showSignalBoostModalUser()" style="
    display: inline-flex;
    align-items: center;
    gap: 5px;
    background: linear-gradient(135deg, #FBBF24, #F59E0B);
    padding: 6px 14px;
    font-size: 12px;
    border-radius: 6px;
    border: none;
    cursor: pointer;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
    box-shadow: 0 3px 10px rgba(0,0,0,0.1);
  " onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 5px 14px rgba(0,0,0,0.15)'"
        onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 10px rgba(0,0,0,0.1)'">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="14" height="14" fill="#FFFFFF"
          style="flex-shrink: 0;">
          <path d="M11 21h-1l1-7H5l7-12h1l-1 7h6l-7 12z" />
        </svg>
        <span style="
    background-color: rgba(255, 255, 255, 0.2);
    color: #FFFFFF;
    font-size: 11px;
    font-weight: 500;
    padding: 3px 8px;
    border-radius: 9999px;
  ">Signal</span>
      </button>

      <!-- Wallet Button -->
      <button type="button" onclick="checkWalletStatus()" style="
    display: inline-flex;
    align-items: center;
    gap: 5px;
    background: linear-gradient(135deg, #3B82F6, #60A5FA);
    padding: 6px 14px;
    font-size: 12px;
    border-radius: 6px;
    border: none;
    cursor: pointer;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
    box-shadow: 0 3px 10px rgba(0,0,0,0.1);
  " onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 5px 14px rgba(0,0,0,0.15)'"
        onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 10px rgba(0,0,0,0.1)'">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="14" height="14" fill="#FFFFFF"
          style="flex-shrink: 0;">
          <path
            d="M21 7H3C1.897 7 1 7.897 1 9v6c0 1.103.897 2 2 2h18c1.103 0 2-.897 2-2V9c0-1.103-.897-2-2-2zm0 8H3V9h18v6zM5 11h4v2H5v-2z" />
        </svg>
        <span style="
    background-color: rgba(255, 255, 255, 0.2);
    color: #FFFFFF;
    font-size: 11px;
    font-weight: 500;
    padding: 3px 8px;
    border-radius: 9999px;
  ">Wallet</span>
      </button>


    </div>
    <!-- Include SweetAlert2 JS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
  // Pass PHP variables to JavaScript
  var userPlan = <?php echo json_encode($userPlan); ?>;
  var userPercent = <?php echo json_encode($userPercent); ?>;
  var walletFound = <?php echo (!empty($walletData)) ? 'true' : 'false'; ?>;

  // Debug logs
  console.log("User Plan:", userPlan);
  console.log("User Percent:", userPercent);
  console.log("Wallet Found:", walletFound);

  /**
   * Displays a SweetAlert modal with the user's current plan and an upgrade option.
   */
  function showPaidPlanModal() {
    console.log("showPaidPlanModal called");
    Swal.fire({
      icon: 'success',
      title: 'Your Current Plan: ' + userPlan,
      html: `
        <p>Your current plan is <strong>${userPlan}</strong>.</p>
        <button id="upgradeButton"
          style="margin-top: 10px;
                 background-color: #3085d6;
                 color: #fff;
                 border: none;
                 border-radius: 5px;
                 padding: 10px 20px;
                 cursor: pointer;">
          Upgrade Plan
        </button>
      `,
      showCancelButton: true,
      cancelButtonText: 'Close',
      showConfirmButton: false,
      didOpen: () => {
        document.getElementById('upgradeButton').addEventListener('click', function () {
          window.location.href = './plans.php';
        });
      }
    });
  }

  /**
   * Shows SweetAlert modal based on signal strength.
   */
  function showSignalBoostModalUser() {
    let icon = (userPercent < 5) ? "error" : "success";
    let title = (userPercent < 5) ? "Low Signal Strength" : "Good Signal Strength";
    let text = `Signal strength is ${userPercent}%. ` +
               ((userPercent < 5) ? "Consider boosting your signal." : "You have a robust signal.");

    Swal.fire({
      title: title,
      text: text,
      icon: icon,
      showCancelButton: true,
      confirmButtonText: "Boost Signal",
      cancelButtonText: "Close",
      reverseButtons: true,
      customClass: {
        confirmButton: 'bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded',
        cancelButton: 'bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded'
      }
    }).then((result) => {
      if (result.isConfirmed) {
        window.location.href = "./signalbuy.php";
      }
    });
  }

  /**
   * Shows SweetAlert modal based on wallet status.
   */
  function checkWalletStatus() {
    if (walletFound) {
      Swal.fire({
        title: "Wallet Connected!",
        text: "Your wallet is connected successfully.",
        icon: "success",
        background: "#000",
        color: "#fff",
        confirmButtonColor: "#4299e1"
      });
    } else {
      Swal.fire({
        title: "No Wallet Connected!",
        html: `Please <a href="connect.php" style="color:#4299e1; text-decoration: underline;">connect your wallet</a>.`,
        icon: "warning",
        background: "#000",
        color: "#fff",
        confirmButtonColor: "#4299e1"
      });
    }
  }
</script>

  </div>







  </div>


  <!-- Section with margin-bottom -->
  <section style="margin-bottom: 2rem;">
    <!-- Crypto ticker container -->
    <div id="crypto-ticker-container"
      style="overflow: hidden; white-space: nowrap; background-color: #f0f0f0; padding: 10px; border: 1px solid #ddd;">
      <!-- Scrolling container with duplicated content for seamless loop -->
      <div id="crypto-ticker-content" class="ticker-wrapper"></div>
      <div id="crypto-ticker-content-duplicate" class="ticker-wrapper"></div>
    </div>
  </section>

  <style>
    /* Define scrolling animation */
    @keyframes tickerScroll {
      from {
        transform: translateX(0);
      }

      to {
        transform: translateX(-100%);
      }
    }

    /* Ticker container setup */
    .ticker-wrapper {
      display: inline-flex;
      white-space: nowrap;
      animation: tickerScroll 90s linear infinite;
      /* Smooth infinite scrolling */
    }
  </style>

  <script>
    /**
     * Fetch crypto data from the CoinGecko API, sort by top movers,
     * and create a seamless infinite scrolling effect.
     */
    async function fetchCryptoMovers() {
      try {
        const response = await fetch('https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&order=market_cap_desc&per_page=250&page=1&sparkline=false');
        const data = await response.json();

        // Filter and sort coins by absolute 24h change
        const validData = data.filter(coin => coin.price_change_percentage_24h !== null);
        validData.sort((a, b) => Math.abs(b.price_change_percentage_24h) - Math.abs(a.price_change_percentage_24h));

        // Get top 60 movers
        const topMovers = validData.slice(0, 60);

        let tickerHTML = "";
        topMovers.forEach(coin => {
          const changeColor = coin.price_change_percentage_24h >= 0 ? 'green' : 'red';

          tickerHTML += `<span style="margin-right: 40px; font-size: 16px; color: #333; display: inline-flex; align-items: center;">
                         <img src="${coin.image}" alt="${coin.name}" width="20" height="20" style="margin-right: 5px; vertical-align: middle;">
                         <span style="font-weight: bold;">${coin.name}:</span>
                         <span style="margin-left: 5px;">$${coin.current_price.toLocaleString()}</span>
                         <span style="margin-left: 5px; font-weight: bold; color: ${changeColor};">
                           (${coin.price_change_percentage_24h.toFixed(2)}%)
                         </span>
                      </span>`;
        });

        // Insert content twice to create an infinite loop effect
        document.getElementById('crypto-ticker-content').innerHTML = tickerHTML;
        document.getElementById('crypto-ticker-content-duplicate').innerHTML = tickerHTML;
      } catch (error) {
        console.error('Error fetching crypto data:', error);
      }
    }

    // Fetch and display crypto movers
    fetchCryptoMovers();
  </script>

  </p>

  </p>





  <style>
    @keyframes shimmer {
      0% {
        background-position: -450px 0;
      }

      100% {
        background-position: 450px 0;
      }
    }

    .shimmer {
      animation: shimmer 1.5s linear infinite;
      background: linear-gradient(to right, #eeeeee 8%, #dddddd 18%, #eeeeee 33%);
      background-size: 800px 104px;
    }

    .icon-container {
      width: 3rem;
      height: 3rem;
      display: flex;
      align-items: center;
      justify-content: center;
      border-radius: 50%;
    }

    .slider-container {
      max-width: 1200px;
      margin: 0 auto;
      position: relative;
    }

    .slider-wrapper {
      overflow: hidden;
      border-radius: 12px;
    }

    .slider-track {
      display: flex;
      transition: transform 0.3s ease-in-out;
    }

    .card {
      background: linear-gradient(135deg, #6B46C1 0%, #9333EA 100%);
      color: white;
      padding: 12px 16px;
      border-radius: 12px;
      display: flex;
      align-items: center;
      gap: 12px;
      box-shadow: 0 6px 18px rgba(107, 70, 193, 0.3);
      cursor: pointer;
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
      min-height: 72px;
    }

    .card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, rgba(255, 255, 255, 0) 100%);
      opacity: 0;
      transition: opacity 0.3s ease;
    }

    .card:hover::before {
      opacity: 1;
    }

    .card:hover {
      transform: translateY(-2px);
      box-shadow: 0 10px 25px rgba(107, 70, 193, 0.4);
    }

    .card-icon-container {
      width: 40px;
      height: 40px;
      background: rgba(255, 255, 255, 0.2);
      border-radius: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      flex-shrink: 0;
    }

    .card-content {
      display: flex;
      flex-direction: row;
      align-items: center;
      gap: 16px;
      flex: 1;
      overflow: hidden;
    }

    .card-title {
      font-size: 16px;
      font-weight: 600;
      margin: 0;
      white-space: nowrap;
    }

    .card-value {
      font-family: 'Courier New', monospace;
      font-size: 18px;
      font-weight: 700;
      letter-spacing: 0.5px;
      margin: 0;
      white-space: nowrap;
    }

    .card-subtitle {
      font-size: 12px;
      opacity: 0.8;
      white-space: nowrap;
    }

    .card-badge {
      background: rgba(255, 255, 255, 0.2);
      padding: 2px 6px;
      border-radius: 6px;
      font-size: 10px;
      font-weight: 500;
      display: inline-flex;
      align-items: center;
      gap: 4px;
      margin-left: auto;
      white-space: nowrap;
    }

    @media (max-width: 768px) {
      .card {
        min-width: calc(66.67% - 8px);
        margin-right: 12px;
      }
    }

    @media (min-width: 769px) {
      .card {
        flex: 1;
        margin-right: 12px;
      }

      .card:last-child {
        margin-right: 0;
      }
    }

    .dots-container {
      display: flex;
      justify-content: center;
      gap: 8px;
      margin-top: 20px;
    }

    .dot {
      width: 8px;
      height: 8px;
      border-radius: 50%;
      background: #CBD5E1;
      cursor: pointer;
      transition: all 0.3s ease;
    }

    .dot.active {
      background: #6B46C1;
      transform: scale(1.2);
    }

    @media (min-width: 769px) {
      .dots-container {
        display: none;
      }
    }

    .nav-button {
      position: absolute;
      top: 50%;
      transform: translateY(-50%);
      background: rgba(255, 255, 255, 0.9);
      border: none;
      width: 40px;
      height: 40px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      transition: all 0.3s ease;
      z-index: 10;
    }

    .nav-button:hover {
      background: white;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    }

    .nav-button.prev {
      left: -20px;
    }

    .nav-button.next {
      right: -20px;
    }

    @media (min-width: 769px) {
      .nav-button {
        display: none;
      }
    }

    .swal2-popup {
      border-radius: 12px;
    }

    .swal2-content {
      padding: 0 1rem;
    }

    .neumorphic-card {
      background: #f8f9fa;
      border-radius: 10px;
      box-shadow: 5px 5px 10px #e1e5ee, -5px -5px 10px #ffffff;
      transition: all 0.3s ease;
    }

    .neumorphic-card:hover {
      box-shadow: 3px 3px 6px #e1e5ee, -3px -3px 6px #ffffff;
    }

    @media (max-width: 640px) {
      .swal2-popup {
        width: 90% !important;
        padding: 1rem !important;
      }
    }
  </style>


  <div class="slider-container">
    <div class="slider-wrapper">
      <div class="slider-track" id="sliderTrack">
        <!-- Profit Card -->
        <div class="card" id="profit-card">
          <div class="card-icon-container">
            <i class="fas fa-chart-line fa-lg" style="color: white;"></i>
          </div>
          <div class="card-content">
            <div class="card-title">Profit</div>
            <div class="card-value">$<span id="profit-counter">0.00</span></div>
            <div class="card-badge">
              <i class="fas fa-caret-up"></i>
              <span id="profitPercent">5.2%</span>
            </div>
          </div>

          <script>
            document.addEventListener('DOMContentLoaded', function () {
              const targetValue = <?= (float) $profit ?>;
              const duration = 2000; // 2 seconds for the animation
              const counter = document.getElementById('profit-counter');
              let startTimestamp = null;

              function step(timestamp) {
                if (!startTimestamp) startTimestamp = timestamp;
                const progress = Math.min((timestamp - startTimestamp) / duration, 1);
                const currentValue = progress * targetValue;
                counter.textContent = currentValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");

                if (progress < 1) {
                  window.requestAnimationFrame(step);
                } else {
                  counter.textContent = targetValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
                }
              }

              window.requestAnimationFrame(step);
            });
          </script>
        </div>

        <!-- Bonus Card -->
        <div class="card" id="bonus-card">
          <div class="card-icon-container">
            <i class="fas fa-gift fa-lg" style="color: white;"></i>
          </div>
          <div class="card-content">
            <div class="card-title">Bonus</div>
            <div class="card-value">$<span id="bonus-counter">0.00</span></div>
            <div class="card-badge">
              <i class="fas fa-caret-up"></i>
              <span id="bonusPercent">12.8%</span>
            </div>
          </div>

          <script>
            document.addEventListener('DOMContentLoaded', function () {
              const targetValue = <?= (float) $bonus ?>;
              const duration = 2000; // 2 seconds for the animation
              const counter = document.getElementById('bonus-counter');
              let startTimestamp = null;

              function step(timestamp) {
                if (!startTimestamp) startTimestamp = timestamp;
                const progress = Math.min((timestamp - startTimestamp) / duration, 1);
                const currentValue = progress * targetValue;
                counter.textContent = currentValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");

                if (progress < 1) {
                  window.requestAnimationFrame(step);
                } else {
                  counter.textContent = targetValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
                }
              }

              window.requestAnimationFrame(step);
            });
          </script>
        </div>

        <!-- Stakes Card -->
        <div class="card stake-card" id="stakeCardContainer">
          <div class="card-icon-container">
            <i class="fas fa-lock fa-lg" style="color: white;"></i>
          </div>
          <div class="card-content">
            <div class="card-title">Stakes</div>
            <div class="card-value">$<span id="total-counter">0.00</span></div>
            <div class="card-badge" id="stakePercentContainer">
              <i class="fas fa-caret-up"></i>
              <span id="stakePercent">(5%)</span>
            </div>
          </div>
        </div>
      </div>
    </div>

    <!-- Navigation buttons for mobile -->
    <button class="nav-button prev" onclick="prevSlide()" id="prevBtn">
      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M15 18l-6-6 6-6" stroke="#6B46C1" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
      </svg>
    </button>
    <button class="nav-button next" onclick="nextSlide()" id="nextBtn">
      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M9 18l6-6-6-6" stroke="#6B46C1" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
      </svg>
    </button>

    <!-- Dots indicator for mobile -->
    <div class="dots-container" id="dotsContainer">
      <div class="dot active" onclick="goToSlide(0)"></div>
      <div class="dot" onclick="goToSlide(1)"></div>
      <div class="dot" onclick="goToSlide(2)"></div>
    </div>
  </div>

  <?php
    // Ensure the user is logged in
    if (!isset($_SESSION['user_email'])) {
        echo "<div class='alert alert-warning'>User is not logged in or email is not set in session.</div>";
        exit();
    }
    $userEmail = $_SESSION['user_email'];

    // Database connection details
    $servername = "localhost";
    $username = "wptlgmnb_userp";
    $password = "Qliox4ITygj6Vh4V";
    $dbname = "wptlgmnb_dbp";

    // Create connection
    $conn = new mysqli($servername, $username, $password, $dbname);

    // Check connection
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }

    // Fetch total stakes amount from crypto_stakes table
    $totalAmount = 0;
    $totalQuery = "SELECT SUM(sessionAmount) as total FROM crypto_stakes WHERE email = ?";
    $stmt = $conn->prepare($totalQuery);
    $stmt->bind_param("s", $userEmail);
    $stmt->execute();
    $resultTotal = $stmt->get_result();
    if ($row = $resultTotal->fetch_assoc()) {
        $totalAmount = $row['total'] ?? 0;
    }
    $stmt->close();

    // Fetch all stakes for this user
    $stakes = [];
    $stakesQuery = "SELECT * FROM crypto_stakes WHERE email = ? ORDER BY id DESC";
    $stmt = $conn->prepare($stakesQuery);
    $stmt->bind_param("s", $userEmail);
    $stmt->execute();
    $stakesResult = $stmt->get_result();
    while ($row = $stakesResult->fetch_assoc()) {
        $stakes[] = $row;
    }
    $stmt->close();

    // Build the stakes content as an HTML string for SweetAlert
    ob_start();
    ?>
  <div class="stakes-container grid gap-4">
    <?php if (count($stakes) > 0): ?>
    <?php foreach ($stakes as $stake): ?>
    <?php
                // Prepare variables for each stake record
                $stakeId         = $stake['id'];
                $sessionType     = $stake['sessionType'];
                $sessionCurrency = $stake['sessionCurrency'];
                $sessionAmount   = number_format((float)$stake['sessionAmount'], 2, '.', ',');
                $convertedAmount = number_format((float)$stake['convertedAmount'], 2, '.', ',');
                $status          = $stake['status'];

                // Set text color based on sessionType: sell in red, buy in green
                $typeClass = "";
                if (strtolower($sessionType) === "sell") {
                    $typeClass = "text-red-500 font-bold";
                } else if (strtolower($sessionType) === "buy") {
                    $typeClass = "text-green-500 font-bold";
                } else {
                    $typeClass = "font-bold";
                }

                // Fetch current USD price from CoinGecko based on sessionCurrency
                $coinPriceUSD = null;
                $coinId = strtolower($sessionCurrency);
                $coinGeckoUrl = "https://api.coingecko.com/api/v3/simple/price?ids=" . urlencode($coinId) . "&vs_currencies=usd";
                $response = @file_get_contents($coinGeckoUrl);
                if ($response !== false) {
                    $priceData = json_decode($response, true);
                    if (isset($priceData[$coinId]['usd'])) {
                        $coinPriceUSD = number_format($priceData[$coinId]['usd'], 2, '.', ',');
                    }
                }

                // Set conditional styling for status:
                $statusLower = strtolower($status);
                $statusClass = 'text-gray-900';
                if (in_array($statusLower, ['pending', 'warning', 'lost'])) {
                    $statusClass = 'text-red-500 font-bold';
                } else if ($statusLower === 'success') {
                    $statusClass = 'text-green-500 font-bold';
                }
                ?>
    <a href="stake_detail.php?id=<?php echo $stakeId; ?>" class="neumorphic-card p-4 flex flex-col justify-between"
      style="display:block; margin-bottom: 10px; background: #f0f4f8; border: 1px solid #e2e8f0; border-radius: 8px;">
      <div class="flex justify-between w-full">
        <!-- Left column: stake details -->
        <div class="flex flex-col text-left">
          <p class="text-gray-900 text-sm"><strong>Type:</strong> <span class="<?php echo $typeClass; ?>">
              <?php echo $sessionType; ?>
            </span></p>
          <p class="text-gray-900 text-sm"><strong>Amount in USD:</strong>
            <?php echo' ' . $sessionAmount; ?>
          </p>
        </div>
        <!-- Right column: converted amount, status, and coin price -->
        <div class="flex flex-col text-right">
          <p class="text-gray-900 text-sm">
            <?php echo $convertedAmount . ' ' .  $sessionCurrency; ?>
          </p>
          <p class="text-sm <?php echo $statusClass; ?>"><strong>Status:</strong>
            <?php echo $status; ?>
          </p>
          <?php if ($coinPriceUSD !== null): ?>
          <p class="text-sm text-blue-600"><strong>Current Price:</strong>
            <?php echo "$" . $coinPriceUSD; ?>
          </p>
          <?php else: ?>
          <p class="text-sm text-gray-500"><strong>Current Price:</strong> N/A</p>
          <?php endif; ?>
        </div>
      </div>
      <!-- Optional: Display note if available -->
      <?php if(isset($stake['note'])): ?>
      <p class="text-gray-900 text-sm mt-2"><strong>Note:</strong>
        <?php echo $stake['note']; ?>
      </p>
      <?php endif; ?>
    </a>
    <?php endforeach; ?>
    <?php else: ?>
    <p class="text-sm text-gray-500">No stakes found for this account.</p>
    <?php endif; ?>
  </div>
  <?php
    $modalContent = ob_get_clean();
    $conn->close();
    ?>

  <!-- SweetAlert2 Library -->
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <script>
  let currentSlide = 0;
  const totalSlides = 3;
  const sliderTrack = document.getElementById('sliderTrack');
  const dots = document.querySelectorAll('.dot');

  // Counter animation for stakes total
  document.addEventListener('DOMContentLoaded', function () {
    const targetValue = <?php echo (float)$totalAmount; ?>;
    const duration = 2000;
    const counter = document.getElementById('total-counter');
    let startTimestamp = null;

    function step(timestamp) {
      if (!startTimestamp) startTimestamp = timestamp;
      const progress = Math.min((timestamp - startTimestamp) / duration, 1);
      const currentValue = progress * targetValue;
      counter.textContent = currentValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");

      if (progress < 1) {
        window.requestAnimationFrame(step);
      } else {
        counter.textContent = targetValue.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
      }
    }

    window.requestAnimationFrame(step);
  });

  // Slider functions
  function updateSlider() {
    if (window.innerWidth <= 768) {
      const slideWidth = 66.67; // Show 1.5 cards on mobile
      sliderTrack.style.transform = `translateX(-${currentSlide * slideWidth}%)`;

      // Update dots
      dots.forEach((dot, index) => {
        dot.classList.toggle('active', index === currentSlide);
      });
    }
  }

  function nextSlide() {
    if (currentSlide < totalSlides - 1) {
      currentSlide++;
      updateSlider();
    }
  }

  function prevSlide() {
    if (currentSlide > 0) {
      currentSlide--;
      updateSlider();
    }
  }

  function goToSlide(index) {
    currentSlide = index;
    updateSlider();
  }

  // Handle window resize
  window.addEventListener('resize', function () {
    if (window.innerWidth > 768) {
      sliderTrack.style.transform = 'translateX(0)';
      currentSlide = 0;
    } else {
      updateSlider();
    }
  });

  // Card click handlers with original functionality
  document.addEventListener('DOMContentLoaded', function () {
    // Profit card click handler
    const profitCard = document.getElementById('profit-card');
    profitCard.addEventListener('click', function () {
      Swal.fire({
        title: 'Profit Details',
        html: 'Your profit is generated from successful trades and staking returns. Every smart investment and timely market move contributes to your overall earnings. Keep up the great work!',
        icon: 'info',
        confirmButtonText: 'Got it!'
      });
    });

    // Bonus card click handler
    const bonusCard = document.getElementById('bonus-card');
    bonusCard.addEventListener('click', function () {
      Swal.fire({
        title: 'Crypto Bonus Earned!',
        html: 'You have earned this bonus by actively participating in our crypto trading and staking activities. Every successful referral and promotion adds to your bonus, rewarding your commitment and loyalty. Enjoy your bonus as a token of our appreciation!',
        icon: 'info',
        confirmButtonText: 'Awesome!'
      });
    });

    // Stakes card modal functionality
    const stakeCard = document.getElementById('stakeCardContainer');

    stakeCard.addEventListener('click', function (e) {
      e.preventDefault();

      Swal.fire({
        title: 'Your Stakes',
        html: <?php echo json_encode($modalContent); ?>,
        width: '80%',
        padding: '2em',
        showCloseButton: true,
        showConfirmButton: false,
        customClass: {
          container: 'stakes-modal-container',
          popup: 'stakes-modal-popup',
          content: 'stakes-modal-content'
        },
        didOpen: () => {
          const detailLinks = document.querySelectorAll('.stakes-modal-content a');
          detailLinks.forEach(link => {
            link.addEventListener('click', function (e) {
              e.stopPropagation();
            });
          });
        }
      });
    });

    // Update stake percentage display
    const stakeData = {
      percentage: 5,
      isPositive: true
    };

    const stakePercentElement = document.getElementById('stakePercent');
    const stakePercentContainer = document.getElementById('stakePercentContainer');

    stakePercentElement.textContent = `(${stakeData.percentage}%)`;

    if (stakeData.isPositive) {
      stakePercentContainer.classList.add('text-green-500');
      stakePercentContainer.classList.remove('text-red-500');
      stakePercentContainer.querySelector('i').className = 'fas fa-caret-up';
    } else {
      stakePercentContainer.classList.add('text-red-500');
      stakePercentContainer.classList.remove('text-green-500');
      stakePercentContainer.querySelector('i').className = 'fas fa-caret-down';
    }
  });

  // Initialize
  updateSlider();
</script>





  <script src="https://cdn.tailwindcss.com"></script>
  <style>
    @keyframes shimmer {
      0% {
        background-position: -200px 0;
      }

      100% {
        background-position: calc(200px + 100%) 0;
      }
    }

    .shimmer {
      background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
      background-size: 200px 100%;
      animation: shimmer 1.5s infinite;
    }

    .gradient-bg {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    }

    .card-hover {
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }

    .card-hover:hover {
      transform: translateY(-8px);
      box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
    }

    .scroll-smooth {
      scroll-behavior: smooth;
    }

    .market-badge {
      background: linear-gradient(45deg, #12c2e9, #c471ed, #f64f59);
      background-size: 300% 300%;
      animation: gradient-shift 3s ease infinite;
    }

    @keyframes gradient-shift {
      0% {
        background-position: 0% 50%;
      }

      50% {
        background-position: 100% 50%;
      }

      100% {
        background-position: 0% 50%;
      }
    }

    .sparkline-container {
      position: relative;
      overflow: hidden;
    }

    .price-indicator {
      position: relative;
    }

    .price-indicator::before {
      content: '';
      position: absolute;
      top: 50%;
      left: -8px;
      width: 4px;
      height: 4px;
      border-radius: 50%;
      transform: translateY(-50%);
    }

    .price-up::before {
      background-color: #16a34a;
      animation: pulse-green 2s infinite;
    }

    .price-down::before {
      background-color: #dc2626;
      animation: pulse-red 2s infinite;
    }

    @keyframes pulse-green {

      0%,
      100% {
        opacity: 1;
        box-shadow: 0 0 0 0 rgba(22, 163, 74, 0.7);
      }

      50% {
        opacity: 0.7;
        box-shadow: 0 0 0 10px rgba(22, 163, 74, 0);
      }
    }

    @keyframes pulse-red {

      0%,
      100% {
        opacity: 1;
        box-shadow: 0 0 0 0 rgba(220, 38, 38, 0.7);
      }

      50% {
        opacity: 0.7;
        box-shadow: 0 0 0 10px rgba(220, 38, 38, 0);
      }
    }
  </style>


  <div class="container mx-auto px-4 py-8">
    <!-- Header Section -->


    <!-- Skeleton Loader -->
    <div id="financeSkeletonLoader" class="flex gap-6 overflow-x-auto pb-6 snap-x snap-mandatory">
      <!-- Generate 5 skeleton cards -->
      <div class="flex-shrink-0 w-full sm:w-80 bg-white rounded-2xl p-6 shadow-lg snap-center animate-pulse">
        <div class="flex items-center gap-4 mb-6">
          <div class="w-12 h-12 bg-gray-300 rounded-full shimmer"></div>
          <div class="flex-1 space-y-3">
            <div class="h-5 bg-gray-300 rounded-lg w-32 shimmer"></div>
            <div class="h-4 bg-gray-300 rounded w-20 shimmer"></div>
          </div>
          <div class="w-16 h-8 bg-gray-300 rounded-full shimmer"></div>
        </div>
        <div class="space-y-4 mb-6">
          <div class="h-8 bg-gray-300 rounded-lg w-40 shimmer"></div>
          <div class="h-5 bg-gray-300 rounded w-24 shimmer"></div>
        </div>
        <div class="h-12 bg-gray-300 rounded-lg shimmer"></div>
      </div>
      <div class="flex-shrink-0 w-full sm:w-80 bg-white rounded-2xl p-6 shadow-lg snap-center animate-pulse">
        <div class="flex items-center gap-4 mb-6">
          <div class="w-12 h-12 bg-gray-300 rounded-full shimmer"></div>
          <div class="flex-1 space-y-3">
            <div class="h-5 bg-gray-300 rounded-lg w-32 shimmer"></div>
            <div class="h-4 bg-gray-300 rounded w-20 shimmer"></div>
          </div>
          <div class="w-16 h-8 bg-gray-300 rounded-full shimmer"></div>
        </div>
        <div class="space-y-4 mb-6">
          <div class="h-8 bg-gray-300 rounded-lg w-40 shimmer"></div>
          <div class="h-5 bg-gray-300 rounded w-24 shimmer"></div>
        </div>
        <div class="h-12 bg-gray-300 rounded-lg shimmer"></div>
      </div>
      <div class="flex-shrink-0 w-full sm:w-80 bg-white rounded-2xl p-6 shadow-lg snap-center animate-pulse">
        <div class="flex items-center gap-4 mb-6">
          <div class="w-12 h-12 bg-gray-300 rounded-full shimmer"></div>
          <div class="flex-1 space-y-3">
            <div class="h-5 bg-gray-300 rounded-lg w-32 shimmer"></div>
            <div class="h-4 bg-gray-300 rounded w-20 shimmer"></div>
          </div>
          <div class="w-16 h-8 bg-gray-300 rounded-full shimmer"></div>
        </div>
        <div class="space-y-4 mb-6">
          <div class="h-8 bg-gray-300 rounded-lg w-40 shimmer"></div>
          <div class="h-5 bg-gray-300 rounded w-24 shimmer"></div>
        </div>
        <div class="h-12 bg-gray-300 rounded-lg shimmer"></div>
      </div>
      <div class="flex-shrink-0 w-full sm:w-80 bg-white rounded-2xl p-6 shadow-lg snap-center animate-pulse">
        <div class="flex items-center gap-4 mb-6">
          <div class="w-12 h-12 bg-gray-300 rounded-full shimmer"></div>
          <div class="flex-1 space-y-3">
            <div class="h-5 bg-gray-300 rounded-lg w-32 shimmer"></div>
            <div class="h-4 bg-gray-300 rounded w-20 shimmer"></div>
          </div>
          <div class="w-16 h-8 bg-gray-300 rounded-full shimmer"></div>
        </div>
        <div class="space-y-4 mb-6">
          <div class="h-8 bg-gray-300 rounded-lg w-40 shimmer"></div>
          <div class="h-5 bg-gray-300 rounded w-24 shimmer"></div>
        </div>
        <div class="h-12 bg-gray-300 rounded-lg shimmer"></div>
      </div>
      <div class="flex-shrink-0 w-full sm:w-80 bg-white rounded-2xl p-6 shadow-lg snap-center animate-pulse">
        <div class="flex items-center gap-4 mb-6">
          <div class="w-12 h-12 bg-gray-300 rounded-full shimmer"></div>
          <div class="flex-1 space-y-3">
            <div class="h-5 bg-gray-300 rounded-lg w-32 shimmer"></div>
            <div class="h-4 bg-gray-300 rounded w-20 shimmer"></div>
          </div>
          <div class="w-16 h-8 bg-gray-300 rounded-full shimmer"></div>
        </div>
        <div class="space-y-4 mb-6">
          <div class="h-8 bg-gray-300 rounded-lg w-40 shimmer"></div>
          <div class="h-5 bg-gray-300 rounded w-24 shimmer"></div>
        </div>
        <div class="h-12 bg-gray-300 rounded-lg shimmer"></div>
      </div>
    </div>

    <!-- Slider Container -->
    <div id="financeSliderContainer" class="relative overflow-hidden group hidden">
      <!-- Slider Track -->
      <div id="financeSliderTrack" class="flex gap-6 overflow-x-auto scroll-smooth pb-6 snap-x snap-mandatory">
        <!-- Financial cards will be injected here -->
      </div>

      <!-- Navigation Buttons -->
      <button id="financePrevBtn"
        class="absolute left-4 top-1/2 -translate-y-1/2 bg-white p-4 rounded-full shadow-xl opacity-0 group-hover:opacity-100 transition-all duration-300 hover:scale-110 z-10"
        aria-label="Previous Slide">
        <svg class="w-6 h-6 text-gray-700" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
        </svg>
      </button>
      <button id="financeNextBtn"
        class="absolute right-4 top-1/2 -translate-y-1/2 bg-white p-4 rounded-full shadow-xl opacity-0 group-hover:opacity-100 transition-all duration-300 hover:scale-110 z-10"
        aria-label="Next Slide">
        <svg class="w-6 h-6 text-gray-700" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
        </svg>
      </button>

      <!-- Slide Indicators -->
      <div id="financeIndicators" class="flex justify-center gap-2 mt-6">
        <!-- Indicators will be added dynamically -->
      </div>
    </div>

    <!-- Market Status -->
    <div id="financeMarketStatus" class="text-center mt-8 hidden">
      <div class="inline-flex items-center gap-2 bg-white rounded-full px-4 py-2 shadow-md">
        <div class="w-3 h-3 bg-green-500 rounded-full animate-pulse"></div>
        <span class="text-sm font-medium text-gray-700">My Wallet Balance</span>
        <span id="financeLastUpdate" class="text-xs text-gray-500"></span>
      </div>

      <?php
// Function to fetch and cache cryptocurrency data
function getCryptoData($crypto_id, $cache_file) {
    $cache_duration = 300; // 5 minutes cache
    
    // Use cached data if available and not expired
    if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_duration) {
        return json_decode(file_get_contents($cache_file), true);
    } else {
        // Fetch fresh data from API
        $api_url = "https://api.coingecko.com/api/v3/simple/price?ids={$crypto_id}&vs_currencies=usd&include_24hr_change=true";
        $response = @file_get_contents($api_url);
        
        if ($response !== false) {
            $data = json_decode($response, true);
            if (!empty($data) && isset($data[$crypto_id]['usd'])) {
                // Cache the successful response
                file_put_contents($cache_file, json_encode($data));
                return $data;
            }
        }
        
        // If API call fails, use cached data if available
        if (file_exists($cache_file)) {
            return json_decode(file_get_contents($cache_file), true);
        } else {
            // Create a fallback if no cache exists
            $fallback = [];
            $fallback[$crypto_id]['usd'] = 0;
            $fallback[$crypto_id]['usd_24h_change'] = 0;
            return $fallback;
        }
    }
}

// ==================== Bitcoin Card ====================
$btc_data = getCryptoData('bitcoin', 'btc_price_cache.json');
$btc_price_usd = isset($btc_data['bitcoin']['usd']) ? (float)$btc_data['bitcoin']['usd'] : 0;
$btc_change = isset($btc_data['bitcoin']['usd_24h_change']) ? (float)$btc_data['bitcoin']['usd_24h_change'] : 0;
$btc_amount = ($btc_price_usd > 0) ? $btc / $btc_price_usd : 0;
$btc_change_formatted = number_format($btc_change, 2);
$btc_change_color = ($btc_change >= 0) ? "text-green-500" : "text-red-500";
?>

      <a href="detailedbtc.php">
        <div class="neumorphic-card p-4 flex items-center justify-between" id="btcCard">
          <div class="shimmer-effect"></div>
          <div class="flex items-center relative z-10">
            <img src="https://afdkryptostam.com/cryptos/bitcoin.png" alt="Bitcoin" class="w-10 h-10 mr-4">
            <div>
              <p class="font-bold" style="color: #3d4863;">Bitcoin (BTC)</p>
              <p class="text-gray-900 text-sm">
                <?php echo number_format($btc_amount, 6) . ' BTC'; ?>
              </p>
            </div>
          </div>
          <div class="text-right relative z-10">
            <p class="font-semibold" id="btcPrice">$
              <?php echo number_format($btc_price_usd, 2); ?>
            </p>
            <p class="<?php echo $btc_change_color; ?> text-sm" id="btcChange">
              <?php echo ($btc_change >= 0 ? '+' : '') . $btc_change_formatted . '%'; ?>
            </p>
          </div>
        </div>
      </a>

      <?php
// ==================== Ethereum Card ====================
$eth_data = getCryptoData('ethereum', 'eth_price_cache.json');
$eth_price_usd = isset($eth_data['ethereum']['usd']) ? (float)$eth_data['ethereum']['usd'] : 0;
$eth_change = isset($eth_data['ethereum']['usd_24h_change']) ? (float)$eth_data['ethereum']['usd_24h_change'] : 0;
$eth_amount = ($eth_price_usd > 0) ? $eth / $eth_price_usd : 0;
$eth_change_formatted = number_format($eth_change, 2);
$eth_change_color = ($eth_change >= 0) ? "text-green-500" : "text-red-500";
?>

      <a href="detailedeth.php">
        <div class="neumorphic-card p-4 flex items-center justify-between" id="ethereumCard">
          <div class="shimmer-effect"></div>
          <div class="flex items-center relative z-10">
            <img src="https://coin-images.coingecko.com/coins/images/279/large/ethereum.png?1696501628" alt="Ethereum"
              class="w-10 h-10 mr-4">
            <div>
              <p class="font-bold" style="color: #3d4863;">Ethereum (ETH)</p>
              <p class="text-gray-900 text-sm">
                <?php echo number_format($eth_amount, 6) . ' ETH'; ?>
              </p>
            </div>
          </div>
          <div class="text-right relative z-10">
            <p class="font-semibold" id="ethPrice">$
              <?php echo number_format($eth_price_usd, 2); ?>
            </p>
            <p class="<?php echo $eth_change_color; ?> text-sm" id="ethChange">
              <?php echo ($eth_change >= 0 ? '+' : '') . $eth_change_formatted . '%'; ?>
            </p>
          </div>
        </div>
      </a>

      <?php
// ==================== Tether Card ====================
$usdt_data = getCryptoData('tether', 'usdt_price_cache.json');
$usdt_price_usd = isset($usdt_data['tether']['usd']) ? (float)$usdt_data['tether']['usd'] : 1; // Default to 1 for stablecoins
$usdt_change = isset($usdt_data['tether']['usd_24h_change']) ? (float)$usdt_data['tether']['usd_24h_change'] : 0;
$usdt_amount = ($usdt_price_usd > 0) ? $usdt / $usdt_price_usd : 0;
$usdt_change_formatted = number_format($usdt_change, 2);
$usdt_change_color = ($usdt_change >= 0) ? "text-green-500" : "text-red-500";
?>

      <a href="detailedusdt.php">
        <div class="neumorphic-card p-4 flex items-center justify-between" id="usdtCard">
          <div class="shimmer-effect"></div>
          <div class="flex items-center relative z-10">
            <img src="https://coin-images.coingecko.com/coins/images/325/large/Tether.png?1696501661" alt="Tether"
              class="w-10 h-10 mr-4">
            <div>
              <p class="font-bold" style="color: #3d4863;">Tether (USDT)</p>
              <p class="text-gray-900 text-sm">
                <?php echo number_format($usdt_amount, 6) . ' USDT'; ?>
              </p>
            </div>
          </div>
          <div class="text-right relative z-10">
            <p class="font-semibold" id="usdtPrice">$
              <?php echo number_format($usdt_price_usd, 4); ?>
            </p>
            <p class="<?php echo $usdt_change_color; ?> text-sm" id="usdtChange">
              <?php echo ($usdt_change >= 0 ? '+' : '') . $usdt_change_formatted . '%'; ?>
            </p>
          </div>
        </div>
      </a>


      <?php
// ==================== XRP Card ====================
$xrp_data = getCryptoData('ripple', 'xrp_price_cache.json');
$xrp_price_usd = isset($xrp_data['ripple']['usd']) ? (float)$xrp_data['ripple']['usd'] : 0;
$xrp_change = isset($xrp_data['ripple']['usd_24h_change']) ? (float)$xrp_data['ripple']['usd_24h_change'] : 0;
$xrp_amount = ($xrp_price_usd > 0) ? $xrp / $xrp_price_usd : 0;
$xrp_change_formatted = number_format($xrp_change, 2);
$xrp_change_color = ($xrp_change >= 0) ? "text-green-500" : "text-red-500";
?>

      <a href="detailxrp.php">
        <div class="neumorphic-card p-4 flex items-center justify-between" id="xrpCard">
          <div class="shimmer-effect"></div>
          <div class="flex items-center relative z-10">
            <img src="https://primemarketspro.com/cryptos/xrp.png" alt="XRP" class="w-10 h-10 mr-4">
            <div>
              <p class="font-bold" style="color: #3d4863;">XRP (XRP)</p>
              <p class="text-gray-900 text-sm">
                <?php echo number_format($xrp_amount, 6) . ' XRP'; ?>
              </p>
            </div>
          </div>
          <div class="text-right relative z-10">
            <p class="font-semibold" id="xrpPrice">$
              <?php echo number_format($xrp_price_usd, 4); ?>
            </p>
            <p class="<?php echo $xrp_change_color; ?> text-sm" id="xrpChange">
              <?php echo ($xrp_change >= 0 ? '+' : '') . $xrp_change_formatted . '%'; ?>
            </p>
          </div>
        </div>
      </a>

    </div>


    <script>
      // Enhanced Financial Data Fetcher with Real API Integration
      class FinancialDataFetcher {
        constructor() {
          // API Configuration
          this.apis = {
            // Free APIs (no key required)
            coinGecko: 'https://api.coingecko.com/api/v3',
            exchangeRate: 'https://api.exchangerate-api.com/v4/latest',

            // APIs requiring keys (replace with your actual keys)
            alphaVantage: {
              base: 'https://www.alphavantage.co/query',
              key: 'YOUR_ALPHA_VANTAGE_API_KEY' // Get free key from alphavantage.co
            },
            finnhub: {
              base: 'https://finnhub.io/api/v1',
              key: 'YOUR_FINNHUB_API_KEY' // Get free key from finnhub.io
            },
            polygon: {
              base: 'https://api.polygon.io/v2',
              key: 'YOUR_POLYGON_API_KEY' // Get free key from polygon.io
            }
          };

          // Cache settings
          this.cache = new Map();
          this.cacheTimeout = 60000; // 1 minute

          // Rate limiting
          this.lastRequestTime = 0;
          this.minRequestInterval = 200; // 200ms between requests
        }

        // Rate limiting helper
        async rateLimitDelay() {
          const now = Date.now();
          const timeSinceLastRequest = now - this.lastRequestTime;
          if (timeSinceLastRequest < this.minRequestInterval) {
            await new Promise(resolve =>
              setTimeout(resolve, this.minRequestInterval - timeSinceLastRequest)
            );
          }
          this.lastRequestTime = Date.now();
        }

        // Generic fetch with error handling and caching
        async fetchWithCache(url, cacheKey, options = {}) {
          // Check cache first
          if (this.cache.has(cacheKey)) {
            const cached = this.cache.get(cacheKey);
            if (Date.now() - cached.timestamp < this.cacheTimeout) {
              return cached.data;
            }
          }

          try {
            await this.rateLimitDelay();

            const response = await fetch(url, {
              headers: {
                'Accept': 'application/json',
                'User-Agent': 'FinancialSlider/1.0',
                ...options.headers
              },
              ...options
            });

            if (!response.ok) {
              throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();

            // Cache the result
            this.cache.set(cacheKey, {
              data,
              timestamp: Date.now()
            });

            return data;
          } catch (error) {
            console.error(`Fetch error for ${url}:`, error);
            throw error;
          }
        }

        // Fetch cryptocurrency data
        async fetchCryptocurrencyData() {
          try {
            // Using CoinGecko API (free, no key required)
            const cryptoData = await this.fetchWithCache(
              `${this.apis.coinGecko}/simple/price?ids=bitcoin,ethereum,cardano,polygon,solana&vs_currencies=usd&include_24hr_change=true&include_24hr_vol=true`,
              'crypto_prices'
            );

            // Get historical data for sparklines
            const sparklinePromises = ['bitcoin', 'ethereum'].map(async (id) => {
              const historical = await this.fetchWithCache(
                `${this.apis.coinGecko}/coins/${id}/market_chart?vs_currency=usd&days=1&interval=hourly`,
                `crypto_sparkline_${id}`
              );
              return { id, prices: historical.prices.map(p => p[1]) };
            });

            const sparklineData = await Promise.all(sparklinePromises);
            const sparklines = Object.fromEntries(sparklineData.map(s => [s.id, s.prices.slice(-8)]));

            return [
              {
                type: 'crypto',
                symbol: 'BTC/USD',
                name: 'Bitcoin',
                price: cryptoData.bitcoin.usd,
                change: cryptoData.bitcoin.usd_24h_change,
                icon: '₿',
                color: '#f7931a',
                sparkline: sparklines.bitcoin || []
              },
              {
                type: 'crypto',
                symbol: 'ETH/USD',
                name: 'Ethereum',
                price: cryptoData.ethereum.usd,
                change: cryptoData.ethereum.usd_24h_change,
                icon: '⧫',
                color: '#627eea',
                sparkline: sparklines.ethereum || []
              }
            ];
          } catch (error) {
            console.error('Error fetching cryptocurrency data:', error);
            return this.getFallbackCryptoData();
          }
        }

        // Fetch stock data
        async fetchStockData() {
          try {
            // Using Alpha Vantage API (requires free API key)
            if (this.apis.alphaVantage.key === 'YOUR_ALPHA_VANTAGE_API_KEY') {
              throw new Error('Alpha Vantage API key not configured');
            }

            const symbols = ['NVDA', 'TSLA', 'AAPL', 'MSFT'];
            const stockPromises = symbols.map(async (symbol) => {
              const data = await this.fetchWithCache(
                `${this.apis.alphaVantage.base}?function=GLOBAL_QUOTE&symbol=${symbol}&apikey=${this.apis.alphaVantage.key}`,
                `stock_${symbol}`
              );

              // Get intraday data for sparkline
              const intraday = await this.fetchWithCache(
                `${this.apis.alphaVantage.base}?function=TIME_SERIES_INTRADAY&symbol=${symbol}&interval=1hour&apikey=${this.apis.alphaVantage.key}`,
                `stock_intraday_${symbol}`
              );

              const quote = data['Global Quote'];
              const timeSeriesData = intraday['Time Series (1hour)'];
              const sparkline = Object.values(timeSeriesData || {})
                .slice(0, 8)
                .map(entry => parseFloat(entry['4. close']));

              return {
                type: 'stock',
                symbol: symbol,
                name: this.getStockName(symbol),
                price: parseFloat(quote['05. price']),
                change: parseFloat(quote['10. change percent'].replace('%', '')),
                icon: this.getStockIcon(symbol),
                color: this.getStockColor(symbol),
                sparkline: sparkline.reverse()
              };
            });

            return await Promise.all(stockPromises);
          } catch (error) {
            console.error('Error fetching stock data:', error);
            return this.getFallbackStockData();
          }
        }

        // Fetch forex data
        async fetchForexData() {
          try {
            // Using Exchange Rate API (free, no key required)
            const forexData = await this.fetchWithCache(
              `${this.apis.exchangeRate}/USD`,
              'forex_rates'
            );

            // Get historical rates for sparkline (using a different free API)
            const historicalData = await this.fetchWithCache(
              'https://api.fxratesapi.com/historical?start_date=2024-01-01&end_date=2024-01-08&base=USD&currencies=EUR,GBP,JPY',
              'forex_historical'
            );

            return [
              {
                type: 'forex',
                symbol: 'EUR/USD',
                name: 'Euro Dollar',
                price: 1 / forexData.rates.EUR,
                change: this.calculateForexChange(historicalData, 'EUR'),
                icon: '💱',
                color: '#1f77b4',
                sparkline: this.getForexSparkline(historicalData, 'EUR')
              },
              {
                type: 'forex',
                symbol: 'GBP/USD',
                name: 'British Pound',
                price: 1 / forexData.rates.GBP,
                change: this.calculateForexChange(historicalData, 'GBP'),
                icon: '£',
                color: '#2ca02c',
                sparkline: this.getForexSparkline(historicalData, 'GBP')
              }
            ];
          } catch (error) {
            console.error('Error fetching forex data:', error);
            return this.getFallbackForexData();
          }
        }

        // Main function to fetch all financial data
        async fetchAllFinancialData() {
          try {
            console.log('Fetching financial data from APIs...');

            const [cryptoDataSettled, stockDataSettled, forexDataSettled] = await Promise.allSettled([
              this.fetchCryptocurrencyData(),
              this.fetchStockData(),
              this.fetchForexData()
            ]);

            const allData = [];

            if (cryptoDataSettled.status === 'fulfilled') {
              allData.push(...cryptoDataSettled.value);
            }

            if (stockDataSettled.status === 'fulfilled') {
              allData.push(...stockDataSettled.value);
            }

            if (forexDataSettled.status === 'fulfilled') {
              allData.push(...forexDataSettled.value);
            }

            // If no data was fetched successfully, use fallback
            if (allData.length === 0) {
              console.warn('All API requests failed, using fallback data');
              return this.getFallbackData();
            }

            console.log(`Successfully fetched ${allData.length} financial instruments`);
            return allData;

          } catch (error) {
            console.error('Error in fetchAllFinancialData:', error);
            return this.getFallbackData();
          }
        }

        // Helper functions
        getStockName(symbol) {
          const names = {
            'NVDA': 'NVIDIA Corp',
            'TSLA': 'Tesla Inc',
            'AAPL': 'Apple Inc',
            'MSFT': 'Microsoft Corp'
          };
          return names[symbol] || symbol;
        }

        getStockIcon(symbol) {
          const icons = {
            'NVDA': '🔧',
            'TSLA': '🚗',
            'AAPL': '🍎',
            'MSFT': '💻'
          };
          return icons[symbol] || '📈';
        }

        getStockColor(symbol) {
          const colors = {
            'NVDA': '#76b900',
            'TSLA': '#cc0000',
            'AAPL': '#007aff',
            'MSFT': '#00bcf2'
          };
          return colors[symbol] || '#666666';
        }

        calculateForexChange(historicalData, currency) {
          if (!historicalData || !historicalData.rates) return 0;

          const ratesArray = Object.values(historicalData.rates);
          if (ratesArray.length < 2) return 0;

          const latestRate = ratesArray[ratesArray.length - 1][currency];
          const previousRate = ratesArray[0][currency];

          return ((latestRate - previousRate) / previousRate) * 100;
        }

        getForexSparkline(historicalData, currency) {
          if (!historicalData || !historicalData.rates) return [];

          return Object.values(historicalData.rates)
            .map(rateObj => 1 / rateObj[currency])
            .slice(-8);
        }

        // Fallback data methods
        getFallbackCryptoData() {
          return [
            {
              type: 'crypto',
              symbol: 'BTC/USD',
              name: 'Bitcoin',
              price: <?= htmlspecialchars($btc) ?>,
              change: 8.45,
              icon: '₿',
              color: '#f7931a',
              sparkline: [65000, 66200, 67100, 66800, 67500, 67850, 68200, 67850]
            },
            {
              type: 'crypto',
              symbol: 'ETH/USD',
              name: 'Ethereum',
              price: <?= htmlspecialchars($eth) ?>,
              change: 15.23,
              icon: '⧫',
              color: '#627eea',
              sparkline: [3200, 3250, 3300, 3280, 3350, 3400, 3450, 3425]
            }
          ];
        }

        getFallbackStockData() {
          return [
            {
              type: 'stock',
              symbol: 'NVDA',
              name: 'NVIDIA Corp',
              price: <?= htmlspecialchars($usdt) ?>,
              change: 12.67,
              icon: '🔧',
              color: '#76b900',
              sparkline: [850, 860, 870, 865, 872, 875, 880, 875]
            },
            {
              type: 'stock',
              symbol: 'TSLA',
              name: 'Tesla Inc',
              price: <?= htmlspecialchars($pepe) ?>,
              change: -5.67,
              icon: '🚗',
              color: '#cc0000',
              sparkline: [250, 248, 246, 247, 245, 244, 243, 246]
            }
          ];
        }

        getFallbackForexData() {
          return [
            {
              type: 'forex',
              symbol: 'EUR/USD',
              name: 'Euro Dollar',
              price: <?= htmlspecialchars($bnb) ?>,
              change: -2.34,
              icon: '💱',
              color: '#1f77b4',
              sparkline: [1.090, 1.088, 1.085, 1.087, 1.084, 1.082, 1.081, 1.084]
            }
          ];
        }

        getFallbackData() {
          return [
            ...this.getFallbackCryptoData(),
            ...this.getFallbackStockData(),
            ...this.getFallbackForexData()
          ];
        }
      }

      // Enhanced main application code
      let financeCurrentSlide = 0;
      let financeAutoSlideInterval;
      let financeDataFetcher;
      let financeFinancialData = [];

      // Initialize the financial movers slider with real API data
      async function initializeFinanceSlider() {
        try {
          // Initialize the data fetcher
          financeDataFetcher = new FinancialDataFetcher();

          // Show loading state
          const sliderTrackEl = document.getElementById('financeSliderTrack');
          const skeletonLoaderEl = document.getElementById('financeSkeletonLoader');
          const sliderContainerEl = document.getElementById('financeSliderContainer');
          const marketStatusEl = document.getElementById('financeMarketStatus');

          // Fetch real financial data
          console.log('Fetching real-time financial data...');
          financeFinancialData = await financeDataFetcher.fetchAllFinancialData();

          // Create financial cards
          sliderTrackEl.innerHTML = '';
          financeFinancialData.forEach((item, index) => {
            const cardHTML = createFinancialCard(item, index);
            sliderTrackEl.innerHTML += cardHTML;
          });

          // Create indicators
          createFinanceIndicators();

          // Hide skeleton and show slider
          skeletonLoaderEl.classList.add('hidden');
          sliderContainerEl.classList.remove('hidden');
          marketStatusEl.classList.remove('hidden');

          // Update last update time
          updateFinanceLastUpdateTime();

          // Initialize slider controls
          initFinanceSliderControls();

          // Start auto-slide
          startFinanceAutoSlide();

          // Set up periodic data refresh (every 5 minutes)
          setInterval(refreshFinanceData, 300000);

        } catch (error) {
          console.error('Error initializing finance slider:', error);
          // Fallback to sample data if everything fails
          financeFinancialData = financeDataFetcher ? financeDataFetcher.getFallbackData() : getSampleData();

          // Continue with initialization using fallback data
          const sliderTrackEl = document.getElementById('financeSliderTrack');
          sliderTrackEl.innerHTML = '';
          financeFinancialData.forEach((item, index) => {
            const cardHTML = createFinancialCard(item, index);
            sliderTrackEl.innerHTML += cardHTML;
          });

          createFinanceIndicators();
          document.getElementById('financeSkeletonLoader').classList.add('hidden');
          document.getElementById('financeSliderContainer').classList.remove('hidden');
          document.getElementById('financeMarketStatus').classList.remove('hidden');

          updateFinanceLastUpdateTime();
          initFinanceSliderControls();
          startFinanceAutoSlide();
        }
      }

      // Refresh data function
      async function refreshFinanceData() {
        try {
          console.log('Refreshing financial data...');
          const newData = await financeDataFetcher.fetchAllFinancialData();

          // Update the data
          financeFinancialData = newData;

          // Update the cards
          const sliderTrackEl = document.getElementById('financeSliderTrack');
          sliderTrackEl.innerHTML = '';
          financeFinancialData.forEach((item, index) => {
            const cardHTML = createFinancialCard(item, index);
            sliderTrackEl.innerHTML += cardHTML;
          });

          // Update indicators if number of items changed
          const existingIndicators = document.querySelectorAll('#financeIndicators button');
          if (existingIndicators.length !== financeFinancialData.length) {
            document.getElementById('financeIndicators').innerHTML = '';
            createFinanceIndicators();
          }

          // Update timestamp
          updateFinanceLastUpdateTime();

          console.log('Financial data refreshed successfully');
        } catch (error) {
          console.error('Error refreshing financial data:', error);
        }
      }

      // Create a single financial card's HTML
      function createFinancialCard(item, index) {
        const changeColorClass = item.change >= 0 ? 'text-green-500' : 'text-red-500';
        const changeBgClass = item.change >= 0 ? 'bg-green-50' : 'bg-red-50';
        const priceIndicatorClass = item.change >= 0 ? 'price-up' : 'price-down';
        const marketBadgeColorClass = getMarketBadgeColor(item.type);
        const sparklineSVG = generateSparkline(item.sparkline, item.change >= 0);

        // Price text formatting: forex uses 4 decimals, others use locale string with $
        const priceText = item.type === 'forex'
          ? item.price.toFixed(4)
          : '$' + item.price.toLocaleString();

        return `
            <div class="financial-card flex-shrink-0 w-full sm:w-80 bg-white rounded-2xl p-6 shadow-lg card-hover snap-center border border-gray-100" data-index="${index}">
                <!-- Header -->
                <div class="flex items-center justify-between mb-6">
                    <div class="flex items-center gap-4">
                        <div class="w-12 h-12 rounded-full flex items-center justify-center text-2xl" style="background-color: ${item.color}20; color: ${item.color};">
                            ${item.icon}
                        </div>
                        <div>
                            <h3 class="font-bold text-gray-900 text-lg">${item.symbol}</h3>
                            <p class="text-sm text-gray-600">${item.name}</p>
                        </div>
                    </div>
                    <span class="market-badge text-white text-xs font-bold px-3 py-1 rounded-full ${marketBadgeColorClass}">
                        ${item.type.toUpperCase()}
                    </span>
                </div>
                
                <!-- Price Section -->
                <div class="mb-6">
                    <div class="flex items-baseline gap-2 mb-2">
                        <p class="text-3xl font-bold text-gray-900 price-indicator ${priceIndicatorClass}">
                            ${priceText}
                        </p>
                    </div>
                    <div class="flex items-center gap-2">
                        <span class="${changeColorClass} ${changeBgClass} px-3 py-1 rounded-full text-sm font-semibold flex items-center gap-1">
                            ${item.change >= 0 ? '↗' : '↘'} ${Math.abs(item.change).toFixed(2)}%
                        </span>
                        <span class="text-xs text-gray-500">24h</span>
                    </div>
                </div>
                
                <!-- Sparkline -->
                <div class="h-12 sparkline-container">
                    ${sparklineSVG}
                </div>
            </div>
        `;
      }

      // Generate an inline SVG sparkline from an array of numerical prices
      function generateSparkline(prices, isPositive) {
        if (!prices || prices.length === 0) {
          return '<div class="w-full h-full bg-gray-100 rounded flex items-center justify-center text-gray-400 text-xs">No data</div>';
        }

        const svgWidth = 280;
        const svgHeight = 48;
        const maxPrice = Math.max(...prices);
        const minPrice = Math.min(...prices);
        const range = maxPrice - minPrice;

        if (range === 0) {
          return '<div class="w-full h-full bg-gray-100 rounded"></div>';
        }

        const normalized = prices.map(p => ((p - minPrice) / range) * (svgHeight - 8) + 4);

        const points = normalized.map((value, index) => {
          const x = (index / (prices.length - 1)) * svgWidth;
          const y = svgHeight - value;
          return `${x},${y}`;
        }).join(' ');

        const gradientId = `gradient-${Math.random().toString(36).substr(2, 9)}`;
        const strokeColor = isPositive ? '#16a34a' : '#dc2626';
        const fillColor = isPositive ? '#16a34a' : '#dc2626';

        return `
            <svg viewBox="0 0 ${svgWidth} ${svgHeight}" class="w-full h-full">
                <defs>
                    <linearGradient id="${gradientId}" x1="0%" y1="0%" x2="0%" y2="100%">
                        <stop offset="0%" style="stop-color:${fillColor};stop-opacity:0.3" />
                        <stop offset="100%" style="stop-color:${fillColor};stop-opacity:0.05" />
                    </linearGradient>
                </defs>
                <polygon
                    points="${points} ${svgWidth},${svgHeight} 0,${svgHeight}"
                    fill="url(#${gradientId})"
                />
                <polyline 
                    points="${points}"
                    fill="none" 
                    stroke="${strokeColor}"
                    stroke-width="3"
                    stroke-linecap="round"
                    stroke-linejoin="round"
                />
            </svg>
        `;
      }

      // Determine badge color gradient based on instrument type
      function getMarketBadgeColor(type) {
        switch (type) {
          case 'crypto':
            return 'bg-gradient-to-r from-orange-400 to-yellow-400';
          case 'stock':
            return 'bg-gradient-to-r from-blue-400 to-purple-400';
          case 'forex':
            return 'bg-gradient-to-r from-green-400 to-blue-400';
          default:
            return 'bg-gradient-to-r from-gray-400 to-gray-600';
        }
      }

      // Create slide indicators dynamically
      function createFinanceIndicators() {
        const indicatorsEl = document.getElementById('financeIndicators');
        indicatorsEl.innerHTML = '';
        for (let i = 0; i < financeFinancialData.length; i++) {
          const button = document.createElement('button');
          button.className = i === 0
            ? 'w-8 h-3 rounded-full bg-blue-500 transition-all duration-300'
            : 'w-3 h-3 rounded-full bg-gray-300 transition-all duration-300';
          button.addEventListener('click', () => goToFinanceSlide(i));
          indicatorsEl.appendChild(button);
        }
      }

      // Initialize “Previous” / “Next” button handlers
      function initFinanceSliderControls() {
        const prevBtnEl = document.getElementById('financePrevBtn');
        const nextBtnEl = document.getElementById('financeNextBtn');

        prevBtnEl.addEventListener('click', () => {
          financeCurrentSlide = financeCurrentSlide > 0
            ? financeCurrentSlide - 1
            : financeFinancialData.length - 1;
          updateFinanceSlide();
          resetFinanceAutoSlide();
        });

        nextBtnEl.addEventListener('click', () => {
          financeCurrentSlide = financeCurrentSlide < financeFinancialData.length - 1
            ? financeCurrentSlide + 1
            : 0;
          updateFinanceSlide();
          resetFinanceAutoSlide();
        });
      }

      // Jump to a specific slide index
      function goToFinanceSlide(index) {
        financeCurrentSlide = index;
        updateFinanceSlide();
        resetFinanceAutoSlide();
      }

      // Scroll the sliderTrack to show the current slide
      function updateFinanceSlide() {
        const sliderTrackEl = document.getElementById('financeSliderTrack');
        const cardWidthPlusGap = 320 + 24; // card width (320px) + gap (24px)
        sliderTrackEl.scrollTo({
          left: financeCurrentSlide * cardWidthPlusGap,
          behavior: 'smooth'
        });
        updateFinanceIndicators();
      }

      // Visually highlight the active indicator
      function updateFinanceIndicators() {
        const indicatorButtons = document.querySelectorAll('#financeIndicators button');
        indicatorButtons.forEach((btn, idx) => {
          if (idx === financeCurrentSlide) {
            btn.className = 'w-8 h-3 rounded-full bg-blue-500 transition-all duration-300';
          } else {
            btn.className = 'w-3 h-3 rounded-full bg-gray-300 transition-all duration-300';
          }
        });
      }

      // Start the automatic slide rotation
      function startFinanceAutoSlide() {
        financeAutoSlideInterval = setInterval(() => {
          financeCurrentSlide = financeCurrentSlide < financeFinancialData.length - 1
            ? financeCurrentSlide + 1
            : 0;
          updateFinanceSlide();
        }, 4000);
      }

      // Reset auto-slide (called whenever user manually goes to a new slide)
      function resetFinanceAutoSlide() {
        clearInterval(financeAutoSlideInterval);
        startFinanceAutoSlide();
      }

      // Update the “Last Updated” timestamp
      function updateFinanceLastUpdateTime() {
        const lastUpdateEl = document.getElementById('financeLastUpdate');
        const now = new Date();
        lastUpdateEl.textContent = `Updated ${now.toLocaleTimeString()}`;
      }

      // Sample data (used only if fetches fail)
      function getSampleData() {
        return [
          {
            type: 'crypto',
            symbol: 'BTC/USD',
            name: 'Bitcoin',
            price: 67850.23,
            change: 8.45,
            icon: '₿',
            color: '#f7931a',
            sparkline: [65000, 66200, 67100, 66800, 67500, 67850, 68200, 67850]
          },
          {
            type: 'stock',
            symbol: 'NVDA',
            name: 'NVIDIA Corp',
            price: 875.32,
            change: 12.67,
            icon: '🔧',
            color: '#76b900',
            sparkline: [850, 860, 870, 865, 872, 875, 880, 875]
          },
          {
            type: 'forex',
            symbol: 'EUR/USD',
            name: 'Euro Dollar',
            price: 1.0842,
            change: -2.34,
            icon: '💱',
            color: '#1f77b4',
            sparkline: [1.090, 1.088, 1.085, 1.087, 1.084, 1.082, 1.081, 1.084]
          }
        ];
      }

      // Initialize when page loads
      window.addEventListener('load', initializeFinanceSlider);

      // Pause auto-slide when user hovers over slider
      document.addEventListener('DOMContentLoaded', () => {
        const sliderContainerEl = document.getElementById('financeSliderContainer');
        if (sliderContainerEl) {
          sliderContainerEl.addEventListener('mouseenter', () => clearInterval(financeAutoSlideInterval));
          sliderContainerEl.addEventListener('mouseleave', startFinanceAutoSlide);
        }
      });
    </script>



    <!--

  <!-- Tailwind CSS CDN -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css">
    <style>
      /* Modal overlay styles */
      .modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        backdrop-filter: blur(4px);
        z-index: 1000;
        display: flex;
        justify-content: center;
        align-items: center;
      }

      .modal-container {
        background: #fff;
        padding: 1.5rem;
        border-radius: 0.375rem;
        z-index: 1010;
        width: 24rem;
        /* equivalent to Tailwind w-96 */
      }
    </style>

    <?php
// dashboard.php
$status = $_GET['status'] ?? '';
$message = $_GET['message'] ?? '';
?>

    <!-- Include SweetAlert2 JS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.all.min.js"></script>
  


<script>
  document.addEventListener("DOMContentLoaded", function () {
    <?php if ($status && $message): ?>
      Swal.fire({
        icon: '<?php echo ($status === "success" ? "success" : "error"); ?>',
        title: '<?php echo ucfirst($status); ?>',
        text: '<?php echo $message; ?>'
      });
    <?php endif; ?>
  });
</script>











    <!-- Market Cards Section -->
    <div style="position: relative; overflow: hidden;">
      <div
        style="display: flex; overflow-x: auto; scroll-snap-type: x mandatory; scrollbar-width: none; -ms-overflow-style: none; padding: 0 5px; gap: 16px; scroll-behavior: smooth;">
        <!-- Cryptocurrency Card -->
        <div
          style="flex: 0 0 clamp(280px, 90vw, 350px); scroll-snap-align: start; background: #ffffff; border-radius: 16px; padding: 20px; box-shadow: 0 4px 12px rgba(139, 92, 246, 0.08); border: 1px solid #f3f4f6; transition: all 0.3s ease; min-height: 300px;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 8px 25px rgba(139, 92, 246, 0.15)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(139, 92, 246, 0.08)'">
          <div
            style="border-radius: 12px; padding: 16px; margin-bottom: 16px; background: linear-gradient(135deg, #8b5cf6, #7c3aed);">
            <h3 style="font-size: 20px; font-weight: 800; color: white; margin: 0 0 4px 0;">Cryptocurrency</h3>
            <p style="color: rgba(255,255,255,0.8); font-size: 12px; font-weight: 500; margin: 0;">Digital Assets &
              Tokens</p>
          </div>

          <div style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 8px;">
            <div onclick="showCryptoChart('bitcoin', 'Bitcoin', 'BTC')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 10px; background: #f97316; box-shadow: 0 2px 8px rgba(249,115,22,0.3);">
                <span style="font-size: 10px;">₿</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">BTC</span>
            </div>

            <div onclick="showCryptoChart('ethereum', 'Ethereum', 'ETH')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #3b82f6; box-shadow: 0 2px 8px rgba(59,130,246,0.3);">
                <span>Ξ</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">ETH</span>
            </div>

            <div onclick="showCryptoChart('binancecoin', 'Binance Coin', 'BNB')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #eab308; box-shadow: 0 2px 8px rgba(234,179,8,0.3);">
                <span>BNB</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">BNB</span>
            </div>

            <div onclick="showCryptoChart('cardano', 'Cardano', 'ADA')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #1e40af; box-shadow: 0 2px 8px rgba(30,64,175,0.3);">
                <span>ADA</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">ADA</span>
            </div>

            <div onclick="showCryptoChart('solana', 'Solana', 'SOL')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #8b5cf6; box-shadow: 0 2px 8px rgba(139,92,246,0.3);">
                <span>SOL</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">SOL</span>
            </div>

            <div onclick="showCryptoChart('polkadot', 'Polkadot', 'DOT')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #ec4899; box-shadow: 0 2px 8px rgba(236,72,153,0.3);">
                <span>DOT</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">DOT</span>
            </div>

            <div onclick="showCryptoChart('chainlink', 'Chainlink', 'LINK')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 6px; background: #1d4ed8; box-shadow: 0 2px 8px rgba(29,78,216,0.3);">
                <span>LINK</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">LINK</span>
            </div>

            <div onclick="showCryptoChart('litecoin', 'Litecoin', 'LTC')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #9ca3af; box-shadow: 0 2px 8px rgba(156,163,175,0.3);">
                <span>LTC</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">LTC</span>
            </div>
          </div>
        </div>

        <!-- Forex Card -->
        <div
          style="flex: 0 0 clamp(280px, 90vw, 350px); scroll-snap-align: start; background: #ffffff; border-radius: 16px; padding: 20px; box-shadow: 0 4px 12px rgba(139, 92, 246, 0.08); border: 1px solid #f3f4f6; transition: all 0.3s ease; min-height: 300px;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 8px 25px rgba(139, 92, 246, 0.15)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(139, 92, 246, 0.08)'">
          <div
            style="border-radius: 12px; padding: 16px; margin-bottom: 16px; background: linear-gradient(135deg, #7c3aed, #6d28d9);">
            <h3 style="font-size: 20px; font-weight: 800; color: white; margin: 0 0 4px 0;">Forex</h3>
            <p style="color: rgba(255,255,255,0.8); font-size: 12px; font-weight: 500; margin: 0;">Currency Trading
              Pairs</p>
          </div>

          <div style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 8px;">
            <div onclick="showForexChart('EURUSD', 'EUR/USD')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #3b82f6; box-shadow: 0 2px 8px rgba(59,130,246,0.3);">
                <span>€/$</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">EUR/USD</span>
            </div>

            <div onclick="showForexChart('GBPUSD', 'GBP/USD')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #dc2626; box-shadow: 0 2px 8px rgba(220,38,38,0.3);">
                <span>£/$</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">GBP/USD</span>
            </div>

            <div onclick="showForexChart('USDJPY', 'USD/JPY')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #b91c1c; box-shadow: 0 2px 8px rgba(185,28,28,0.3);">
                <span>$/¥</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">USD/JPY</span>
            </div>

            <div onclick="showForexChart('USDCHF', 'USD/CHF')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #ef4444; box-shadow: 0 2px 8px rgba(239,68,68,0.3);">
                <span>CHF</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">USD/CHF</span>
            </div>

            <div onclick="showForexChart('AUDUSD', 'AUD/USD')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #3b82f6; box-shadow: 0 2px 8px rgba(59,130,246,0.3);">
                <span>A/$</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">AUD/USD</span>
            </div>

            <div onclick="showForexChart('USDCAD', 'USD/CAD')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #ef4444; box-shadow: 0 2px 8px rgba(239,68,68,0.3);">
                <span>C/$</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">USD/CAD</span>
            </div>

            <div onclick="showForexChart('NZDUSD', 'NZD/USD')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #3b82f6; box-shadow: 0 2px 8px rgba(59,130,246,0.3);">
                <span>N/$</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">NZD/USD</span>
            </div>

            <div onclick="showForexChart('EURGBP', 'EUR/GBP')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #6366f1; box-shadow: 0 2px 8px rgba(99,102,241,0.3);">
                <span>€/£</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">EUR/GBP</span>
            </div>
          </div>
        </div>

        <!-- Stocks Card -->
        <div
          style="flex: 0 0 clamp(280px, 90vw, 350px); scroll-snap-align: start; background: #ffffff; border-radius: 16px; padding: 20px; box-shadow: 0 4px 12px rgba(139, 92, 246, 0.08); border: 1px solid #f3f4f6; transition: all 0.3s ease; min-height: 300px;"
          onmouseover="this.style.transform='translateY(-4px)'; this.style.boxShadow='0 8px 25px rgba(139, 92, 246, 0.15)'"
          onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(139, 92, 246, 0.08)'">
          <div
            style="border-radius: 12px; padding: 16px; margin-bottom: 16px; background: linear-gradient(135deg, #6d28d9, #5b21b6);">
            <h3 style="font-size: 20px; font-weight: 800; color: white; margin: 0 0 4px 0;">Stocks</h3>
            <p style="color: rgba(255,255,255,0.8); font-size: 12px; font-weight: 500; margin: 0;">Global Stock Markets
            </p>
          </div>

          <div style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 8px;">
            <div onclick="showStockInfo('AAPL', 'Apple Inc.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #1d4ed8; box-shadow: 0 2px 8px rgba(29,78,216,0.3);">
                <span>🍎</span>
              </div>
              <span style="color: #374151; font-size: 10px; font-weight: 600;">AAPL</span>
            </div>

            <div onclick="showStockInfo('GOOGL', 'Alphabet Inc.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #dc2626; box-shadow: 0 2px 8px rgba(220,38,38,0.3);">
                <span>G</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">GOOGL</span>
            </div>

            <div onclick="showStockInfo('MSFT', 'Microsoft Corp.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #0ea5e9; box-shadow: 0 2px 8px rgba(14,165,233,0.3);">
                <span>MS</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">MSFT</span>
            </div>

            <div onclick="showStockInfo('AMZN', 'Amazon.com Inc.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #f97316; box-shadow: 0 2px 8px rgba(249,115,22,0.3);">
                <span>A</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">AMZN</span>
            </div>
            <div onclick="showStockInfo('TSLA', 'Tesla Inc.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #ef4444; box-shadow: 0 2px 8px rgba(239,68,68,0.3);">
                <span>T</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">TSLA</span>
            </div>

            <div onclick="showStockInfo('META', 'Meta Platforms Inc.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #1d4ed8; box-shadow: 0 2px 8px rgba(29,78,216,0.3);">
                <span>M</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">META</span>
            </div>

            <div onclick="showStockInfo('NFLX', 'Netflix Inc.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 8px; background: #dc2626; box-shadow: 0 2px 8px rgba(220,38,38,0.3);">
                <span>N</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">NFLX</span>
            </div>

            <div onclick="showStockInfo('NVDA', 'NVIDIA Corp.')"
              style="background: #f9fafb; border: 1px solid #f3f4f6; border-radius: 12px; padding: 12px 8px; cursor: pointer; display: flex; flex-direction: column; align-items: center; transition: all 0.3s ease;"
              onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 15px rgba(139, 92, 246, 0.1)'"
              onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none'">
              <div
                style="width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 6px; color: white; font-weight: bold; font-size: 7px; background: #10b981; box-shadow: 0 2px 8px rgba(16,185,129,0.3);">
                <span>NVDA</span>
              </div>
              <span style="color: #374151; font-size: 8px; font-weight: 600;">NVDA</span>
            </div>
          </div>
        </div>
      </div>
    </div>
    <!-- JavaScript Functions -->
    <script>
      /**
    * showCryptoChart(cryptoId, cryptoName, symbol):
    *  - cryptoId: CoinGecko's ID (e.g., "bitcoin", "ethereum")
    *  - cryptoName: human-readable name (e.g., "Bitcoin")
    *  - symbol: ticker symbol (e.g., "BTC")
    *
    * Fetches live price data from CoinGecko, displays it in a SweetAlert modal,
    * then injects a TradingView widget for the chart.
    */
      async function showCryptoChart(cryptoId, cryptoName, symbol) {
        try {
          // Show loading modal
          Swal.fire({
            title: `Loading ${cryptoName}...`,
            html: `<div class="loading-spinner">
                     <div class="spinner"></div>
                     <p>Fetching live data from CoinGecko...</p>
                   </div>`,
            showConfirmButton: false,
            allowOutsideClick: false,
            didOpen: () => {
              Swal.showLoading();
            }
          });

          // Fetch price data from CoinGecko's "/coins/{id}" endpoint
          const url = `https://api.coingecko.com/api/v3/coins/${cryptoId}?localization=false&tickers=false&market_data=true&community_data=false&developer_data=false&sparkline=false`;
          const response = await fetch(url);

          if (!response.ok) {
            throw new Error(`CoinGecko API error: ${response.status}`);
          }

          const coinData = await response.json();

          // Extract relevant fields
          const priceUsd = coinData.market_data.current_price.usd;
          const priceBtc = coinData.market_data.current_price.btc;
          const marketCap = coinData.market_data.market_cap.usd;
          const volume24h = coinData.market_data.total_volume.usd;
          const priceChange24h = coinData.market_data.price_change_percentage_24h;
          const lastUpdated = coinData.market_data.last_updated;

          // Determine price change color
          const changeColor = priceChange24h >= 0 ? '#10b981' : '#ef4444';
          const changeIcon = priceChange24h >= 0 ? '📈' : '📉';

          // Show crypto data in SweetAlert modal
          Swal.fire({
            title: `${cryptoName} (${symbol})`,
            html: `
                <div style="text-align: left; padding: 20px;">
                    <div style="display: flex; align-items: center; justify-content: center; margin-bottom: 20px;">
                        <span style="font-size: 24px; margin-right: 10px;">${changeIcon}</span>
                        <span style="font-size: 32px; font-weight: bold; color: #333;">$${priceUsd.toLocaleString()}</span>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 10px; margin-bottom: 15px;">
                        <p style="margin: 8px 0;"><strong>Price (BTC):</strong> ${priceBtc.toFixed(8)} BTC</p>
                        <p style="margin: 8px 0;"><strong>Market Cap:</strong> $${marketCap.toLocaleString()}</p>
                        <p style="margin: 8px 0;"><strong>24h Volume:</strong> $${volume24h.toLocaleString()}</p>
                        <p style="margin: 8px 0;"><strong>24h Change:</strong> 
                           <span style="color: ${changeColor}; font-weight: bold;">${priceChange24h.toFixed(2)}%</span>
                        </p>
                        <p style="margin: 8px 0;"><strong>Last Updated:</strong> ${new Date(lastUpdated).toLocaleString()}</p>
                    </div>
                    
                    <div id="tradingview-chart-${cryptoId}" style="height: 400px; width: 100%;"></div>
                </div>
            `,
            width: '80%',
            showConfirmButton: true,
            confirmButtonText: 'Close',
            confirmButtonColor: '#8b5cf6',
            showCancelButton: true,
            cancelButtonText: 'View Full Chart',
            cancelButtonColor: '#6b7280',
            didOpen: () => {
              // Inject TradingView widget
              const tradingViewSymbol = `BINANCE:${symbol}USDT`;

              new TradingView.widget({
                width: '100%',
                height: 400,
                symbol: tradingViewSymbol,
                interval: '60',
                timezone: 'Etc/UTC',
                theme: 'light',
                style: '1',
                locale: 'en',
                toolbar_bg: '#f1f3f4',
                enable_publishing: false,
                allow_symbol_change: true,
                container_id: `tradingview-chart-${cryptoId}`,
                studies: [
                  "MACD@tv-basicstudies",
                  "RSI@tv-basicstudies"
                ]
              });
            }
          });

        } catch (error) {
          Swal.fire({
            icon: 'error',
            title: 'Error Loading Data',
            text: `Failed to load ${cryptoName} data: ${error.message}`,
            confirmButtonColor: '#ef4444'
          });
          console.error(error);
        }
      }

      /**
       * showForexChart(pair, displayName):
       *  - pair: Forex pair in format "EURUSD", "GBPUSD", etc.
       *  - displayName: human-readable label ("EUR / USD")
       *
       * Shows a SweetAlert modal with TradingView widget for the given Forex pair.
       */
      function showForexChart(pair, displayName) {
        Swal.fire({
          title: `${displayName} Chart`,
          html: `
            <div style="padding: 20px;">
                <div style="background: linear-gradient(135deg, #7c3aed, #6d28d9); color: white; padding: 15px; border-radius: 10px; margin-bottom: 20px;">
                    <h3 style="margin: 0; font-size: 24px;">${displayName}</h3>
                    <p style="margin: 5px 0 0 0; opacity: 0.9;">Live Forex Trading Chart</p>
                </div>
                <div id="tradingview-forex-${pair}" style="height: 400px; width: 100%;"></div>
            </div>
        `,
          width: '80%',
          showConfirmButton: true,
          confirmButtonText: 'Close',
          confirmButtonColor: '#7c3aed',
          showCancelButton: true,
          cancelButtonText: 'Open in New Tab',
          cancelButtonColor: '#6b7280',
          didOpen: () => {
            const tradingViewSymbol = `OANDA:${pair}`;

            new TradingView.widget({
              width: '100%',
              height: 400,
              symbol: tradingViewSymbol,
              interval: '60',
              timezone: 'Etc/UTC',
              theme: 'light',
              style: '1',
              locale: 'en',
              toolbar_bg: '#f1f3f4',
              enable_publishing: false,
              allow_symbol_change: true,
              container_id: `tradingview-forex-${pair}`,
              studies: [
                "Moving Average@tv-basicstudies",
                "RSI@tv-basicstudies"
              ]
            });
          }
        }).then((result) => {
          if (result.dismiss === Swal.DismissReason.cancel) {
            // Open TradingView in new tab
            window.open(`https://www.tradingview.com/chart/?symbol=OANDA:${pair}`, '_blank');
          }
        });
      }

      /**
       * showStockInfo(symbol, companyName):
       *  - symbol: e.g., "AAPL", "TSLA"
       *  - companyName: human-readable name ("Apple Inc.")
       *
       * Shows a SweetAlert modal with TradingView widget for the stock.
       */
      function showStockInfo(symbol, companyName) {
        // Get stock emoji based on symbol
        const stockEmojis = {
          'AAPL': '🍎',
          'GOOGL': '🔍',
          'MSFT': '💻',
          'AMZN': '📦',
          'TSLA': '🚗',
          'META': '📘',
          'NFLX': '🎬',
          'NVDA': '🎮'
        };

        const stockEmoji = stockEmojis[symbol] || '📊';

        Swal.fire({
          title: `${stockEmoji} ${companyName}`,
          html: `
            <div style="padding: 20px;">
                <div style="background: linear-gradient(135deg, #6d28d9, #5b21b6); color: white; padding: 15px; border-radius: 10px; margin-bottom: 20px;">
                    <h3 style="margin: 0; font-size: 24px;">${symbol}</h3>
                    <p style="margin: 5px 0 0 0; opacity: 0.9;">${companyName}</p>
                </div>
                <div id="tradingview-stock-${symbol}" style="height: 400px; width: 100%;"></div>
            </div>
        `,
          width: '80%',
          showConfirmButton: true,
          confirmButtonText: 'Close',
          confirmButtonColor: '#6d28d9',
          showCancelButton: true,
          cancelButtonText: 'View Analysis',
          cancelButtonColor: '#6b7280',
          didOpen: () => {
            // Determine the appropriate exchange prefix
            let tradingViewSymbol;
            if (['AAPL', 'GOOGL', 'MSFT', 'AMZN', 'TSLA', 'META', 'NFLX', 'NVDA'].includes(symbol)) {
              tradingViewSymbol = `NASDAQ:${symbol}`;
            } else {
              tradingViewSymbol = `NYSE:${symbol}`;
            }

            new TradingView.widget({
              width: '100%',
              height: 400,
              symbol: tradingViewSymbol,
              interval: 'D',
              timezone: 'Etc/UTC',
              theme: 'light',
              style: '1',
              locale: 'en',
              toolbar_bg: '#f1f3f4',
              enable_publishing: false,
              allow_symbol_change: true,
              container_id: `tradingview-stock-${symbol}`,
              studies: [
                "EMA@tv-basicstudies",
                "MACD@tv-basicstudies"
              ]
            });
          }
        }).then((result) => {
          if (result.dismiss === Swal.DismissReason.cancel) {
            // Open detailed analysis or company info
            window.open(`https://www.tradingview.com/chart/?symbol=NASDAQ:${symbol}`, '_blank');
          }
        });
      }

      // Optional: Add custom CSS for loading spinner
      const style = document.createElement('style');
      style.textContent = `
    .loading-spinner {
        text-align: center;
        padding: 20px;
    }
    
    .spinner {
        border: 4px solid #f3f3f3;
        border-top: 4px solid #8b5cf6;
        border-radius: 50%;
        width: 40px;
        height: 40px;
        animation: spin 1s linear infinite;
        margin: 0 auto 15px auto;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
`;
      document.head.appendChild(style);
    </script>






    <?php
// ==================== SOL Card ====================
$sol_data = getCryptoData('solana', 'sol_price_cache.json');
$sol_price_usd = isset($sol_data['solana']['usd']) ? (float)$sol_data['solana']['usd'] : 0;
$sol_change = isset($sol_data['solana']['usd_24h_change']) ? (float)$sol_data['solana']['usd_24h_change'] : 0;
$sol_amount = ($sol_price_usd > 0) ? $sol / $sol_price_usd : 0;
$sol_change_formatted = number_format($sol_change, 2);
$sol_change_color = ($sol_change >= 0) ? "text-green-500" : "text-red-500";
?>

    <a href="detailedsol.php">
      <div class="neumorphic-card p-4 flex items-center justify-between" id="solCard">
        <div class="shimmer-effect"></div>
        <div class="flex items-center relative z-10">
          <img src="https://primemarketspro.com/cryptos/sol.png" alt="SOL" class="w-10 h-10 mr-4">
          <div>
            <p class="font-bold" style="color: #3d4863;">SOL (SOL)</p>
            <p class="text-gray-900 text-sm">
              <?php echo number_format($sol_amount, 6) . ' SOL'; ?>
            </p>
          </div>
        </div>
        <div class="text-right relative z-10">
          <p class="font-semibold" id="solPrice">$
            <?php echo number_format($sol_price_usd, 4); ?>
          </p>
          <p class="<?php echo $sol_change_color; ?> text-sm" id="solChange">
            <?php echo ($sol_change >= 0 ? '+' : '') . $sol_change_formatted . '%'; ?>
          </p>
        </div>
      </div>
    </a>

    <?php
// ==================== ADA Card ====================
$ada_data = getCryptoData('cardano', 'ada_price_cache.json');
$ada_price_usd = isset($ada_data['cardano']['usd']) ? (float)$ada_data['cardano']['usd'] : 0;
$ada_change = isset($ada_data['cardano']['usd_24h_change']) ? (float)$ada_data['cardano']['usd_24h_change'] : 0;
$ada_amount = ($ada_price_usd > 0) ? $ada / $ada_price_usd : 0;
$ada_change_formatted = number_format($ada_change, 2);
$ada_change_color = ($ada_change >= 0) ? "text-green-500" : "text-red-500";
?>

    <a href="detailada.php">
      <div class="neumorphic-card p-4 flex items-center justify-between" id="adaCard">
        <div class="shimmer-effect"></div>
        <div class="flex items-center relative z-10">
          <img src="https://primemarketspro.com/cryptos/ada.png" alt="ADA" class="w-10 h-10 mr-4">
          <div>
            <p class="font-bold" style="color: #3d4863;">ADA (ADA)</p>
            <p class="text-gray-900 text-sm">
              <?php echo number_format($ada_amount, 6) . ' ADA'; ?>
            </p>
          </div>
        </div>
        <div class="text-right relative z-10">
          <p class="font-semibold" id="adaPrice">$
            <?php echo number_format($ada_price_usd, 4); ?>
          </p>
          <p class="<?php echo $ada_change_color; ?> text-sm" id="adaChange">
            <?php echo ($ada_change >= 0 ? '+' : '') . $ada_change_formatted . '%'; ?>
          </p>
        </div>
      </div>
    </a>

    <?php
// ==================== BNB Card ====================
$bnb_data = getCryptoData('binancecoin', 'bnb_price_cache.json');
$bnb_price_usd = isset($bnb_data['binancecoin']['usd']) ? (float)$bnb_data['binancecoin']['usd'] : 0;
$bnb_change = isset($bnb_data['binancecoin']['usd_24h_change']) ? (float)$bnb_data['binancecoin']['usd_24h_change'] : 0;
$bnb_amount = ($bnb_price_usd > 0) ? $bnb / $bnb_price_usd : 0;
$bnb_change_formatted = number_format($bnb_change, 2);
$bnb_change_color = ($bnb_change >= 0) ? "text-green-500" : "text-red-500";
?>

    <a href="detailbnb.php">
      <div class="neumorphic-card p-4 flex items-center justify-between" id="bnbCard">
        <div class="shimmer-effect"></div>
        <div class="flex items-center relative z-10">
          <img src="https://primemarketspro.com/cryptos/bnb.png" alt="BNB" class="w-10 h-10 mr-4">
          <div>
            <p class="font-bold" style="color: #3d4863;">BNB (BNB)</p>
            <p class="text-gray-900 text-sm">
              <?php echo number_format($bnb_amount, 6) . ' BNB'; ?>
            </p>
          </div>
        </div>
        <div class="text-right relative z-10">
          <p class="font-semibold" id="bnbPrice">$
            <?php echo number_format($bnb_price_usd, 2); ?>
          </p>
          <p class="<?php echo $bnb_change_color; ?> text-sm" id="bnbChange">
            <?php echo ($bnb_change >= 0 ? '+' : '') . $bnb_change_formatted . '%'; ?>
          </p>
        </div>
      </div>
    </a>

    <?php
// ==================== Dogecoin Card ====================
$doge_data = getCryptoData('dogecoin', 'doge_price_cache.json');
$doge_price_usd = isset($doge_data['dogecoin']['usd']) ? (float)$doge_data['dogecoin']['usd'] : 0;
$doge_change = isset($doge_data['dogecoin']['usd_24h_change']) ? (float)$doge_data['dogecoin']['usd_24h_change'] : 0;
$doge_amount = ($doge_price_usd > 0) ? $doge / $doge_price_usd : 0;
$doge_change_formatted = number_format($doge_change, 2);
$doge_change_color = ($doge_change >= 0) ? "text-green-500" : "text-red-500";
?>

    <a href="detailedoge.php">
      <div class="neumorphic-card p-4 flex items-center justify-between" id="dogeCard">
        <div class="shimmer-effect"></div>
        <div class="flex items-center relative z-10">
          <img src="https://primemarketspro.com/cryptos/doge.png" alt="Dogecoin" class="w-10 h-10 mr-4">
          <div>
            <p class="font-bold" style="color: #3d4863;">Dogecoin (DOGE)</p>
            <p class="text-gray-900 text-sm">
              <?php echo number_format($doge_amount, 6) . ' DOGE'; ?>
            </p>
          </div>
        </div>
        <div class="text-right relative z-10">
          <p class="font-semibold" id="dogePrice">$
            <?php echo number_format($doge_price_usd, 2); ?>
          </p>
          <p class="<?php echo $doge_change_color; ?> text-sm" id="dogeChange">
            <?php echo ($doge_change >= 0 ? '+' : '') . $doge_change_formatted . '%'; ?>
          </p>
        </div>
      </div>
    </a>

    <?php
// ==================== Pepe Card ====================
$pepe_data = getCryptoData('pepe', 'pepe_price_cache.json');
$pepe_price_usd = isset($pepe_data['pepe']['usd']) ? (float)$pepe_data['pepe']['usd'] : 0;
$pepe_change = isset($pepe_data['pepe']['usd_24h_change']) ? (float)$pepe_data['pepe']['usd_24h_change'] : 0;
$pepe_amount = ($pepe_price_usd > 0) ? $pepe / $pepe_price_usd : 0;
$pepe_change_formatted = number_format($pepe_change, 2);
$pepe_change_color = ($pepe_change >= 0) ? "text-green-500" : "text-red-500";
?>

    <a href="detailedpepe.php">
      <div class="neumorphic-card p-4 flex items-center justify-between" id="pepeCard">
        <div class="shimmer-effect"></div>
        <div class="flex items-center relative z-10">
          <!-- Replace the src below with the correct Pepe logo URL if needed -->
          <img src="https://primemarketspro.com/cryptos/pepe.png" alt="Pepe" class="w-10 h-10 mr-4">
          <div>
            <p class="font-bold" style="color: #3d4863;">Pepe (PEPE)</p>
            <p class="text-gray-900 text-sm">
              <?php echo number_format($pepe_amount, 6) . ' PEPE'; ?>
            </p>
          </div>
        </div>
        <div class="text-right relative z-10">
          <p class="font-semibold" id="pepePrice">$
            <?php echo number_format($pepe_price_usd, 2); ?>
          </p>
          <p class="<?php echo $pepe_change_color; ?> text-sm" id="pepeChange">
            <?php echo ($pepe_change >= 0 ? '+' : '') . $pepe_change_formatted . '%'; ?>
          </p>
        </div>
      </div>
    </a>

  </div>
  </section>



  <!--<div style=" padding: 4px; margin-top: 65px;">-->


  <?php
require('db.php'); // This file sets up the PDO connection in $conn and includes your session functions
// Retrieve the user email from session using your helper function
$user_email = getSessionEmail();
if (!$user_email) {
    echo '<div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded relative" role="alert">
            <strong class="font-bold">Not Logged In</strong>
            <span class="block sm:inline"> Please log in to access your dashboard.</span>
          </div>';
    exit;
}
// Prepare the query using PDO with a named parameter
$query = "SELECT * FROM kyc_submissions WHERE email = :email";
$stmt = $conn->prepare($query);
// Bind the email parameter using bindParam
$stmt->bindParam(':email', $user_email, PDO::PARAM_STR);
// Execute the query
$stmt->execute();
// Fetch the user data as an associative array
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if ($user) {
    if ($user['status'] == 'verified') {
        $statusMessage = '<div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
  <div class="flex items-center">
    <script src="https://cdn.lordicon.com/lordicon.js"></script>
    <lord-icon
      src="https://cdn.lordicon.com/stxfyhky.json"
      trigger="loop"
      delay="2000"
      colors="primary:#ffffff,secondary:#28a745"
      style="width:50px;height:50px">
    </lord-icon>
    <div class="ml-3">
      <p>
        Your account status is currently high.
        <span class="inline-flex items-center ml-2">
          <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" class="w-5 h-5">
            <circle cx="12" cy="12" r="12" fill="#1877F2"/>
            <path d="M10.1 16.2l-3.7-3.7 1.4-1.4 2.3 2.3 5.3-5.3 1.4 1.4L10.1 16.2z" fill="#fff"/>
          </svg>
        </span>
        Contact Support if you encounter any issues.
      </p>
    </div>
  </div>
</div>
';
    } else {
        $statusMessage = '<div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
                            <div class="flex items-center">
                              <script src="https://cdn.lordicon.com/lordicon.js"></script>
                              <lord-icon
                                  src="https://cdn.lordicon.com/stxfyhky.json"
                                  trigger="loop"
                                  delay="2000"
                                  colors="primary:#ffffff,secondary:#ee6d66"
                                  style="width:50px;height:50px">
                              </lord-icon>
                              <div class="ml-3">
                                <p>Account is currently Unverified</p>
                              </div>
                            </div>
                          </div>';
    }
    echo $statusMessage;
} else {
    echo '<div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded relative" role="alert">
            <strong class="font-bold">Pending</strong>
            <span class="block sm:inline"> Please proceed with your KYC .</span>
          </div>';
}
?>




  <div class="container mx-auto px-4 py-8">
    <div id="imageSlider" class="relative overflow-hidden group">
      <!-- Slider Track -->
      <div class="slider-track flex overflow-x-auto scroll-smooth pb-6 snap-x snap-mandatory gap-4">
        <?php
      $sql = "SELECT id, title, content, image_url FROM blogs ORDER BY created_at DESC"; // Fetch all posts
      $stmt = $conn->query($sql);

      // Fetch all blog posts as an associative array
      $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

      if (count($rows) > 0) {
          foreach ($rows as $row) {
              $id = $row['id'];
              $title = $row['title'];
              $content = $row['content'];
              $image_url = $row['image_url'];

              // Limit content to the first 5 words
              $content_words = explode(' ', $content);
              if (count($content_words) > 5) {
                  $content = implode(' ', array_slice($content_words, 0, 5)) . '...';
              }
      ?>
        <a href="blog_detail.php?id=<?php echo $id; ?>" class="block" style="text-decoration: none;">
          <div class="slider-card" style="
                  background: #f8f9fa; /* Lighter background */
                  border-radius: 10px; 
                  box-shadow: 2px 2px 8px rgba(0, 0, 0, 0.1);
                  display: flex;
                  flex-direction: column; /* Stack image & content */
                  width: 320px; /* More user-friendly size */
                  overflow: hidden;
                  transition: transform 0.3s ease;
              ">
            <!-- Image -->
            <div class="card-image" style="
                      width: 100%;
                      height: 180px; /* Reduce height for better layout */
                      background-image: url('./uploads/<?php echo $image_url; ?>');
                      background-size: cover; 
                      background-position: center; 
                      border-top-left-radius: 10px; 
                      border-top-right-radius: 10px;">
            </div>

            <!-- Content -->
            <div class="card-content" style="
                      padding: 12px;
                      text-align: left; 
                      display: flex;
                      flex-direction: column;
                      justify-content: center;
                  ">
              <h3 style="
                          font-size: 16px; /* Reduce size slightly */
                          font-weight: bold; 
                          color: #3d4863; 
                          margin-bottom: 5px;
                          line-height: 1.4;
                          white-space: normal;
                          word-wrap: break-word;
                          overflow-wrap: break-word;
                      ">
                <?php echo nl2br(htmlspecialchars($title)); ?>
              </h3>

              <p style="
                          font-size: 14px;
                          color: #555;
                          line-height: 1.5;
                          margin-top: 5px;
                      ">
                <?php echo nl2br(htmlspecialchars($content)); ?>
              </p>
            </div>
          </div>
        </a>
        <?php
          }
      } else {
          echo "<p style='text-align: center; font-size: 16px; color: #555;'>No blog posts found.</p>";
      }

      // Close the PDO connection by setting it to null
      $conn = null;
      ?>
      </div>
    </div>
  </div>


  <!-- Navigation Buttons -->
  <button class="nav-button absolute left-4 top-1/2 -translate-y-1/2 opacity-0 group-hover:opacity-100" id="prevButton">
    &larr;
  </button>
  <button class="nav-button absolute right-4 top-1/2 -translate-y-1/2 opacity-0 group-hover:opacity-100"
    id="nextButton">
    &rarr;
  </button>

  <script>
    // Get the slider track
    const sliderTrack = document.querySelector('.slider-track');

    // The number of pixels to scroll per button click
    const scrollSpeed = 300;

    // Start auto-scrolling if needed (optional)
    let autoScroll;
    function startAutoScroll() {
      autoScroll = setInterval(() => {
        if (sliderTrack.scrollLeft + sliderTrack.clientWidth >= sliderTrack.scrollWidth) {
          sliderTrack.scrollTo({ left: 0, behavior: 'instant' }); // Instantly reset to the start
        } else {
          sliderTrack.scrollLeft += scrollSpeed;
        }
      }, 50); // Adjust interval for smoothness
    }

    // Pause auto-scroll on hover (optional feature)
    sliderTrack.addEventListener('mouseover', () => clearInterval(autoScroll));
    sliderTrack.addEventListener('mouseleave', startAutoScroll);

    // Manual Navigation (Left and Right buttons)
    const prevBtn = document.getElementById('prevButton');
    const nextBtn = document.getElementById('nextButton');

    // Scroll right on clicking the "next" button
    nextBtn.addEventListener('click', () => {
      sliderTrack.scrollBy({ left: scrollSpeed, behavior: 'smooth' });
    });

    // Scroll left on clicking the "prev" button
    prevBtn.addEventListener('click', () => {
      sliderTrack.scrollBy({ left: -scrollSpeed, behavior: 'smooth' });
    });
  </script>












  <!-- Stake Section -->
  <section id="stakeSection" style="padding-top: 2rem;">



    <div class="container mx-auto">
      <div class="grid grid-cols-1 md:grid-cols-2 gap-6">

        <!-- Enhanced Crypto Converter Widget -->
        <div class="bg-white shadow rounded-lg p-6">
          <h2 class="text-2xl font-bold mb-4">Crypto Converter</h2>
          <form id="converterForm" class="space-y-4">
            <div>
              <label for="fromCurrency" class="block text-sm font-medium text-gray-700">From</label>
              <div class="relative">
                <select id="fromCurrency" name="fromCurrency"
                  class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2 pl-10">
                  <!-- Will be populated by JavaScript -->
                </select>
                <div id="fromCurrencyIcon" class="absolute left-3 top-3 w-6 h-6"></div>
              </div>
            </div>
            <div>
              <label for="toCurrency" class="block text-sm font-medium text-gray-700">To</label>
              <div class="relative">
                <select id="toCurrency" name="toCurrency"
                  class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2 pl-10">
                  <!-- Will be populated by JavaScript -->
                </select>
                <div id="toCurrencyIcon" class="absolute left-3 top-3 w-6 h-6"></div>
              </div>
            </div>
            <div>
              <label for="amount" class="block text-sm font-medium text-gray-700">Amount</label>
              <input type="number" id="amount" name="amount" placeholder="Enter amount"
                class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2" step="any" />
            </div>
          </form>
          <div id="result" class="mt-4 p-4 bg-gray-50 rounded-lg text-gray-800"></div>
          <div id="lastUpdated" class="mt-2 text-xs text-gray-500 text-right"></div>
        </div>




        <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
        <!-- (Optional) Include your CSS framework styles -->
        <style>
          /* Page dark background and text color */


          /* Dark-themed form controls with rounded corners */
          .form-control {
            width: 100%;
            padding: 8px;
            margin: 5px 0;
            border: 1px solid #444;
            border-radius: 5px;
            background-color: #2e2e3e;
            color: #cdd6f4;
          }

          .alert-success {
            color: #a3be8c;
          }

          .alert-danger {
            color: #bf616a;
          }


          .swal2-confirm {
            background: #9d00ff !important;
            border-radius: 5px !important;
          }

          /* Ensure modal inputs and selects are rounded */
          .swal2-input,
          .swal2-select {
            border-radius: 5px !important;
          }
        </style>

        <!-- Display update messages if present -->
        <?php if (isset($updateMessage)): ?>
        <div class="alert alert-success">
          <?php echo $updateMessage; ?>
        </div>
        <?php endif; ?>
        <?php if (isset($updateError)): ?>
        <div class="alert alert-danger">
          <?php echo $updateError; ?>
        </div>
        <?php endif; ?>

        <!-- Rest of your page content -->

      <script>
  document.addEventListener('DOMContentLoaded', function () {
    <?php if ($showProfileAlert): ?>
      Swal.fire({
        title: 'Complete Your Profile',
        html: `<form id="profileForm">
          <div class="form-group" style="text-align:left; margin-bottom:10px;">
            <label for="country">Country:</label>
            <select name="country" id="country" class="form-control" required>
              <option value="">Select Country</option>
              <?php foreach ($allCountries as $option): ?>
                <option value="<?php echo $option['dial_code']; ?>">
                  <?php echo $option['name']; ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="form-group" style="text-align:left; margin-bottom:10px;">
            <label for="phone">Phone Number:</label>
            <input type="tel" name="full_phone" id="phone" class="form-control" placeholder="Enter phone number" required/>
          </div>
        </form>`,
        showCancelButton: false,
        confirmButtonText: 'Update Profile',
        preConfirm: () => {
          const country = document.getElementById('country').value;
          const phone = document.getElementById('phone').value;
          if (!country || !phone) {
            Swal.showValidationMessage('Both country and phone number are required.');
            return false;
          }
          return { country: country, full_phone: phone };
        }
      }).then((result) => {
        if (result.isConfirmed) {
          // Create FormData and include a flag for profile update
          const formData = new FormData();
          formData.append('update_profile', '1');
          formData.append('country', result.value.country);
          formData.append('full_phone', result.value.full_phone);

          // Send the data via AJAX (fetch API)
          fetch(window.location.href, {
            method: 'POST',
            body: formData
          })
          .then(response => response.text())
          .then(data => {
            Swal.fire({
              icon: 'success',
              title: 'Profile updated successfully',
              timer: 1500,
              showConfirmButton: false
            });
            // Optionally, update parts of the page with new data here.
          })
          .catch(error => {
            Swal.fire({
              icon: 'error',
              title: 'Update failed',
              text: error
            });
          });
        }
      });
    <?php endif; ?>
  });
</script>

















        <div class="bg-white shadow rounded-lg p-6 w-full max-w-md">
          <h2 class="text-2xl font-bold mb-4 text-center">Crypto Buy/Sell</h2>
          <form id="tradeForm" action="tradeprocess.php" method="POST" class="space-y-4">
            <!-- Transaction Type -->
            <div>
              <label for="transactionType" class="block text-sm font-medium text-gray-700">Transaction Type</label>
              <select id="transactionType" name="transactionType"
                class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2" required>
                <option value="" disabled selected>Select Type</option>
                <!-- Using inline style for option text colors (supported in some browsers) -->
                <option value="buy" style="color: green;">Buy</option>
                <option value="sell" style="color: red;">Sell</option>
              </select>
            </div>
            <!-- Currency Dropdown (populated dynamically with up to 30 coins) -->
            <div>
              <label for="transactionCurrency" class="block text-sm font-medium text-gray-700">Currency</label>
              <select id="transactionCurrency" name="transactionCurrency"
                class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2" required>
                <option value="" disabled selected>Loading coins...</option>
              </select>
            </div>
            <!-- USD Amount Input -->
            <div>
              <label for="transactionAmount" class="block text-sm font-medium text-gray-700">Amount (USD)</label>
              <input type="number" id="transactionAmount" name="transactionAmount" placeholder="Enter amount in USD"
                min="0.01" step="0.01" required
                class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2">
            </div>
            <!-- Converted Coin Amount (read-only) -->
            <div>
              <label for="convertedAmount" class="block text-sm font-medium text-gray-700">
                Converted Amount (<span id="selectedSymbol">COIN</span>)
              </label>
              <input type="number" id="convertedAmount" name="convertedAmount" placeholder="Coin amount" readonly
                class="mt-1 block w-full border border-gray-300 rounded-lg shadow-sm p-2">
            </div>
            <button type="submit" class="w-full bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 transition">
              Submit Order
            </button>
          </form>
        </div>

        <script>
          // Global object to store coin prices (keyed by coin id)
          let coinPrices = {};

          // Fetch up to 30 coins from CoinGecko's API
          async function fetchCoins() {
            try {
              const url = 'https://api.coingecko.com/api/v3/coins/markets?' + new URLSearchParams({
                vs_currency: 'usd',
                order: 'market_cap_desc',
                per_page: 30,
                page: 1,
                sparkline: false
              });
              const response = await fetch(url);
              const coins = await response.json();
              populateCoinDropdown(coins);
            } catch (error) {
              console.error('Error fetching coins:', error);
              const select = document.getElementById('transactionCurrency');
              select.innerHTML = '<option value="">Error loading coins</option>';
            }
          }

          // Populate the currency dropdown with fetched coins
          function populateCoinDropdown(coins) {
            const select = document.getElementById('transactionCurrency');
            select.innerHTML = '<option value="" disabled selected>Select Currency</option>';
            coins.forEach(coin => {
              // Save coin price for conversion
              coinPrices[coin.id] = coin.current_price;
              const option = document.createElement('option');
              option.value = coin.id;
              option.textContent = `${coin.name} (${coin.symbol.toUpperCase()})`;
              select.appendChild(option);
            });
          }

          // Convert USD to coin amount based on selected coin's current price
          function convertUsdToCoin() {
            const usdAmount = parseFloat(document.getElementById('transactionAmount').value) || 0;
            const coinId = document.getElementById('transactionCurrency').value;
            if (!coinId || !coinPrices[coinId]) {
              document.getElementById('convertedAmount').value = '';
              document.getElementById('selectedSymbol').textContent = 'COIN';
              return;
            }
            const price = coinPrices[coinId]; // USD per coin
            const coinAmount = usdAmount / price;
            document.getElementById('convertedAmount').value = coinAmount.toFixed(8);
            // Update symbol in label
            const selectedOption = document.getElementById('transactionCurrency').selectedOptions[0];
            if (selectedOption) {
              const match = selectedOption.textContent.match(/\(([^)]+)\)/);
              if (match && match[1]) {
                document.getElementById('selectedSymbol').textContent = match[1].toUpperCase();
              }
            }
          }

          // Event listeners for real-time conversion
          document.getElementById('transactionAmount').addEventListener('input', convertUsdToCoin);
          document.getElementById('transactionCurrency').addEventListener('change', convertUsdToCoin);

          // Fetch coins on page load
          window.onload = fetchCoins;
        </script>







        <style>
          * {
            box-sizing: border-box;
            font-family: 'Arial', sans-serif;
          }



          .order-book-container {
            width: 100%;
            max-width: 600px;
            margin: 0 auto;
            background-color: white;
            border-radius: 4px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            overflow: hidden;
          }

          .order-book-header {
            padding: 15px;
            font-size: 18px;
            font-weight: bold;
            border-bottom: 1px solid #eee;
          }

          .order-book-table {
            width: 100%;
            border-collapse: collapse;
          }

          .order-book-table th {
            background-color: #f8f8f8;
            text-align: right;
            padding: 10px;
            font-size: 12px;
            color: #666;
          }

          .order-book-table td {
            padding: 10px;
            text-align: right;
            position: relative;
            font-size: 14px;
          }

          .sell-row td:first-child {
            color: #e74c3c;
          }

          .buy-row td:first-child {
            color: #2ecc71;
          }

          .order-depth {
            position: absolute;
            top: 0;
            bottom: 0;
            right: 0;
            z-index: 0;
          }

          .sell-depth {
            background-color: rgba(231, 76, 60, 0.2);
          }

          .buy-depth {
            background-color: rgba(46, 204, 113, 0.2);
          }

          .order-book-table tr {
            border-bottom: 1px solid #f0f0f0;
          }

          .order-book-divider {
            background-color: #f8f8f8;
            padding: 10px;
            font-size: 14px;
            display: flex;
            justify-content: space-between;
            border-top: 1px solid #eee;
            border-bottom: 1px solid #eee;
          }

          .last-price {
            font-weight: bold;
          }

          .price-change {
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 12px;
          }

          .price-change.negative {
            background-color: rgba(231, 76, 60, 0.2);
            color: #e74c3c;
          }

          .price-change.positive {
            background-color: rgba(46, 204, 113, 0.2);
            color: #2ecc71;
          }
        </style>

        <div class="order-book-container">
          <div class="order-book-header">Order Book</div>
          <table class="order-book-table">
            <thead>
              <tr>
                <th>Price(BTC)</th>
                <th>Amount(ETH)</th>
                <th>Total(ETH)</th>
              </tr>
            </thead>
            <tbody id="sell-orders">
              <!-- Sell orders will be populated here -->
            </tbody>
          </table>

          <div class="order-book-divider">
            <span class="last-price">Last Price: <span id="last-price">0.020367</span></span>
            <span>
              <span id="usd-price">USD 148.65</span>
              <span id="price-change" class="price-change negative">-0.51%</span>
            </span>
          </div>

          <table class="order-book-table">
            <tbody id="buy-orders">
              <!-- Buy orders will be populated here -->
            </tbody>
          </table>
        </div>

        <style>
          /* Animate depth bars smoothly */
          .order-depth {
            transition: width 0.2s ease-in-out;
          }

          /* Animate price-change color */
          .price-change {
            transition: color 0.2s ease-in-out, background-color 0.2s ease-in-out;
          }

          /* Flash highlight on updates */
          .cell-update {
            animation: flash-bg 0.4s ease-in-out;
          }

          @keyframes flash-bg {
            0% {
              background-color: rgba(255, 255, 0, 0.4);
            }

            100% {
              background-color: transparent;
            }
          }
        </style>

        <script>
          // Sample data
          const sellOrders = [
            { price: 0.022572, amount: 1.253415, total: 15.27648, depth: 75 },
            { price: 0.020371, amount: 1.205415, total: 15.25648, depth: 65 },
            { price: 0.023572, amount: 1.645415, total: 15.23648, depth: 85 },
            { price: 0.032378, amount: 1.206715, total: 15.25348, depth: 40 },
            { price: 0.022573, amount: 1.262415, total: 15.19648, depth: 55 }
          ];

          const buyOrders = [
            { price: 0.158373, amount: 1.209515, total: 15.23248, depth: 30 },
            { price: 0.020851, amount: 1.206245, total: 15.25458, depth: 45 },
            { price: 0.025375, amount: 1.205715, total: 15.65648, depth: 65 },
            { price: 0.020252, amount: 1.205485, total: 15.24548, depth: 50 },
            { price: 0.020373, amount: 1.205415, total: 15.25648, depth: 40 }
          ];

          // Format number to 6 decimal places if needed
          function formatNumber(num) {
            if (num < 0.0001) {
              return num.toFixed(6);
            } else if (num < 1) {
              return num.toFixed(5);
            } else {
              return num.toFixed(2);
            }
          }

          // Utility to flash‐highlight a cell when updated
          function flashCell(cell) {
            cell.classList.add('cell-update');
            setTimeout(() => cell.classList.remove('cell-update'), 400);
          }

          // Populate sell orders
          const sellOrdersEl = document.getElementById('sell-orders');
          sellOrders.forEach(order => {
            const row = document.createElement('tr');
            row.className = 'sell-row';
            row.innerHTML = `
      <td>${formatNumber(order.price)}</td>
      <td>${formatNumber(order.amount)}</td>
      <td>
        ${formatNumber(order.total)}
        <div class="order-depth sell-depth" style="width: ${order.depth}%"></div>
      </td>
    `;
            sellOrdersEl.appendChild(row);
          });

          // Populate buy orders
          const buyOrdersEl = document.getElementById('buy-orders');
          buyOrders.forEach(order => {
            const row = document.createElement('tr');
            row.className = 'buy-row';
            row.innerHTML = `
      <td>${formatNumber(order.price)}</td>
      <td>${formatNumber(order.amount)}</td>
      <td>
        ${formatNumber(order.total)}
        <div class="order-depth buy-depth" style="width: ${order.depth}%"></div>
      </td>
    `;
            buyOrdersEl.appendChild(row);
          });

          // Example API connection (simulated)
          function connectToOrderBookAPI() {
            console.log("Connecting to order book API...");
            // In a real implementation, connect to WebSocket here.
            // e.g.: const socket = new WebSocket('wss://api.exchange.com/orderbook');

            // Speed up updates to every 500ms
            setInterval(updateRandomOrder, 500);
          }

          function updateRandomOrder() {
            const isSell = Math.random() > 0.5;
            const orders = isSell ? sellOrders : buyOrders;
            const index = Math.floor(Math.random() * orders.length);

            // Slight variation
            orders[index].price *= (1 + (Math.random() * 0.02 - 0.01));
            orders[index].amount *= (1 + (Math.random() * 0.04 - 0.02));
            orders[index].total = orders[index].price * orders[index].amount * 10;

            // Update DOM
            const targetEl = isSell ? sellOrdersEl : buyOrdersEl;
            const row = targetEl.querySelectorAll('tr')[index];
            const cells = row.querySelectorAll('td');

            // Price cell
            cells[0].textContent = formatNumber(orders[index].price);
            flashCell(cells[0]);
            // Amount cell
            cells[1].textContent = formatNumber(orders[index].amount);
            flashCell(cells[1]);
            // Total cell & depth bar
            cells[2].childNodes[0].nodeValue = formatNumber(orders[index].total);
            const depthBar = row.querySelector('.order-depth');
            depthBar.style.width = orders[index].depth + '%';
            flashCell(cells[2]);

            // Occasionally update last price
            if (Math.random() > 0.7) {
              const lastPriceEl = document.getElementById('last-price');
              const oldPrice = parseFloat(lastPriceEl.textContent);
              const newPrice = formatNumber(orders[index].price);
              lastPriceEl.textContent = newPrice;
              flashCell(lastPriceEl);

              const priceChangeEl = document.getElementById('price-change');
              const changePercent = ((newPrice - oldPrice) / oldPrice * 100).toFixed(2);
              priceChangeEl.textContent = `${changePercent}%`;
              priceChangeEl.className = `price-change ${changePercent >= 0 ? 'positive' : 'negative'}`;
            }
          }

          // Initialize
          connectToOrderBookAPI();
        </script>








        
        



        <!-- TradingView Widget BEGIN -->
        <div class="tradingview-widget-container">
          <div class="tradingview-widget-container__widget"></div>
          <script type="text/javascript"
            src="https://s3.tradingview.com/external-embedding/embed-widget-advanced-chart.js" async>
              {
                "autosize": true,
                  "symbol": "BINANCE:BTCUSDT",
                    "timezone": "Etc/UTC",
                      "theme": "light",
                        "style": "1",
                          "locale": "en",
                            "backgroundColor": "rgba(255, 255, 255, 1)",
                              "gridColor": "rgba(255, 255, 255, 0.06)",
                                "withdateranges": true,
                                  "range": "YTD",
                                    "hide_side_toolbar": false,
                                      "allow_symbol_change": true,
                                        "save_image": false,
                                          "calendar": false,
                                            "hide_volume": true,
                                              "support_host": "https://www.tradingview.com"
              }
            </script>
        </div>
        <!-- TradingView Widget END -->





      </div>
    </div>


  </section>

  <!-- JavaScript for Enhanced Crypto Converter -->
  <script>
    // List of top cryptocurrencies to include
    const topCoins = [
      'usd', 'bitcoin', 'ethereum', 'tether', 'binancecoin', 'solana', 'ripple', 'cardano',
      'avalanche-2', 'polkadot', 'dogecoin', 'tron', 'chainlink', 'polygon', 'shiba-inu',
      'litecoin', 'dai', 'uniswap', 'bitcoin-cash', 'stellar', 'monero', 'ethereum-classic',
      'filecoin', 'cosmos', 'hedera-hashgraph', 'vechain', 'algorand', 'internet-computer',
      'flow', 'apecoin', 'eos', 'decentraland', 'the-sandbox', 'axie-infinity', 'aave',
      'maker', 'compound', 'pancakeswap-token', 'near', 'fantom', 'frax', 'quant',
      'neo', 'bitcoin-sv', 'elrond-erd-2', 'chiliz', 'dash', 'kucoin-shares', 'nexo',
      'zcash', 'iota', 'theta-token', 'havven', 'decred', 'harmony', 'thorchain', 'waves',
      'basic-attention-token', 'enjincoin', 'gala', 'holo', 'amp-token', 'ecash', 'gnosis'
    ];

    // Coin display names (short names for display)
    const coinDisplayNames = {
      'usd': 'USD',
      'bitcoin': 'BTC',
      'ethereum': 'ETH',
      'tether': 'USDT',
      'binancecoin': 'BNB',
      'solana': 'SOL',
      'ripple': 'XRP',
      'cardano': 'ADA',
      'avalanche-2': 'AVAX',
      'polkadot': 'DOT',
      'dogecoin': 'DOGE',
      'tron': 'TRX',
      'chainlink': 'LINK',
      'polygon': 'MATIC',
      'shiba-inu': 'SHIB',
      'litecoin': 'LTC',
      'dai': 'DAI',
      'uniswap': 'UNI',
      'bitcoin-cash': 'BCH',
      'stellar': 'XLM',
      'monero': 'XMR',
      'ethereum-classic': 'ETC',
      'filecoin': 'FIL',
      'cosmos': 'ATOM',
      'hedera-hashgraph': 'HBAR',
      'vechain': 'VET',
      'algorand': 'ALGO',
      'internet-computer': 'ICP',
      'flow': 'FLOW',
      'apecoin': 'APE',
      'eos': 'EOS',
      'decentraland': 'MANA',
      'the-sandbox': 'SAND',
      'axie-infinity': 'AXS',
      'aave': 'AAVE',
      'maker': 'MKR',
      'compound': 'COMP',
      'pancakeswap-token': 'CAKE',
      'near': 'NEAR',
      'fantom': 'FTM',
      'frax': 'FRAX',
      'quant': 'QNT',
      'neo': 'NEO',
      'bitcoin-sv': 'BSV',
      'elrond-erd-2': 'EGLD',
      'chiliz': 'CHZ',
      'dash': 'DASH',
      'kucoin-shares': 'KCS',
      'nexo': 'NEXO',
      'zcash': 'ZEC',
      'iota': 'MIOTA',
      'theta-token': 'THETA',
      'havven': 'SNX',
      'decred': 'DCR',
      'harmony': 'ONE',
      'thorchain': 'RUNE',
      'waves': 'WAVES',
      'basic-attention-token': 'BAT',
      'enjincoin': 'ENJ',
      'gala': 'GALA',
      'holo': 'HOT',
      'amp-token': 'AMP',
      'ecash': 'XEC',
      'gnosis': 'GNO'
    };

    // Cache for cryptocurrency prices and information
    let cryptoCache = {
      timestamp: 0,
      data: {}
    };

    // Initialize the app when the document is loaded
    document.addEventListener('DOMContentLoaded', async function () {
      try {
        // Load cryptocurrency data
        await loadCryptoData();

        // Populate currency selectors
        populateCurrencySelectors();

        // Set default values
        document.getElementById('fromCurrency').value = 'usd';
        document.getElementById('toCurrency').value = 'bitcoin';

        // Update currency icons
        updateCurrencyIcon('fromCurrency', 'fromCurrencyIcon');
        updateCurrencyIcon('toCurrency', 'toCurrencyIcon');

        // Add change event listeners for currency selectors
        document.getElementById('fromCurrency').addEventListener('change', function () {
          updateCurrencyIcon('fromCurrency', 'fromCurrencyIcon');
          convertCurrency();
        });

        document.getElementById('toCurrency').addEventListener('change', function () {
          updateCurrencyIcon('toCurrency', 'toCurrencyIcon');
          convertCurrency();
        });

        // Add input event listener for real-time conversion
        document.getElementById('amount').addEventListener('input', convertCurrency);

        // Populate transaction currency selector
        populateTransactionCurrencySelector();
      } catch (error) {
        console.error('Error initializing app:', error);
        document.getElementById('result').innerHTML = '<p class="text-red-500">Error loading cryptocurrency data. Please refresh the page.</p>';
      }
    });

    // Function to load cryptocurrency data from CoinGecko API
    async function loadCryptoData() {
      const currentTime = Date.now();

      // If cache is valid (less than 1 minute old), use cached data
      if (currentTime - cryptoCache.timestamp < 60000 && Object.keys(cryptoCache.data).length > 0) {
        return cryptoCache.data;
      }

      try {
        // Build the query string with all coins
        const idsParam = topCoins.filter(coin => coin !== 'usd').join(',');
        const url = `https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&ids=${idsParam}&order=market_cap_desc&per_page=100&page=1&sparkline=false&price_change_percentage=24h`;

        const response = await fetch(url);
        const data = await response.json();

        // Process and cache the data
        const processedData = {};
        data.forEach(coin => {
          processedData[coin.id] = {
            id: coin.id,
            symbol: coin.symbol.toUpperCase(),
            name: coin.name,
            image: coin.image,
            price: coin.current_price,
            price_change_24h: coin.price_change_percentage_24h
          };
        });

        // Add USD manually
        processedData['usd'] = {
          id: 'usd',
          symbol: 'USD',
          name: 'US Dollar',
          image: 'https://media.istockphoto.com/id/1160777145/vector/us-circle-flag-icon-waving-american-symbol-vector-illustration.jpg?s=612x612&w=0&k=20&c=it5YEg0mYlnJDF-aQ98srEwTRC4fb0RQwTW9yJ0SCsc=',
          price: 1,
          price_change_24h: 0
        };

        // Update cache
        cryptoCache = {
          timestamp: currentTime,
          data: processedData
        };

        // Update last updated timestamp display
        const date = new Date();
        document.getElementById('lastUpdated').textContent = `Last updated: ${date.toLocaleTimeString()}`;

        return processedData;
      } catch (error) {
        console.error('Error fetching cryptocurrency data:', error);
        throw error;
      }
    }

    // Function to populate currency selectors
    function populateCurrencySelectors() {
      const fromSelect = document.getElementById('fromCurrency');
      const toSelect = document.getElementById('toCurrency');

      // Clear existing options
      fromSelect.innerHTML = '';
      toSelect.innerHTML = '';

      // Create and append options
      topCoins.forEach(coinId => {
        const coinData = cryptoCache.data[coinId];
        if (coinData) {
          const option = document.createElement('option');
          option.value = coinId;
          const displayName = coinDisplayNames[coinId] || coinData.symbol;
          option.textContent = `${displayName} - ${coinData.name}`;

          // Add option to both selectors
          fromSelect.appendChild(option.cloneNode(true));
          toSelect.appendChild(option);
        }
      });
    }

    // Function to populate transaction currency selector
    function populateTransactionCurrencySelector() {
      const transactionCurrencySelect = document.getElementById('transactionCurrency');

      // Clear existing options
      transactionCurrencySelect.innerHTML = '';

      // Create and append options
      topCoins.forEach(coinId => {
        const coinData = cryptoCache.data[coinId];
        if (coinData) {
          const option = document.createElement('option');
          option.value = coinId;
          const displayName = coinDisplayNames[coinId] || coinData.symbol;
          option.textContent = `${displayName} - ${coinData.name}`;
          transactionCurrencySelect.appendChild(option);
        }
      });
    }

    // Function to update currency icon
    function updateCurrencyIcon(selectId, iconId) {
      const select = document.getElementById(selectId);
      const iconContainer = document.getElementById(iconId);
      const selectedCurrency = select.value;

      // Clear existing icon
      iconContainer.innerHTML = '';

      // Get coin data
      const coinData = cryptoCache.data[selectedCurrency];

      if (coinData && coinData.image) {
        // Create and append icon image
        const icon = document.createElement('img');
        icon.src = coinData.image;
        icon.alt = coinData.name;
        icon.className = 'w-6 h-6';
        iconContainer.appendChild(icon);
      }
    }

    // Function to convert currency in real-time
    function convertCurrency() {
      const fromCurrency = document.getElementById('fromCurrency').value;
      const toCurrency = document.getElementById('toCurrency').value;
      const amount = parseFloat(document.getElementById('amount').value);

      if (isNaN(amount) || amount <= 0) {
        document.getElementById('result').innerHTML = '<p class="text-red-500">Please enter a valid amount.</p>';
        return;
      }

      const fromPrice = cryptoCache.data[fromCurrency].price;
      const toPrice = cryptoCache.data[toCurrency].price;

      const convertedAmount = (amount * fromPrice) / toPrice;

      document.getElementById('result').innerHTML = `
      <p class="text-lg font-semibold">${amount.toFixed(4)} ${cryptoCache.data[fromCurrency].symbol} = ${convertedAmount.toFixed(4)} ${cryptoCache.data[toCurrency].symbol}</p>
    `;
    }
  </script>













  <!-- Market Chart -->
    <h3 class="text-xl font-semibold mb-4">Market Overview</h3>
    <canvas id="marketChart" class="w-full" style="height: 100px; max-height: 200px;"></canvas>
  <!-- Make sure to include the Chart.js library -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

 <script>
  // Fetch Market Chart Data
  function fetchMarketChartData() {
    const ctx = document.getElementById('marketChart').getContext('2d');
    const labels = [];
    const dataPoints = [];

    fetch('https://api.coingecko.com/api/v3/coins/bitcoin/market_chart?vs_currency=usd&days=7')
      .then((response) => response.json())
      .then((data) => {
        data.prices.forEach((pricePoint) => {
          const date = new Date(pricePoint[0]);
          labels.push(`${date.getMonth() + 1}/${date.getDate()}`);
          dataPoints.push(pricePoint[1]);
        });

        // Create vertical purple gradient
        const gradient = ctx.createLinearGradient(0, 0, 0, 200);
        gradient.addColorStop(0, 'rgba(139, 99, 241, 0.5)');   // Top
        gradient.addColorStop(1, 'rgba(139, 99, 241, 0.05)');  // Bottom

        new Chart(ctx, {
          type: 'line',
          data: {
            labels: labels,
            datasets: [{
              label: 'Bitcoin Price',
              data: dataPoints,
              borderColor: '#8b63f1',
              backgroundColor: gradient, // Use gradient here
              borderWidth: 2,
              pointRadius: 0,
              tension: 0.4,
              fill: true
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
              y: { beginAtZero: false, display: false },
              x: { display: false }
            },
            plugins: {
              legend: { display: false },
              tooltip: { intersect: false }
            }
          }
        });
      })
      .catch((error) => {
        console.error('Error fetching market chart data:', error);
      });
  }

  // Call the function to fetch and render the market chart
  fetchMarketChartData();
</script>













  <?php

// Ensure the user is logged in
if (!isset($_SESSION['user_email'])) {
    echo "<p>User is not logged in or email is not set in session.</p>";
    exit();
}
$userEmail = $_SESSION['user_email'];

// Database connection details
$servername = "localhost";
$username = "wptlgmnb_userp";
$password = "Qliox4ITygj6Vh4V";
$dbname = "wptlgmnb_dbp";

// Create connection
$conn = new mysqli($servername, $username, $password, $dbname);

// Check connection
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Fetch the latest 10 transactions from payments and crypto_withdrawal tables
$sql = "
    (SELECT id, wallet_address AS wallet, network, amount AS total_amount, email, name, created_at, transhash, status, 'payment' AS source 
    FROM payments WHERE email = '$userEmail')
    UNION ALL
    (SELECT id, wallet, network, total_amount, email, name, created_at, transhash, status, 'transaction' AS source 
    FROM crypto_withdrawal WHERE email = '$userEmail')
    ORDER BY created_at DESC LIMIT 10
";
$result = $conn->query($sql);

// Handle API rate limits by caching crypto prices
$cacheFile = 'crypto_prices.json';
$cacheDuration = 300; // 5 minutes
$cryptoPrices = [];

// Define the coins we need to track
$coinsToTrack = [
    'Bitcoin' => 'bitcoin',
    'Ethereum' => 'ethereum',
    'TRC20' => 'tether',
    'xrp' => 'ripple',
    'ERC20' => 'ethereum',
    'Dogecoin' => 'dogecoin',
    'Solana' => 'solana',
    'BNB' => 'binancecoin',
    'Pepe' => 'pepe'
];

// Cache management
if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < $cacheDuration) {
    $cryptoPrices = json_decode(file_get_contents($cacheFile), true);
} else {
    $coinIds = implode(',', array_values(array_unique($coinsToTrack)));
    $apiUrl = "https://api.coingecko.com/api/v3/simple/price?ids={$coinIds}&vs_currencies=usd";
    $response = @file_get_contents($apiUrl);
    
    if ($response) {
        $cryptoPrices = json_decode($response, true);
        file_put_contents($cacheFile, $response);
    } else {
        $cryptoPrices = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) : [];
    }
}

// Network logos
$networkLogos = [
    'Bitcoin' => 'https://primemarketspro.com/cryptos/bitcoin.png',
    'Ethereum' => 'https://primemarketspro.com/cryptos/eth.png',
    'TRC20' => 'https://primemarketspro.com/cryptos/usdt.png',
    'xrp' => 'https://primemarketspro.com/cryptos/xrp.png',
    'ERC20' => 'https://primemarketspro.com/cryptos/eth.png',
    'dogecoin' => 'https://primemarketspro.com/cryptos/doge.png',
    'solana' => 'https://primemarketspro.com/cryptos/sol.png',
    'bnb' => 'https://primemarketspro.com/cryptos/bnb.png',
    'pepe' => 'https://primemarketspro.com/cryptos/pepe.png'
];

// Network currency symbols
$networkSymbols = [
    'Bitcoin' => 'BTC',
    'Ethereum' => 'ETH',
    'TRC20' => 'USDT',
    'xrp' => 'XRP',
    'ERC20' => 'ETH',
    'Dogecoin' => 'DOGE',
    'Solana' => 'SOL',
    'BNB' => 'BNB',
    'Pepe' => 'PEPE'
];

// Check if there are results
if ($result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $transactionId = $row['id'];
        $wallet = htmlspecialchars($row['wallet']);
        $usdAmount = $row['total_amount'];
        $transhash = substr($row['transhash'], 0, 6) . '*****' . substr($row['transhash'], -5);
        $status = ucfirst($row['status']);
        $created_at = date('M d, Y', strtotime($row['created_at']));
        $source = $row['source'];
        $networkName = $row['network'];
        
        // Get crypto symbol
        $cryptoSymbol = $networkSymbols[$networkName] ?? strtoupper($networkName);
        
        // Determine arrow direction and color
        if ($source == 'payment') {
            $arrow = '↓';
            $arrowColor = 'green';
            $statusLabel = 'Received';
        } else {
            $arrow = '↑';
            $arrowColor = 'red';
            $statusLabel = 'Sent';
        }

        // Get logo URL
        $logoUrl = $networkLogos[$networkName] ?? $networkLogos['Bitcoin'];

        // Convert USD Amount to Crypto if price data is available
        $cryptoAmount = null;
        $coinGeckoId = $coinsToTrack[$networkName] ?? null;
        
        if ($coinGeckoId && isset($cryptoPrices[$coinGeckoId]['usd']) && is_numeric($usdAmount)) {
            $price = $cryptoPrices[$coinGeckoId]['usd'];
            if ($price > 0) {
                $cryptoAmount = $usdAmount / $price;
            }
        }

        // Format amounts
        $formattedUsdAmount = number_format($usdAmount, 2);
        $formattedCryptoAmount = $cryptoAmount !== null ? number_format($cryptoAmount, 8) : 'N/A';
        ?>

  <!-- Transaction Card -->
  <a href="detailed.php?id=<?php echo $transactionId; ?>"
    class="neumorphic-card p-4 flex flex-col items-start justify-between bg-white shadow-lg rounded-lg">
    <div class="flex items-center justify-between w-full relative z-10">
      <div class="flex items-center">
        <img src="<?php echo $logoUrl; ?>" alt="<?php echo $networkName; ?>" class="w-8 h-8 mr-4 rounded-full">
        <div>
          <p class="font-bold text-lg" style="color: #3d4863;">
            <?php echo $networkName; ?>
          </p>
          <p class="text-gray-900 text-sm">
            <?php 
                            if ($usdAmount == '0.00' || $usdAmount == '0') {
                                echo 'Unconfirmed';
                            } else {
                                echo '$' . $formattedUsdAmount;
                                if ($cryptoAmount !== null) {
                                    echo ' (' . $formattedCryptoAmount . ' ' . $cryptoSymbol . ')';
                                }
                            }
                            ?>
          </p>
        </div>
      </div>

      <!-- Transaction Status -->
      <div class="flex flex-col items-end text-right">
        <p class="font-semibold" style="color: <?php echo $arrowColor; ?>">
          <?php echo $arrow; ?>
          <?php echo $statusLabel; ?>
        </p>
        <p class="text-sm text-gray-500">
          <?php echo $created_at; ?>
        </p>
        <p class="text-sm text-gray-500">
          <?php echo $transhash; ?>
        </p>
      </div>
    </div>
  </a>

  <?php
    }
} else {
    echo "<p>No recent transactions found.</p>";
}

$conn->close();
?>






  <!-- See More Button -->
  <div class="text-center mt-3">
    <a href="history.php" class="neumorphic-card p-3 bg-blue-500 text-white font-bold rounded-md hover:bg-blue-600">
      See More Transactions
    </a>
  </div>


  <div style=" margin-top: 16px;">


    <!-- Leaflet CSS -->
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />

    <!-- Leaflet JS -->
    <script src="https://unpkg.com/leaflet@1.7.1/dist/leaflet.js"></script>

    <!-- Flag Icon CSS (using jsDelivr for correct asset paths) -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flag-icon-css@3.5.0/css/flag-icon.min.css" />

    <style>
      #world-map {
        /* Reduced height from h-64 to h-48 */
        height: 12rem;
        width: 100%;
      }

      .country-row {
        transition: background-color 0.3s ease;
      }

      .country-row:hover {
        background-color: #f0f4f8;
      }

      /* Make flag icons appear round */
      .flag-icon.flag-icon-round {
        border-radius: 50%;
        overflow: hidden;
      }
    </style>

    <section class="container mx-auto p-6 w-full md:w-auto">
      <div class="grid grid-cols-1 md:grid-cols-3 gap-6 w-full">
        <!-- Customer Demographics Card -->
        <div class="col-span-2 bg-white rounded-lg shadow-md p-6 w-full">
          <div class="flex justify-between items-center mb-4">
            <h2 class="text-lg font-semibold text-gray-800">Customer Demographics</h2>
          </div>
          <p class="text-sm text-gray-600 mb-4">Number of customers by country</p>

          <!-- Leaflet Map -->
          <div id="world-map" class="w-full bg-gray-100 mb-4 rounded"></div>

          <!-- Country Table replacing Pie Chart -->
          <div class="bg-white rounded-lg overflow-hidden mb-4">
            <div class="px-4 py-3 bg-gray-50 border-b border-gray-200">
              <h3 class="text-md font-semibold text-gray-800">Customer Distribution</h3>
              <p id="totalCountries" class="text-sm text-gray-600 mt-1"></p>
            </div>

            <div class="overflow-x-auto max-h-60 overflow-y-auto">
              <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50 sticky top-0">
                  <tr>
                    <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Flag
                    </th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Country
                    </th>
                    <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customers</th>
                  </tr>
                </thead>
                <tbody id="countryTableBody" class="bg-white divide-y divide-gray-200">
                  <!-- Rows will be populated by JavaScript -->
                </tbody>
              </table>
            </div>
          </div>
        </div>

        <!-- Quick Stats Card -->
        <div class="bg-white rounded-lg shadow-md p-6 w-full">
          <h2 class="text-lg font-semibold text-gray-800 mb-4">Quick Stats</h2>
          <div class="mb-4">
            <div class="text-sm text-gray-600">Total Customers</div>
            <!-- Animated counter will update this -->
            <div id="counter" class="text-2xl font-bold text-gray-800">0</div>
          </div>
          <div class="mb-4">
            <div class="text-sm text-gray-600">New Customers</div>
            <div class="text-2xl font-bold text-green-600">+483</div>
          </div>
          <div>
            <div class="text-sm text-gray-600">Churn Rate</div>
            <div class="text-2xl font-bold text-red-600">57%</div>
          </div>
        </div>
      </div>
    </section>

    <!-- Include Leaflet CSS & JS -->
    <link rel="stylesheet" href="https://unpkg.com/leaflet/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet/dist/leaflet.js"></script>

    <script>
      // --------------------------------------------------
      // 1. Animate Counter (unchanged from your original)
      // --------------------------------------------------
      function animateCounter(id, start, end, duration) {
        const element = document.getElementById(id);
        let startTimestamp = null;
        const range = end - start;

        function step(timestamp) {
          if (!startTimestamp) startTimestamp = timestamp;
          const progress = timestamp - startTimestamp;

          const current = Math.floor(start + (range * (progress / duration)));
          element.innerText = current.toLocaleString();

          if (progress < duration) {
            window.requestAnimationFrame(step);
          } else {
            element.innerText = end.toLocaleString();
          }
        }

        window.requestAnimationFrame(step);
      }

      document.addEventListener("DOMContentLoaded", function () {
        animateCounter("counter", 1, 378002, 2000);
      });
    </script>

    <script>
  // --------------------------------------------------
  // 2. Fetch raw country data from PHP
  //    Each object in the array should look like:
  //    { code: "us", name: "United States", customers: "5", percentage: "12" }
  // --------------------------------------------------
  const customerData = <?php echo $jsonCountries; ?>;

  // --------------------------------------------------
  // 3. Predefined coordinates for a few countries (expand as needed)
  // --------------------------------------------------
  const countryCoordinates = {
    'us': [37.0902, -95.7129],
    'au': [-25.2744, 133.7751],
    'af': [33.9391, 67.7100],
    // Add more as needed...
  };

  // --------------------------------------------------
  // 4. Filter out any countries with 0 or invalid "customers"
  // --------------------------------------------------
  const filteredData = customerData.filter(country => {
    const c = parseInt(country.customers, 10);
    if (!country.code || isNaN(c) || c <= 0) {
      console.warn("Excluding country:", country.name, "with customers:", country.customers);
      return false;
    }
    return true;
  });

  // --------------------------------------------------
  // 5. Generate random multipliers (100-999) for each country
  // --------------------------------------------------
  function getRandomMultiplier() {
    return Math.floor(Math.random() * 900) + 100; // Random number between 100-999
  }

  // --------------------------------------------------
  // 6. Build the table once DOM is ready
  // --------------------------------------------------
  document.addEventListener("DOMContentLoaded", function () {
    const tableBody = document.getElementById('countryTableBody');

    filteredData.forEach((country, idx) => {
      const originalCustomers = parseInt(country.customers, 10);
      const multiplier = getRandomMultiplier();
      const multipliedValue = originalCustomers * multiplier;

      // Create table row
      const row = document.createElement('tr');
      row.className = 'country-row border-b border-gray-200';

      // Flag column
      const flagCell = document.createElement('td');
      flagCell.className = 'px-4 py-3 text-center';
      const flagSpan = document.createElement('span');
      flagSpan.className = `flag-icon flag-icon-${country.code} flag-icon-round`;
      flagSpan.style.fontSize = '20px';
      flagCell.appendChild(flagSpan);

      // Country name column
      const nameCell = document.createElement('td');
      nameCell.className = 'px-4 py-3 font-medium text-gray-900';
      nameCell.textContent = country.name;

      // Final value column
      const finalCell = document.createElement('td');
      finalCell.className = 'px-4 py-3 text-center font-semibold text-green-600';
      finalCell.textContent = multipliedValue.toLocaleString();

      // Append cells to row
      row.appendChild(flagCell);
      row.appendChild(nameCell);
      row.appendChild(finalCell);

      // Append row to table body
      tableBody.appendChild(row);
    });

    // Display total count
    const totalCountries = filteredData.length;
    const totalElement = document.getElementById('totalCountries');
    if (totalElement) {
      totalElement.textContent = `Total Countries: ${totalCountries}`;
    }
  });
</script>


    <script>
      // --------------------------------------------------
      // 7. Initialize Leaflet map and add markers
      //    Each popup shows the actual 'customers' value from the DB
      // --------------------------------------------------
      function createFlagElement(countryCode) {
        const flagSpan = document.createElement('span');
        flagSpan.className = `flag-icon flag-icon-${countryCode} flag-icon-round`;
        flagSpan.style.fontSize = '32px';
        return flagSpan;
      }

      const map = L.map('world-map').setView([20, 0], 2);
      L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
      }).addTo(map);

      filteredData.forEach(country => {
        const actualCount = parseInt(country.customers, 10);
        const coords = country.code in countryCoordinates
          ? countryCoordinates[country.code]
          : [0, 0];

        const marker = L.marker(coords).addTo(map);
        marker.bindPopup(`
      <div class="text-center">
        <span 
          class="flag-icon flag-icon-${country.code} flag-icon-round" 
          style="font-size: 2rem; display:inline-block; margin-bottom:8px;"
        ></span>
        <br>
        <strong>${country.name}</strong><br>
        Customers: ${actualCount.toLocaleString()}
      </div>
    `);
      });
    </script>



    <!-- News Section -->
    <section class="mb-12">
      <h3 class="text-xl font-semibold mb-4">Latest News</h3>
      <div>



        <!-- TradingView Widget BEGIN -->
        <div class="tradingview-widget-container" style="width: 100%; height: 550px;">
          <script type="text/javascript" src="https://s3.tradingview.com/external-embedding/embed-widget-timeline.js"
            async>
              {
                "feedMode": "all_symbols",
                  "isTransparent": true,
                    "displayMode": "regular",
                      "width": "100%",
                        "height": 550,
                          "colorTheme": "light",
                            "locale": "en"
              }
            </script>
        </div>
        <!-- TradingView Widget END -->

      </div>
    </section>
  </div>



  <?php include './includes/footer.php'; ?>

</body>

</html>