<?php
// Enable full error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Log PHP errors to a file
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error_log.txt');

// Include required files
require_once 'db.php';         // Database connection ($conn)
require_once '../session.php'; // Starts session, expects $_SESSION['user_id'] to be set

try {
    // 1) Only allow POST requests
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        header('Location: dashboard.php');
        exit;
    }

    // 2) Ensure user is authenticated
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('You must be logged in to place an order.');
    }

    $userId = (int) $_SESSION['user_id'];

    // 3) Sanitize and fetch POST data
    $pairName       = trim($_POST['pairName'] ?? '');
    $price          = filter_var($_POST['price'] ?? 0, FILTER_VALIDATE_FLOAT);
    $orderType      = trim($_POST['orderType'] ?? '');
    $lotSize        = filter_var($_POST['lotSize'] ?? 0, FILTER_VALIDATE_FLOAT);
    $stopLoss       = isset($_POST['stopLoss']) && $_POST['stopLoss'] !== '' 
                        ? filter_var($_POST['stopLoss'], FILTER_VALIDATE_FLOAT) 
                        : null;
    $takeProfit     = isset($_POST['takeProfit']) && $_POST['takeProfit'] !== '' 
                        ? filter_var($_POST['takeProfit'], FILTER_VALIDATE_FLOAT) 
                        : null;
    $marginRequired = filter_var($_POST['marginRequired'] ?? 0, FILTER_VALIDATE_FLOAT);
    $pipValue       = filter_var($_POST['pipValue'] ?? 0, FILTER_VALIDATE_FLOAT);

    $spreadRaw      = trim($_POST['spread'] ?? '0');
    $spread         = filter_var(str_ireplace('pips', '', $spreadRaw), FILTER_VALIDATE_FLOAT);
    
    $estimatedCost  = filter_var($_POST['estimatedCost'] ?? 0, FILTER_VALIDATE_FLOAT);
    $createdAt      = date('Y-m-d H:i:s');

    // 4) Validate inputs strictly
    if ($pairName === '') {
        throw new Exception('Missing currency pair.');
    }
    if ($price === false || $price <= 0) {
        throw new Exception('Invalid price.');
    }
    if (!in_array($orderType, ['market', 'pending', 'stop', 'limit'], true)) {
        throw new Exception('Invalid order type.');
    }
    if ($lotSize === false || $lotSize <= 0) {
        throw new Exception('Lot size must be greater than zero.');
    }
    if ($marginRequired === false || $marginRequired <= 0) {
        throw new Exception('Margin required must be greater than zero.');
    }
    if ($pipValue === false || $pipValue <= 0) {
        throw new Exception('Pip value must be greater than zero.');
    }
    if ($spread === false || $spread < 0) {
        throw new Exception('Invalid spread value.');
    }
    if ($estimatedCost === false || $estimatedCost < 0) {
        throw new Exception('Invalid estimated cost.');
    }

    // 5) Fetch user balance and profile
    $stmt = $conn->prepare("SELECT usdt, email, name FROM users WHERE id = ?");
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        throw new Exception('User account not found.');
    }

    $user = $result->fetch_assoc();
    $usdtBalance = (float) $user['usdt'];
    $email       = $user['email'];
    $name        = $user['name'];

    // 6) Ensure user has sufficient funds
    if ($usdtBalance < $marginRequired) {
        throw new Exception('Insufficient USDT balance to place this order.');
    }

    // 7) Start transaction
    $conn->begin_transaction();

    // 8) Insert order into forex_orders table
    $insertOrder = $conn->prepare("
        INSERT INTO forex_orders
            (user_id, email, name, pair, order_type, lot_size, price, stop_loss,
             take_profit, margin_required, pip_value, spread, estimated_cost, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");

    /*
    Bind params types:
    i = integer
    s = string
    d = double (float)
    For stopLoss and takeProfit, which can be NULL, bind_param requires a workaround:
    Use bindValue with mysqli_stmt_bind_param and passing NULL as a param, or pass NULL as a string 'NULL' but better to use bind_param with a variable that can be NULL.

    PHP's mysqli does not support direct NULL in bind_param.
    So we use this trick:

    If stopLoss or takeProfit are NULL, bind_param expects a variable that is NULL.

    Solution: Use separate variables and use 'd' with nullable param or 's' with nullable string and use null or pass NULL as NULL.
    */

    // We must use variables
    $stopLossParam = $stopLoss !== null ? $stopLoss : null;
    $takeProfitParam = $takeProfit !== null ? $takeProfit : null;

    // Use "d" type for floats; if NULL, bind_param doesn't accept NULL directly,
    // so we need to bind as a string and pass NULL for database.

    // We'll use 's' type for nullable float fields, then pass NULL or string.
    // Then CAST inside SQL or make sure DB allows string 'NULL' -> better is to use NULL as SQL null.

    // But since it's float column, better to bind as float or NULL with the helper below.

    // Use a helper function to bind NULLs (needs PHP 8+ or extension), but here, workaround:

    // We'll use this method:
    // If value is null, bind as NULL with "bind_param" and "bindValue" workaround is not available in mysqli,
    // so we do prepared statement with ? and pass NULL via a variable with "d" type cast.

    // The simplest fix:
    // Instead of using bind_param, run SQL with ? and use bind_param for all values, for null values use NULL.

    // So change SQL to allow NULLs (assuming DB columns allow NULL)

    // The best way:

    $insertOrder->bind_param(
        'isssiddiddidds',
        $userId,
        $email,
        $name,
        $pairName,
        $orderType,
        $lotSize,
        $price,
        $stopLossParam,
        $takeProfitParam,
        $marginRequired,
        $pipValue,
        $spread,
        $estimatedCost,
        $createdAt
    );

    // This works only if $stopLossParam and $takeProfitParam are float or null,
    // but mysqli requires variables for NULL handling, so we'll adjust with a trick:

    // Trick: We use a small wrapper to convert NULL to 0 and then update the SQL to accept NULL
    // or call bind_param with 'd' and nullable variable but it may fail.

    // Because bind_param does not accept NULLs for float params, you can:

    // Option 1: Use 's' instead of 'd' for these two params and pass null or string.
    // Option 2: Use prepared statements with execute and bindValue with PDO instead of mysqli.

    // Here, let's use 's' for nullable fields and pass NULL or string.

    // So we change SQL and bind_param like this:

    $insertOrder = $conn->prepare("
        INSERT INTO forex_orders
            (user_id, email, name, pair, order_type, lot_size, price, stop_loss,
             take_profit, margin_required, pip_value, spread, estimated_cost, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");

    // Change bind_param types: 'isssissiddidds'
    // stop_loss and take_profit as strings to pass NULL properly

    $stopLossParamStr = $stopLossParam !== null ? (string)$stopLossParam : null;
    $takeProfitParamStr = $takeProfitParam !== null ? (string)$takeProfitParam : null;

    $insertOrder->bind_param(
        'isssissiddidds',
        $userId,
        $email,
        $name,
        $pairName,
        $orderType,
        $lotSize,
        $price,
        $stopLossParamStr,
        $takeProfitParamStr,
        $marginRequired,
        $pipValue,
        $spread,
        $estimatedCost,
        $createdAt
    );

    $insertOrder->execute();

    // 9) Deduct margin from user's USDT balance
    $newBalance = $usdtBalance - $marginRequired;
    $updateBalance = $conn->prepare("UPDATE users SET usdt = ? WHERE id = ?");
    $updateBalance->bind_param('di', $newBalance, $userId);
    $updateBalance->execute();

    // 10) Commit transaction
    $conn->commit();

    // 11) Redirect to dashboard with success message
    header('Location: dashboard.php?status=success&message=' . urlencode("Order placed successfully. {$marginRequired} USDT reserved."));
    exit;

} catch (Exception $error) {
    // Roll back transaction if active
    if ($conn->in_transaction) {
        $conn->rollback();
    }

    // Log and redirect with error
    error_log("Forex Order Error: " . $error->getMessage());
    header('Location: dashboard.php?status=error&message=' . urlencode($error->getMessage()));
    exit;
}
?>
