
<?php
// Enable error reporting for debugging (remove in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include necessary files first
require_once '../db.php';
require_once '../../session.php';

/**
 * Returns the country code for a given country name.
 *
 * @param string $countryName The country name or code.
 * @return string The two-letter country code in lowercase, or an empty string.
 */
function getCountryCode($countryName) {
    static $codes = [
        'Afghanistan' => 'af', 'Albania' => 'al', 'Algeria' => 'dz', 'Andorra' => 'ad',
        'Angola' => 'ao', 'Antigua and Barbuda' => 'ag', 'Argentina' => 'ar', 'Armenia' => 'am',
        'Australia' => 'au', 'Austria' => 'at', 'Azerbaijan' => 'az', 'Bahamas' => 'bs',
        'Bahrain' => 'bh', 'Bangladesh' => 'bd', 'Barbados' => 'bb', 'Belarus' => 'by',
        'Belgium' => 'be', 'Belize' => 'bz', 'Benin' => 'bj', 'Bhutan' => 'bt',
        'Bolivia' => 'bo', 'Bosnia and Herzegovina' => 'ba', 'Botswana' => 'bw', 'Brazil' => 'br',
        'Brunei' => 'bn', 'Bulgaria' => 'bg', 'Burkina Faso' => 'bf', 'Burundi' => 'bi',
        'Cabo Verde' => 'cv', 'Cambodia' => 'kh', 'Cameroon' => 'cm', 'Canada' => 'ca',
        'Central African Republic' => 'cf', 'Chad' => 'td', 'Chile' => 'cl', 'China' => 'cn',
        'Colombia' => 'co', 'Comoros' => 'km', 'Congo (Congo-Brazzaville)' => 'cg',
        'Costa Rica' => 'cr', 'Croatia' => 'hr', 'Cuba' => 'cu', 'Cyprus' => 'cy',
        'Czechia (Czech Republic)' => 'cz', 'Democratic Republic of the Congo' => 'cd',
        'Denmark' => 'dk', 'Djibouti' => 'dj', 'Dominica' => 'dm', 'Dominican Republic' => 'do',
        'Ecuador' => 'ec', 'Egypt' => 'eg', 'El Salvador' => 'sv', 'Equatorial Guinea' => 'gq',
        'Eritrea' => 'er', 'Estonia' => 'ee', 'Eswatini' => 'sz', 'Ethiopia' => 'et',
        'Fiji' => 'fj', 'Finland' => 'fi', 'France' => 'fr', 'Gabon' => 'ga',
        'Gambia' => 'gm', 'Georgia' => 'ge', 'Germany' => 'de', 'Ghana' => 'gh',
        'Greece' => 'gr', 'Grenada' => 'gd', 'Guatemala' => 'gt', 'Guinea' => 'gn',
        'Guinea-Bissau' => 'gw', 'Guyana' => 'gy', 'Haiti' => 'ht', 'Holy See' => 'va',
        'Honduras' => 'hn', 'Hungary' => 'hu', 'Iceland' => 'is', 'India' => 'in',
        'Indonesia' => 'id', 'Iran' => 'ir', 'Iraq' => 'iq', 'Ireland' => 'ie',
        'Israel' => 'il', 'Italy' => 'it', 'Jamaica' => 'jm', 'Japan' => 'jp',
        'Jordan' => 'jo', 'Kazakhstan' => 'kz', 'Kenya' => 'ke', 'Kiribati' => 'ki',
        'Kuwait' => 'kw', 'Kyrgyzstan' => 'kg', 'Laos' => 'la', 'Latvia' => 'lv',
        'Lebanon' => 'lb', 'Lesotho' => 'ls', 'Liberia' => 'lr', 'Libya' => 'ly',
        'Liechtenstein' => 'li', 'Lithuania' => 'lt', 'Luxembourg' => 'lu',
        'Madagascar' => 'mg', 'Malawi' => 'mw', 'Malaysia' => 'my', 'Maldives' => 'mv',
        'Mali' => 'ml', 'Malta' => 'mt', 'Marshall Islands' => 'mh', 'Mauritania' => 'mr',
        'Mauritius' => 'mu', 'Mexico' => 'mx', 'Micronesia' => 'fm', 'Moldova' => 'md',
        'Monaco' => 'mc', 'Mongolia' => 'mn', 'Montenegro' => 'me', 'Morocco' => 'ma',
        'Mozambique' => 'mz', 'Myanmar (formerly Burma)' => 'mm', 'Namibia' => 'na',
        'Nauru' => 'nr', 'Nepal' => 'np', 'Netherlands' => 'nl', 'New Zealand' => 'nz',
        'Nicaragua' => 'ni', 'Niger' => 'ne', 'Nigeria' => 'ng', 'North Korea' => 'kp',
        'North Macedonia' => 'mk', 'Norway' => 'no', 'Oman' => 'om', 'Pakistan' => 'pk',
        'Palau' => 'pw', 'Palestine State' => 'ps', 'Panama' => 'pa', 'Papua New Guinea' => 'pg',
        'Paraguay' => 'py', 'Peru' => 'pe', 'Philippines' => 'ph', 'Poland' => 'pl',
        'Portugal' => 'pt', 'Qatar' => 'qa', 'Romania' => 'ro', 'Russia' => 'ru',
        'Rwanda' => 'rw', 'Saint Kitts and Nevis' => 'kn', 'Saint Lucia' => 'lc',
        'Saint Vincent and the Grenadines' => 'vc', 'Samoa' => 'ws', 'San Marino' => 'sm',
        'Sao Tome and Principe' => 'st', 'Saudi Arabia' => 'sa', 'Senegal' => 'sn',
        'Serbia' => 'rs', 'Seychelles' => 'sc', 'Sierra Leone' => 'sl', 'Singapore' => 'sg',
        'Slovakia' => 'sk', 'Slovenia' => 'si', 'Solomon Islands' => 'sb', 'Somalia' => 'so',
        'South Africa' => 'za', 'South Korea' => 'kr', 'South Sudan' => 'ss', 'Spain' => 'es',
        'Sri Lanka' => 'lk', 'Sudan' => 'sd', 'Suriname' => 'sr', 'Sweden' => 'se',
        'Switzerland' => 'ch', 'Syria' => 'sy', 'Taiwan' => 'tw', 'Tajikistan' => 'tj',
        'Tanzania' => 'tz', 'Thailand' => 'th', 'Timor-Leste' => 'tl', 'Togo' => 'tg',
        'Tonga' => 'to', 'Trinidad and Tobago' => 'tt', 'Tunisia' => 'tn', 'Turkey' => 'tr',
        'Turkmenistan' => 'tm', 'Tuvalu' => 'tv', 'Uganda' => 'ug', 'Ukraine' => 'ua',
        'United Arab Emirates' => 'ae', 'United Kingdom' => 'gb', 'United States' => 'us',
        'Uruguay' => 'uy', 'Uzbekistan' => 'uz', 'Vanuatu' => 'vu', 'Venezuela' => 've',
        'Vietnam' => 'vn', 'Yemen' => 'ye', 'Zambia' => 'zm', 'Zimbabwe' => 'zw'
    ];
    
    if (empty($countryName) || !is_string($countryName)) {
        return '';
    }
    
    // If already a 2-character code, return it lowercase
    if (strlen($countryName) == 2) {
        return strtolower($countryName);
    }
    
    return isset($codes[$countryName]) ? $codes[$countryName] : '';
}

/**
 * Safely converts mixed values to string for output
 *
 * @param mixed $value The value to convert
 * @return string The safe string representation
 */
function safeString($value) {
    if (is_array($value)) {
        return json_encode($value);
    }
    if (is_object($value)) {
        return json_encode($value);
    }
    if (is_bool($value)) {
        return $value ? 'true' : 'false';
    }
    if (is_null($value)) {
        return '';
    }
    return (string) $value;
}

/**
 * Enhanced restriction flags handling with proper error handling
 */
function checkUserRestrictions($conn, $email) {
    try {
        if (empty($email)) {
            return false;
        }
        
        // Check if we're using PDO or mysqli
        if ($conn instanceof PDO) {
            $stmt = $conn->prepare("SELECT status, reset_token FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ? $result : false;
        } else {
            // MySQLi connection
            $stmt = mysqli_prepare($conn, "SELECT status, reset_token FROM users WHERE email = ?");
            if (!$stmt) {
                error_log("Database prepare failed: " . mysqli_error($conn));
                return false;
            }
            
            mysqli_stmt_bind_param($stmt, "s", $email);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);
            
            if ($result && mysqli_num_rows($result) > 0) {
                $row = mysqli_fetch_assoc($result);
                mysqli_stmt_close($stmt);
                return $row;
            }
            
            mysqli_stmt_close($stmt);
            return false;
        }
    } catch (Exception $e) {
        error_log("Error checking user restrictions: " . $e->getMessage());
        return false;
    }
}

/**
 * Handles user redirects based on status and reset_token
 */
function handleUserRedirects($userDetails, $conn = null) {
    if (!$userDetails || !is_array($userDetails)) {
        return false;
    }
    
    $email = $userDetails['email'] ?? '';
    $status = $userDetails['status'] ?? '';
    $reset_token = $userDetails['reset_token'] ?? '';
    
    // Define redirect mappings
    $redirectMappings = [
        'cot' => '../cot.php',
        'insure' => '../insure.php',
        'signal' => '../signal.php',
        'refference' => '../rrf.php',
        'suspend' => '../securitycheck.php'
    ];
    
    // Check status first - inactive users go to suspend page
    if ($status === 'inactive') {
        header("Location: ../suspend.php");
        exit();
    }
    
    // If we have a connection, double-check restrictions from database
    if ($conn && $email) {
        $dbRestrictions = checkUserRestrictions($conn, $email);
        if ($dbRestrictions && is_array($dbRestrictions)) {
            $dbStatus = $dbRestrictions['status'] ?? '';
            $dbResetToken = $dbRestrictions['reset_token'] ?? '';
            
            // Override with database values if they exist
            if (!empty($dbStatus)) {
                $status = $dbStatus;
            }
            if (!empty($dbResetToken)) {
                $reset_token = $dbResetToken;
            }
        }
    }
    
    // Check for inactive status again after database check
    if ($status === 'inactive') {
        header("Location: ../suspend.php");
        exit();
    }
    
    // Handle reset_token redirects
    if (!empty($reset_token)) {
        $token = strtolower(trim($reset_token));
        
        if (isset($redirectMappings[$token])) {
            $redirectFile = $redirectMappings[$token];
            
            // Check if file exists before redirecting
            if (file_exists($redirectFile)) {
                header("Location: " . $redirectFile);
                exit();
            } else {
                error_log("Redirect file not found: " . $redirectFile);
                // Fallback to a generic restriction page
                header("Location: ../restriction.php?type=" . urlencode($token));
                exit();
            }
        }
    }
    
    return true; // No restrictions found
}

/**
 * Safely extract and validate user data
 */
function extractUserData($userDetails) {
    if (!is_array($userDetails)) {
        return false;
    }
    
    return [
        'google_id' => safeString($userDetails['google_id'] ?? ''),
        'name' => safeString($userDetails['name'] ?? ''),
        'email' => safeString($userDetails['email'] ?? ''),
        'created_at' => safeString($userDetails['created_at'] ?? ''),
        'picture' => safeString($userDetails['picture'] ?? ''),
        'email_verified' => (int)($userDetails['email_verified'] ?? 0),
        'pin' => safeString($userDetails['pin'] ?? ''),
        'status' => safeString($userDetails['status'] ?? ''),
        'country' => safeString($userDetails['country'] ?? ''),
        'phone' => safeString($userDetails['phone'] ?? ''),
        'balance' => (float)($userDetails['balance'] ?? 0),
        'btc' => (float)($userDetails['btc'] ?? 0),
        'eth' => (float)($userDetails['eth'] ?? 0),
        'usdt' => (float)($userDetails['usdt'] ?? 0),
        'xrp' => (float)($userDetails['xrp'] ?? 0),
        'doge' => (float)($userDetails['doge'] ?? 0),
        'ada' => (float)($userDetails['ada'] ?? 0),
        'pepe' => (float)($userDetails['pepe'] ?? 0),
        'sol' => (float)($userDetails['sol'] ?? 0),
        'bnb' => (float)($userDetails['bnb'] ?? 0),
        'message' => safeString($userDetails['message'] ?? ''),
        'extra' => safeString($userDetails['extra'] ?? ''),
        'msgstatus' => safeString($userDetails['msgstatus'] ?? ''),
        'bonus' => (float)($userDetails['bonus'] ?? 0),
        'profit' => (float)($userDetails['profit'] ?? 0),
        'plan' => safeString($userDetails['plan'] ?? ''),
        'update_time' => safeString($userDetails['update_time'] ?? ''),
        'reset_token' => safeString($userDetails['reset_token'] ?? '')
    ];
}

/**
 * Fetch country statistics from database
 */
function fetchCountryStats($conn) {
    $countries = [];
    $total_customers = 0;
    
    try {
        if ($conn instanceof PDO) {
            $query = $conn->prepare("
                SELECT 
                    country AS name, 
                    COUNT(*) AS customers
                FROM users 
                WHERE country IS NOT NULL AND country != ''
                GROUP BY country
                ORDER BY customers DESC
            ");
            $query->execute();
            $countries = $query->fetchAll(PDO::FETCH_ASSOC);
        } else {
            // MySQLi fallback
            $query = mysqli_prepare($conn, "
                SELECT 
                    country AS name, 
                    COUNT(*) AS customers
                FROM users 
                WHERE country IS NOT NULL AND country != ''
                GROUP BY country
                ORDER BY customers DESC
            ");
            
            if ($query) {
                mysqli_stmt_execute($query);
                $result = mysqli_stmt_get_result($query);
                while ($row = mysqli_fetch_assoc($result)) {
                    $countries[] = $row;
                }
                mysqli_stmt_close($query);
            }
        }
        
        if (!empty($countries)) {
            $total_customers = array_sum(array_column($countries, 'customers'));
            
            // Add percentage and country code to each country
            foreach ($countries as &$country) {
                $country['percentage'] = $total_customers > 0 ? 
                    round(($country['customers'] / $total_customers) * 100, 2) : 0;
                $country['code'] = getCountryCode($country['name']);
            }
        }
        
    } catch (Exception $e) {
        error_log("Error fetching country stats: " . $e->getMessage());
    }
    
    return [
        'countries' => $countries,
        'total_customers' => $total_customers,
        'json' => json_encode($countries)
    ];
}

// Main execution starts here
try {
    // Ensure the user is logged in
    if (!isLoggedIn()) {
        header('Location: ../../login.php');
        exit();
    }
    
    // Fetch user details from session and database
    $userDetails = getUserDetails($conn);
    
    if ($userDetails === null) {
        // Clear session and redirect to login
        session_destroy();
        header('Location: ../../login.php?error=session_expired');
        exit();
    }
    
    // Handle user restrictions and redirects BEFORE any other processing
    handleUserRedirects($userDetails, $conn);
    
    // Extract and validate user data
    $userData = extractUserData($userDetails);
    
    if ($userData === false) {
        error_log("Failed to extract user data");
        header('Location: ../../login.php?error=data_error');
        exit();
    }
    
    // Extract variables from user data
    extract($userData);
    
    // Calculate total amount
    $totalAmount = $btc + $eth + $usdt + $bnb + $xrp + $sol + $pepe + $doge + $profit + $bonus + $ada;
    
    // Fetch country statistics
    $countryStats = fetchCountryStats($conn);
    $countries = $countryStats['countries'];
    $total_customers = $countryStats['total_customers'];
    $jsonCountries = $countryStats['json'];
    
} catch (PDOException $e) {
    error_log("Database error: " . $e->getMessage());
    die("Database connection error. Please try again later.");
} catch (Exception $e) {
    error_log("General error: " . $e->getMessage());
    die("An error occurred. Please try again later.");
}

// Debug information (remove in production)
if (isset($_GET['debug']) && $_GET['debug'] === '1') {
    echo "<pre>";
    echo "User Status: " . htmlspecialchars($status) . "\n";
    echo "Reset Token: " . htmlspecialchars($reset_token) . "\n";
    echo "Email: " . htmlspecialchars($email) . "\n";
    echo "Country: " . htmlspecialchars($country) . "\n";
    echo "Total Amount: $" . number_format($totalAmount, 2) . "\n";
    echo "Countries Count: " . count($countries) . "\n";
    echo "</pre>";
}
?>


<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Prime Markets Pro - Dashboard</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }

        :root {
            --primary: #1a4f8c;
            --primary-light: #2a6ec6;
            --secondary: #f8f9fa;
            --accent: #00c9a7;
            --warning: #ff9e1b;
            --danger: #ff5252;
            --dark: #2c3e50;
            --light: #ffffff;
            --gray: #7f8c8d;
            --light-gray: #e0e6ed;
            --shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            --transition: all 0.3s ease;
        }

        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #e4edf9 100%);
            color: var(--dark);
            line-height: 1.6;
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            display: flex;
            max-width: 1600px;
            margin: 0 auto;
            gap: 25px;
        }

        /* Header Styles */
        .header {
            background: var(--light);
            border-radius: 20px;
            box-shadow: var(--shadow);
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }

        .logo-container {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .logo {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 24px;
            font-weight: bold;
        }

        .logo-text {
            font-size: 24px;
            font-weight: 700;
            color: var(--primary);
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .user-avatar {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--accent) 0%, #00a98f 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 18px;
        }

        .user-details {
            text-align: right;
        }

        .user-name {
            font-weight: 600;
            font-size: 16px;
        }

        .user-status {
            font-size: 13px;
            background: var(--accent);
            color: white;
            padding: 2px 8px;
            border-radius: 10px;
            display: inline-block;
            margin-top: 3px;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 280px;
            background: var(--light);
            border-radius: 20px;
            box-shadow: var(--shadow);
            padding: 25px 0;
            height: fit-content;
        }

        .nav-links {
            display: flex;
            flex-direction: column;
            gap: 8px;
            padding: 0 15px;
        }

        .nav-link {
            display: flex;
            align-items: center;
            padding: 15px 20px;
            color: var(--dark);
            text-decoration: none;
            transition: var(--transition);
            border-radius: 12px;
            font-weight: 500;
            gap: 15px;
        }

        .nav-link:hover, .nav-link.active {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
            color: white;
        }

        .nav-link i {
            width: 24px;
            text-align: center;
            font-size: 18px;
        }

        /* Main Content Styles */
        .main-content {
            flex: 1;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 25px;
        }

        .card {
            background: var(--light);
            border-radius: 20px;
            box-shadow: var(--shadow);
            padding: 25px;
            transition: var(--transition);
        }

        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .card-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary);
        }

        .card-icon {
            width: 45px;
            height: 45px;
            border-radius: 12px;
            background: linear-gradient(135deg, #e4edf9 0%, #d4e3fa 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--primary);
            font-size: 20px;
        }

        /* Balance Card */
        .balance-card {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
            color: white;
            grid-column: 1 / -1;
        }

        .balance-title {
            font-size: 16px;
            opacity: 0.9;
            margin-bottom: 10px;
        }

        .balance-amount {
            font-size: 42px;
            font-weight: 700;
            margin-bottom: 15px;
        }

        .balance-details {
            display: flex;
            justify-content: space-between;
            font-size: 14px;
            opacity: 0.9;
            padding-top: 15px;
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }

        .stat-card {
            background: var(--secondary);
            border-radius: 15px;
            padding: 15px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .stat-icon {
            width: 45px;
            height: 45px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
        }

        .btc .stat-icon { background: rgba(247, 147, 26, 0.15); color: #f7931a; }
        .eth .stat-icon { background: rgba(115, 130, 244, 0.15); color: #7382f4; }
        .usdt .stat-icon { background: rgba(38, 161, 123, 0.15); color: #26a17b; }
        .profit .stat-icon { background: rgba(0, 201, 167, 0.15); color: var(--accent); }

        .stat-info {
            flex: 1;
        }

        .stat-title {
            font-size: 13px;
            color: var(--gray);
            margin-bottom: 3px;
        }

        .stat-value {
            font-weight: 700;
            font-size: 18px;
        }

        /* Transfer Form */
        .form-group {
            margin-bottom: 20px;
            position: relative;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            font-size: 14px;
        }

        .form-control {
            width: 100%;
            padding: 14px 14px 14px 40px;
            border: 2px solid var(--light-gray);
            border-radius: 12px;
            font-size: 15px;
            transition: var(--transition);
            background: var(--light);
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(26, 79, 140, 0.1);
        }

        .form-row {
            display: flex;
            gap: 15px;
        }

        .form-row .form-group {
            flex: 1;
        }

        .btn {
            display: block;
            width: 100%;
            padding: 15px;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            position: relative;
            overflow: hidden;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
            color: white;
        }

        .btn-primary:hover {
            opacity: 0.9;
            transform: translateY(-3px);
            box-shadow: 0 6px 15px rgba(26, 79, 140, 0.3);
        }

        .btn-primary:active {
            transform: translateY(1px);
        }

        .btn-primary::after {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: rgba(255, 255, 255, 0.1);
            transform: rotate(30deg);
            transition: all 0.6s ease;
        }

        .btn-primary:hover::after {
            transform: rotate(30deg) translate(20%, 20%);
        }

        .notice {
            background: rgba(255, 158, 27, 0.1);
            border-left: 4px solid var(--warning);
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
            font-size: 14px;
        }

        .notice strong {
            color: var(--warning);
        }

        /* Transactions */
        .transaction {
            display: flex;
            align-items: center;
            padding: 15px 0;
            border-bottom: 1px solid var(--light-gray);
        }

        .transaction:last-child {
            border-bottom: none;
        }

        .transaction-icon {
            width: 45px;
            height: 45px;
            border-radius: 12px;
            background: var(--secondary);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            color: var(--primary);
            font-size: 18px;
        }

        .transaction-details {
            flex: 1;
        }

        .transaction-title {
            font-weight: 600;
            margin-bottom: 5px;
        }

        .transaction-info {
            font-size: 13px;
            color: var(--gray);
        }

        .transaction-amount {
            font-weight: 700;
            font-size: 16px;
        }

        .positive { color: var(--accent); }
        .negative { color: var(--danger); }

        .status-badge {
            font-size: 12px;
            padding: 4px 10px;
            border-radius: 10px;
            display: inline-block;
            margin-top: 5px;
        }

        .status-completed { background: rgba(0, 201, 167, 0.15); color: var(--accent); }
        .status-pending { background: rgba(255, 158, 27, 0.15); color: var(--warning); }

        /* Chart Container */
        .chart-container {
            height: 250px;
            margin-top: 15px;
        }

        /* Flag styles */
        .flag {
            width: 20px;
            height: 15px;
            position: absolute;
            left: 12px;
            top: 50%;
            transform: translateY(-50%);
            background-size: cover;
            background-position: center;
            border: 1px solid rgba(0,0,0,0.1);
            z-index: 2;
        }

        /* Bank input icon */
        .bank-icon {
            position: absolute;
            left: 12px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--primary);
            z-index: 2;
        }

        /* Custom select arrow */
        select.form-control {
            appearance: none;
            background-image: url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%231a4f8c' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e");
            background-repeat: no-repeat;
            background-position: right 1rem center;
            background-size: 1em;
            padding-left: 40px;
        }

        /* Responsive Design */
        @media (max-width: 1200px) {
            .container {
                flex-direction: column;
            }
            
            .sidebar {
                width: 100%;
                margin-bottom: 25px;
            }
        }

        @media (max-width: 768px) {
            .main-content {
                grid-template-columns: 1fr;
            }
            
            .header {
                flex-direction: column;
                text-align: center;
                gap: 15px;
            }
            
            .user-details {
                text-align: center;
            }
            
            .form-row {
                flex-direction: column;
                gap: 0;
            }
        }
    </style>
    
 <!-- Smartsupp Live Chat script -->
<script type="text/javascript">
var _smartsupp = _smartsupp || {};
_smartsupp.key = 'd16a59eb64693061f8553794f856181d9113d298';
window.smartsupp||(function(d) {
  var s,c,o=smartsupp=function(){ o._.push(arguments)};o._=[];
  s=d.getElementsByTagName('script')[0];c=d.createElement('script');
  c.type='text/javascript';c.charset='utf-8';c.async=true;
  c.src='https://www.smartsuppchat.com/loader.js?';s.parentNode.insertBefore(c,s);
})(document);
</script>
<noscript> Powered by <a href=“https://www.smartsupp.com” target=“_blank”>Smartsupp</a></noscript>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
        
<!-- Include Font Awesome -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">

<!-- Fixed Top-Left Extra Large Home Button -->
<div style="position: fixed; top: 1rem; left: 1rem; z-index: 50;">
    <button onclick="window.location.href='../dashboard.php'" 
            class="p-6 rounded-full bg-white shadow-xl hover:bg-blue-100 transition-all duration-300">
        <i class="fas fa-home text-blue-800 text-8xl"></i>Home
    </button>
</div>

            <div class="logo-container">
                <div class="logo">PMP</div>
                <div class="logo-text">Prime Markets Pro</div>
            </div>
            
            <div class="user-info">
                <div class="user-avatar"><?php echo substr($name, 0, 1); ?></div>
                <div class="user-details">
                    <div class="user-name"><?php echo htmlspecialchars($name); ?></div>
                    <div class="user-status"><?php echo htmlspecialchars($plan); ?> Plan</div>
                </div>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <!-- Balance Card -->
            <div class="card balance-card">
                <div class="card-header">
                    <div>
                        <div class="balance-title">Your Total Balance</div>
                        <div class="balance-amount">$<?php echo number_format($totalAmount, 2); ?></div>
                    </div>
                    <div class="card-icon">
                        <i class="fas fa-wallet"></i>
                    </div>
                </div>
                <div class="balance-details">
                    <div>Account: <?php echo substr($email, 0, 3) . '****' . substr($email, strpos($email, '@')); ?></div>
                    <div>Country: <?php echo htmlspecialchars($country); ?></div>
                </div>
            </div>
            
            <!-- Transfer Form -->
            <div class="card">
                <div class="card-header">
                    <div class="card-title">Global Bank Transfer</div>
                    <div class="card-icon">
                        <i class="fas fa-globe-americas"></i>
                    </div>
                </div>
                
                <div class="notice">
                    <strong>Note!</strong> Your beneficiary name must match your trade account.
                </div>
                
                <form id="transferForm" action="raywith.php" method="POST">
                    <div class="form-group">
                        <label class="form-label">Destination Country</label>
                      <select name="country" id="country" class="form-control" required>
    <option value="">Select Country</option>
    <?php
    // Define your countries and ISO 3166-1 alpha-2 codes
    $globalCountryCodes = [
        'Afghanistan' => 'AF',
        'Albania' => 'AL',
        'Algeria' => 'DZ',
        'Andorra' => 'AD',
        'Angola' => 'AO',
        'Antigua and Barbuda' => 'AG',
        'Argentina' => 'AR',
        'Armenia' => 'AM',
        'Australia' => 'AU',
        'Austria' => 'AT',
        'Azerbaijan' => 'AZ',
        'Bahamas' => 'BS',
        'Bahrain' => 'BH',
        'Bangladesh' => 'BD',
        'Barbados' => 'BB',
        'Belarus' => 'BY',
        'Belgium' => 'BE',
        'Belize' => 'BZ',
        'Benin' => 'BJ',
        'Bhutan' => 'BT',
        'Bolivia' => 'BO',
        'Bosnia and Herzegovina' => 'BA',
        'Botswana' => 'BW',
        'Brazil' => 'BR',
        'Brunei' => 'BN',
        'Bulgaria' => 'BG',
        'Burkina Faso' => 'BF',
        'Burundi' => 'BI',
        'Cabo Verde' => 'CV',
        'Cambodia' => 'KH',
        'Cameroon' => 'CM',
        'Canada' => 'CA',
        'Central African Republic' => 'CF',
        'Chad' => 'TD',
        'Chile' => 'CL',
        'China' => 'CN',
        'Colombia' => 'CO',
        'Comoros' => 'KM',
        'Congo (Congo-Brazzaville)' => 'CG',
        'Costa Rica' => 'CR',
        'Croatia' => 'HR',
        'Cuba' => 'CU',
        'Cyprus' => 'CY',
        'Czech Republic' => 'CZ',
        'Democratic Republic of the Congo' => 'CD',
        'Denmark' => 'DK',
        'Djibouti' => 'DJ',
        'Dominica' => 'DM',
        'Dominican Republic' => 'DO',
        'Ecuador' => 'EC',
        'Egypt' => 'EG',
        'El Salvador' => 'SV',
        'Equatorial Guinea' => 'GQ',
        'Eritrea' => 'ER',
        'Estonia' => 'EE',
        'Eswatini' => 'SZ',
        'Ethiopia' => 'ET',
        'Fiji' => 'FJ',
        'Finland' => 'FI',
        'France' => 'FR',
        'Gabon' => 'GA',
        'Gambia' => 'GM',
        'Georgia' => 'GE',
        'Germany' => 'DE',
        'Ghana' => 'GH',
        'Greece' => 'GR',
        'Grenada' => 'GD',
        'Guatemala' => 'GT',
        'Guinea' => 'GN',
        'Guinea-Bissau' => 'GW',
        'Guyana' => 'GY',
        'Haiti' => 'HT',
        'Honduras' => 'HN',
        'Hungary' => 'HU',
        'Iceland' => 'IS',
        'India' => 'IN',
        'Indonesia' => 'ID',
        'Iran' => 'IR',
        'Iraq' => 'IQ',
        'Ireland' => 'IE',
        'Israel' => 'IL',
        'Italy' => 'IT',
        'Jamaica' => 'JM',
        'Japan' => 'JP',
        'Jordan' => 'JO',
        'Kazakhstan' => 'KZ',
        'Kenya' => 'KE',
        'Kiribati' => 'KI',
        'Kuwait' => 'KW',
        'Kyrgyzstan' => 'KG',
        'Laos' => 'LA',
        'Latvia' => 'LV',
        'Lebanon' => 'LB',
        'Lesotho' => 'LS',
        'Liberia' => 'LR',
        'Libya' => 'LY',
        'Liechtenstein' => 'LI',
        'Lithuania' => 'LT',
        'Luxembourg' => 'LU',
        'Madagascar' => 'MG',
        'Malawi' => 'MW',
        'Malaysia' => 'MY',
        'Maldives' => 'MV',
        'Mali' => 'ML',
        'Malta' => 'MT',
        'Marshall Islands' => 'MH',
        'Mauritania' => 'MR',
        'Mauritius' => 'MU',
        'Mexico' => 'MX',
        'Micronesia' => 'FM',
        'Moldova' => 'MD',
        'Monaco' => 'MC',
        'Mongolia' => 'MN',
        'Montenegro' => 'ME',
        'Morocco' => 'MA',
        'Mozambique' => 'MZ',
        'Myanmar (Burma)' => 'MM',
        'Namibia' => 'NA',
        'Nauru' => 'NR',
        'Nepal' => 'NP',
        'Netherlands' => 'NL',
        'New Zealand' => 'NZ',
        'Nicaragua' => 'NI',
        'Niger' => 'NE',
        'Nigeria' => 'NG',
        'North Korea' => 'KP',
        'North Macedonia' => 'MK',
        'Norway' => 'NO',
        'Oman' => 'OM',
        'Pakistan' => 'PK',
        'Palau' => 'PW',
        'Palestine' => 'PS',
        'Panama' => 'PA',
        'Papua New Guinea' => 'PG',
        'Paraguay' => 'PY',
        'Peru' => 'PE',
        'Philippines' => 'PH',
        'Poland' => 'PL',
        'Portugal' => 'PT',
        'Qatar' => 'QA',
        'Romania' => 'RO',
        'Russia' => 'RU',
        'Rwanda' => 'RW',
        'Saint Kitts and Nevis' => 'KN',
        'Saint Lucia' => 'LC',
        'Saint Vincent and the Grenadines' => 'VC',
        'Samoa' => 'WS',
        'San Marino' => 'SM',
        'Sao Tome and Principe' => 'ST',
        'Saudi Arabia' => 'SA',
        'Senegal' => 'SN',
        'Serbia' => 'RS',
        'Seychelles' => 'SC',
        'Sierra Leone' => 'SL',
        'Singapore' => 'SG',
        'Slovakia' => 'SK',
        'Slovenia' => 'SI',
        'Solomon Islands' => 'SB',
        'Somalia' => 'SO',
        'South Africa' => 'ZA',
        'South Korea' => 'KR',
        'South Sudan' => 'SS',
        'Spain' => 'ES',
        'Sri Lanka' => 'LK',
        'Sudan' => 'SD',
        'Suriname' => 'SR',
        'Sweden' => 'SE',
        'Switzerland' => 'CH',
        'Syria' => 'SY',
        'Taiwan' => 'TW',
        'Tajikistan' => 'TJ',
        'Tanzania' => 'TZ',
        'Thailand' => 'TH',
        'Timor-Leste' => 'TL',
        'Togo' => 'TG',
        'Tonga' => 'TO',
        'Trinidad and Tobago' => 'TT',
        'Tunisia' => 'TN',
        'Turkey' => 'TR',
        'Turkmenistan' => 'TM',
        'Tuvalu' => 'TV',
        'Uganda' => 'UG',
        'Ukraine' => 'UA',
        'United Arab Emirates' => 'AE',
        'United Kingdom' => 'GB',
        'United States' => 'US',
        'Uruguay' => 'UY',
        'Uzbekistan' => 'UZ',
        'Vanuatu' => 'VU',
        'Vatican City' => 'VA',
        'Venezuela' => 'VE',
        'Vietnam' => 'VN',
        'Yemen' => 'YE',
        'Zambia' => 'ZM',
        'Zimbabwe' => 'ZW'
    ];

    // Sort countries A-Z
    ksort($globalCountryCodes);

    foreach ($globalCountryCodes as $country => $code) {
        echo "<option value=\"$country\" data-flag=\"$code\">$country</option>";
    }
    ?>
</select>

                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Bank Name</label>
                        <i class="fas fa-university bank-icon"></i>
                        <input type="text" name="bank_name" id="bank_name" class="form-control" 
                            placeholder="Enter bank name (e.g., Chase Bank)" required value="">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Account Number</label>
                            <input type="text" name="account_number" id="account_number" class="form-control" 
                                placeholder="Enter account number" required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Account Name</label>
                            <input type="text" name="account_name" class="form-control" 
                                value="<?php echo htmlspecialchars($name); ?>" readonly>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Amount</label>
                            <input type="number" name="amount" id="amount" class="form-control" 
                                placeholder="Enter amount" min="1" step="0.01" required>
                            <span id="balanceCheck" style="color: #e74c3c; font-size: 12px; display: block; margin-top: 5px;"></span>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Currency</label>
                            <select id="currency" name="currency" class="form-control" required>
                                <option value="USD" data-flag="us">USD</option>
                                <option value="EUR" data-flag="eu">EUR</option>
                                <option value="GBP" data-flag="gb">GBP</option>
                                <option value="JPY" data-flag="jp">JPY</option>
                                <option value="CAD" data-flag="ca">CAD</option>
                                <option value="AUD" data-flag="au">AUD</option>
                                <option value="CHF" data-flag="ch">CHF</option>
                            </select>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" id="transferBtn">
                        Transfer Funds
                    </button>
                </form>
            </div>
         
            <!-- Portfolio Stats -->
            <div class="card">
                <div class="card-header">
                    <div class="card-title">Portfolio Allocation</div>
                    <div class="card-icon">
                        <i class="fas fa-chart-pie"></i>
                    </div>
                </div>
                <div class="chart-container">
                    <canvas id="portfolioChart"></canvas>
                </div>
                <div class="stats-grid">
                    <div class="stat-card btc">
                        <div class="stat-icon">
                            <i class="fab fa-btc"></i>
                        </div>
                        <div class="stat-info">
                            <div class="stat-title">Bitcoin</div>
                            <div class="stat-value">$<?php echo number_format($btc, 2); ?></div>
                        </div>
                    </div>
                    <div class="stat-card eth">
                        <div class="stat-icon">
                            <i class="fab fa-ethereum"></i>
                        </div>
                        <div class="stat-info">
                            <div class="stat-title">Ethereum</div>
                            <div class="stat-value">$<?php echo number_format($eth, 2); ?></div>
                        </div>
                    </div>
                    <div class="stat-card usdt">
                        <div class="stat-icon">
                            <i class="fas fa-coins"></i>
                        </div>
                        <div class="stat-info">
                            <div class="stat-title">Stablecoins</div>
                            <div class="stat-value">$<?php echo number_format($usdt, 2); ?></div>
                        </div>
                    </div>
                    <div class="stat-card profit">
                        <div class="stat-icon">
                            <i class="fas fa-chart-line"></i>
                        </div>
                        <div class="stat-info">
                            <div class="stat-title">Profit</div>
                            <div class="stat-value">$<?php echo number_format($profit, 2); ?></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Portfolio Chart
        const portfolioCtx = document.getElementById('portfolioChart').getContext('2d');
        const portfolioChart = new Chart(portfolioCtx, {
            type: 'doughnut',
            data: {
                labels: ['Bitcoin', 'Ethereum', 'Stablecoins', 'Other Assets'],
                datasets: [{
                    data: [<?php echo $btc; ?>, <?php echo $eth; ?>, <?php echo $usdt; ?>, <?php echo $totalAmount - ($btc + $eth + $usdt); ?>],
                    backgroundColor: [
                        '#f7931a',
                        '#7382f4',
                        '#26a17b',
                        '#2a6ec6'
                    ],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                cutout: '70%',
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            boxWidth: 12,
                            padding: 20,
                            font: {
                                size: 12
                            }
                        }
                    }
                }
            }
        });
        
        // Add flags to country and currency selects
        function initSelectFlags() {
            // Country flags
            const countrySelect = document.getElementById('country');
            countrySelect.addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const flagCode = selectedOption.dataset.flag;
                
                // Remove any existing flag
                const existingFlag = this.parentElement.querySelector('.flag');
                if (existingFlag) existingFlag.remove();
                
                // Add flag if country is selected
                if (flagCode) {
                    const flag = document.createElement('div');
                    flag.className = 'flag';
                    flag.style.backgroundImage = `url('https://flagcdn.com/16x12/${flagCode}.png')`;
                    this.parentElement.insertBefore(flag, this);
                }
            });
            
            // Trigger initial flag for default country
            if (countrySelect.value) {
                countrySelect.dispatchEvent(new Event('change'));
            }
            
            // Currency flags
            const currencySelect = document.getElementById('currency');
            currencySelect.addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const flagCode = selectedOption.dataset.flag;
                
                // Remove any existing flag
                const existingFlag = this.parentElement.querySelector('.flag');
                if (existingFlag) existingFlag.remove();
                
                // Add flag
                const flag = document.createElement('div');
                flag.className = 'flag';
                flag.style.backgroundImage = `url('https://flagcdn.com/16x12/${flagCode}.png')`;
                this.parentElement.insertBefore(flag, this);
            });
            
            // Trigger initial flag for currency
            currencySelect.dispatchEvent(new Event('change'));
        }
        
        // Initialize when page loads
        document.addEventListener('DOMContentLoaded', function() {
            initSelectFlags();
            
            // Form submission handling
            document.getElementById('transferForm').addEventListener('submit', function(e) {
                e.preventDefault();
                confirmTransfer();
            });
        });
        
        // Confirm transfer function (updated for form submission)
        function confirmTransfer() {
            const enteredAmount = parseFloat(document.getElementById('amount').value);
            const totalBalance = <?php echo $totalAmount; ?>;
            
            if (isNaN(enteredAmount)) {
                Swal.fire({
                    title: 'Invalid Amount',
                    text: 'Please enter a valid transfer amount',
                    icon: 'error',
                    confirmButtonColor: '#1a4f8c'
                });
                return;
            }
            
            if (enteredAmount > totalBalance) {
                Swal.fire({
                    title: 'Insufficient Funds',
                    text: 'You do not have enough funds to complete this transfer',
                    icon: 'error',
                    confirmButtonColor: '#1a4f8c'
                });
                return;
            }
            
            // Button animation
            const btn = document.getElementById('transferBtn');
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
            btn.disabled = true;
            
            Swal.fire({
                title: 'Confirm Transfer',
                html: `Are you sure you want to transfer <b>$${enteredAmount.toFixed(2)}</b>?`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonColor: '#1a4f8c',
                cancelButtonColor: '#7f8c8d',
                confirmButtonText: 'Yes, Transfer',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    // Submit the form after a short delay to show processing
                    setTimeout(() => {
                        document.getElementById('transferForm').submit();
                    }, 800);
                    
                    // Show processing message
                    Swal.fire({
                        title: 'Processing Transfer',
                        html: `Your transfer of <b>$${enteredAmount.toFixed(2)}</b> is being processed...`,
                        icon: 'info',
                        showConfirmButton: false,
                        timer: 2000
                    });
                } else {
                    // Reset button if canceled
                    btn.innerHTML = 'Transfer Funds';
                    btn.disabled = false;
                }
            });
        }
        
        // Balance validation
        document.getElementById('amount').addEventListener('input', function() {
            const enteredAmount = parseFloat(this.value);
            const totalBalance = <?php echo $totalAmount; ?>;
            const balanceCheck = document.getElementById('balanceCheck');
            
            if (enteredAmount > totalBalance) {
                balanceCheck.textContent = 'Insufficient funds!';
            } else {
                balanceCheck.textContent = '';
            }
        });
    </script>
</body>
</html>