<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
ini_set("log_errors", 1);
ini_set("error_log", __DIR__ . "/error_log.txt");

require_once '../../db.php';
require_once '../../session.php';

try {
    if ($_SERVER["REQUEST_METHOD"] !== "POST") {
        throw new Exception("Invalid request method");
    }
    if (!isset($_SESSION['user_id'])) {
        throw new Exception("User not logged in");
    }

    $userId = (int)$_SESSION['user_id'];
    // Transaction type for a subscription purchase.
    $transactionType = 'subscription';

    // Retrieve POST parameters from the subscription form.
    // 'amount' may include a "$" sign which we remove and convert to float.
    $amountRaw = $_POST['amount'] ?? '0';
    $amount = (float)str_replace('$', '', $amountRaw);
    $plan = $_POST['plan'] ?? '';

    // Validate the input.
    if ($amount <= 0) {
        throw new Exception("Amount must be positive");
    }
    if (empty($plan)) {
        throw new Exception("No subscription plan selected");
    }

    $conn->begin_transaction();

    try {
        // Lock the user's currency balances for update.
        $stmt = $conn->prepare("SELECT bnb, sol, ada, doge, btc, eth, usdt, xrp FROM users WHERE id = ? FOR UPDATE");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows === 0) {
            throw new Exception("User not found");
        }
        
        $userBalances = array_map('floatval', $result->fetch_assoc());
        $newBalances = $userBalances;
        
        // Deduct the subscription amount from the user's available balances.
        // The deduction follows a currency priority list.
        $remaining = $amount;
        foreach (['bnb', 'sol', 'ada', 'doge', 'btc', 'eth', 'usdt', 'xrp'] as $currency) {
            if ($remaining <= 0) break;
            $deduct = min($userBalances[$currency], $remaining);
            $newBalances[$currency] -= $deduct;
            $remaining -= $deduct;
        }
        if ($remaining > 0) {
            throw new Exception("Insufficient funds for subscription purchase");
        }
        
        // Update the user's balances and set the current subscription plan.
        $update = $conn->prepare("UPDATE users SET 
            bnb = ?, sol = ?, ada = ?, doge = ?,
            btc = ?, eth = ?, usdt = ?, xrp = ?,
            plan = ?
            WHERE id = ?");
        $update->bind_param("ddddddddsi",
            $newBalances['bnb'], $newBalances['sol'], $newBalances['ada'], $newBalances['doge'],
            $newBalances['btc'], $newBalances['eth'], $newBalances['usdt'], $newBalances['xrp'],
            $plan, $userId
        );
        $update->execute();

        // Retrieve the user's name and email from the POST data if available;
        // otherwise, get them from the database.
        $name = $_POST['name'] ?? '';
        $email = $_POST['email'] ?? '';
        if (!$name || !$email) {
            $userData = $conn->query("SELECT name, email FROM users WHERE id = $userId")->fetch_assoc();
            $name = $userData['name'];
            $email = $userData['email'];
        }
        
        // Record the subscription transaction in the piadplan table.
        $transid = uniqid('TX', true);
        $status = 'completed';
        $transactionDate = date("Y-m-d H:i:s");
        
        $insert = $conn->prepare("INSERT INTO paidplan (
            transid, user_id, name, email, 
            amount, plan, status, transactionDate
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $insert->bind_param("sissdsss",
            $transid, $userId, $name, $email,
            $amount, $plan, $status, $transactionDate
        );
        $insert->execute();
        
        $conn->commit();
        header("Location: ../dashboard.php?status=success&message=" . urlencode("Subscription purchased successfully"));
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }
} catch (Exception $e) {
    error_log("Subscription Transaction Error: " . $e->getMessage());
    header("Location: ../dashboard.php?status=error&message=" . urlencode($e->getMessage()));
    exit();
}
?>